const { Email } = require("./email.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const nodemailer = require("nodemailer");

// Configure email transport
const emailConfig = {
	host: process.env.SMTP_HOST || "smtp.gmail.com",
	port: parseInt(process.env.SMTP_PORT || "587"),
	secure: process.env.SMTP_SECURE === "true",
	auth: {
		user: process.env.SMTP_USER || "",
		pass: process.env.SMTP_PASS || "",
	},
};

let transporter = null;

if (emailConfig.auth.user && emailConfig.auth.pass) {
	transporter = nodemailer.createTransport(emailConfig);
	logger.info("Email transport configured", {
		host: emailConfig.host,
		port: emailConfig.port,
		user: emailConfig.auth.user,
	});
} else {
	logger.error(
		"Email credentials not configured - email sending will not work",
		{
			hasUser: !!emailConfig.auth.user,
			hasPass: !!emailConfig.auth.pass,
		},
	);
}

class EmailService {
	async sendEmail({
		from,
		to,
		cc,
		bcc,
		subject,
		text,
		html,
		contentType,
		sentBy,
	}) {
		const emailRecord = new Email({
			from: from || process.env.SMTP_FROM || emailConfig.auth.user,
			to: Array.isArray(to) ? to : [to],
			cc: cc || [],
			bcc: bcc || [],
			subject,
			text: contentType === "text" ? text : undefined,
			html: contentType === "html" ? html : undefined,
			contentType: contentType || (html ? "html" : "text"),
			status: "pending",
			sentBy,
			isBulk: false,
		});

		try {
			if (!transporter) {
				emailRecord.status = "failed";
				emailRecord.error = "Email service not configured";
				emailRecord.failedAt = new Date();
				await emailRecord.save();
				throw new ApiError(500, "Email service not configured");
			}

			if (!to || to.length === 0) {
				throw new ApiError(400, "Recipient email(s) required");
			}

			if (!subject) {
				throw new ApiError(400, "Email subject required");
			}

			if (!text && !html) {
				throw new ApiError(400, "Email content required");
			}

			const mailOptions = {
				from: emailRecord.from,
				to: Array.isArray(to) ? to.join(", ") : to,
				subject,
			};

			if (cc && cc.length > 0) {
				mailOptions.cc = Array.isArray(cc) ? cc.join(", ") : cc;
			}

			if (bcc && bcc.length > 0) {
				mailOptions.bcc = Array.isArray(bcc) ? bcc.join(", ") : bcc;
			}

			if (contentType === "html" || html) {
				mailOptions.html = html;
			} else {
				mailOptions.text = text;
			}

			const info =
				process.env.NODE_ENV === "development"
					? {
							messageId: `dev-${Date.now()}@example.com`,
							accepted: [recipient],
							rejected: [],
					  }
					: await transporter.sendMail(mailOptions);

			emailRecord.status = "sent";
			emailRecord.messageId = info.messageId;
			emailRecord.sentAt = new Date();
			emailRecord.accepted = info.accepted;
			emailRecord.rejected = info.rejected;
			await emailRecord.save();

			logger.info("Email sent successfully", {
				messageId: info.messageId,
				recipients: Array.isArray(to) ? to.length : 1,
				subject,
				emailId: emailRecord._id,
			});

			return {
				success: true,
				emailId: emailRecord._id,
				messageId: info.messageId,
				recipients: Array.isArray(to) ? to.length : 1,
				accepted: info.accepted,
				rejected: info.rejected,
			};
		} catch (error) {
			emailRecord.status = "failed";
			emailRecord.error = error.message;
			emailRecord.failedAt = new Date();
			await emailRecord.save();

			logger.error("Error sending email:", error);
			throw new ApiError(500, `Failed to send email: ${error.message}`);
		}
	}

	async sendBulkEmail({
		from,
		recipients,
		cc,
		bcc,
		subject,
		text,
		html,
		contentType,
		sentBy,
		batchSize = 50, // Send 50 emails per batch
		delayBetweenBatches = 2000, // 2 second delay between batches
		maxRetries = 3, // Retry failed emails up to 3 times
		bulkJobId = null, // Job ID from worker
		bulkEmailId = null, // Bulk email ID from controller
	}) {
		// Use provided bulkEmailId or generate one
		const bulkId = bulkEmailId || `bulk_${Date.now()}_${Math.random()
			.toString(36)
			.substr(2, 9)}`;
		const startTime = Date.now();

		// Create bulk tracking entry
		const bulkTrackingEntry = new Email({
			from: from || process.env.SMTP_FROM || emailConfig.auth.user,
			to: [],
			subject: `[BULK] ${subject}`,
			text,
			html,
			contentType: contentType || (html ? "html" : "text"),
			status: "pending",
			sentBy,
			isBulk: true,
			bulkId,
			bulkJobId,
			bulkRecipientCount: recipients?.length || 0,
			bulkProgress: "processing",
			bulkStartedAt: new Date(),
		});

		try {
			await bulkTrackingEntry.save();

			if (!transporter) {
				bulkTrackingEntry.bulkProgress = "failed";
				bulkTrackingEntry.status = "failed";
				bulkTrackingEntry.error = "Email service not configured";
				bulkTrackingEntry.bulkCompletedAt = new Date();
				await bulkTrackingEntry.save();
				throw new ApiError(500, "Email service not configured");
			}

			if (!recipients || recipients.length === 0) {
				bulkTrackingEntry.bulkProgress = "failed";
				bulkTrackingEntry.status = "failed";
				bulkTrackingEntry.error = "Recipients required";
				bulkTrackingEntry.bulkCompletedAt = new Date();
				await bulkTrackingEntry.save();
				throw new ApiError(400, "Recipients required");
			}

			if (!subject) {
				bulkTrackingEntry.bulkProgress = "failed";
				bulkTrackingEntry.status = "failed";
				bulkTrackingEntry.error = "Email subject required";
				bulkTrackingEntry.bulkCompletedAt = new Date();
				await bulkTrackingEntry.save();
				throw new ApiError(400, "Email subject required");
			}

			if (!text && !html) {
				bulkTrackingEntry.bulkProgress = "failed";
				bulkTrackingEntry.status = "failed";
				bulkTrackingEntry.error = "Email content required";
				bulkTrackingEntry.bulkCompletedAt = new Date();
				await bulkTrackingEntry.save();
				throw new ApiError(400, "Email content required");
			}

			// Remove duplicates and validate emails
			const uniqueRecipients = [
				...new Set(
					recipients.filter((email) => {
						const isValid = email && /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
						if (!isValid) {
							logger.warn(`Invalid email skipped: ${email}`, { bulkId });
						}
						return isValid;
					}),
				),
			];

			logger.info(`Starting bulk email send`, {
				bulkId,
				totalRecipients: uniqueRecipients.length,
				batches: Math.ceil(uniqueRecipients.length / batchSize),
				batchSize,
				delayBetweenBatches,
			});

			// Split recipients into batches
			const batches = [];
			for (let i = 0; i < uniqueRecipients.length; i += batchSize) {
				batches.push(uniqueRecipients.slice(i, i + batchSize));
			}

			const allResults = [];
			let totalSent = 0;
			let totalFailed = 0;

			// Process each batch
			for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
				const batch = batches[batchIndex];

				logger.info(`Processing batch ${batchIndex + 1}/${batches.length}`, {
					bulkId,
					batchSize: batch.length,
					progress: `${totalSent + totalFailed}/${uniqueRecipients.length}`,
				});

				// Send emails in current batch with concurrency limit
				const batchResults = await this._sendBatchWithRetry({
					batch,
					from,
					cc,
					bcc,
					subject,
					text,
					html,
					contentType,
					sentBy,
					bulkId,
					maxRetries,
					batchIndex,
				});

				// Collect results
				allResults.push(...batchResults);
				totalSent += batchResults.filter((r) => r.success).length;
				totalFailed += batchResults.filter((r) => !r.success).length;

				logger.info(`Batch ${batchIndex + 1} completed`, {
					bulkId,
					batchSent: batchResults.filter((r) => r.success).length,
					batchFailed: batchResults.filter((r) => !r.success).length,
					totalSent,
					totalFailed,
				});

				// Add delay between batches (except after last batch)
				if (batchIndex < batches.length - 1) {
					await this._sleep(delayBetweenBatches);
				}
			}

			const duration = ((Date.now() - startTime) / 1000).toFixed(2);

			logger.info(`Bulk email completed`, {
				bulkId,
				totalSent,
				totalFailed,
				duration: `${duration}s`,
				successRate: `${((totalSent / uniqueRecipients.length) * 100).toFixed(
					1,
				)}%`,
			});

			// Update tracking entry with final results
			bulkTrackingEntry.bulkProgress = "completed";
			bulkTrackingEntry.status = totalSent > 0 ? "sent" : "failed";
			bulkTrackingEntry.bulkSentCount = totalSent;
			bulkTrackingEntry.bulkFailedCount = totalFailed;
			bulkTrackingEntry.bulkCompletedAt = new Date();
			bulkTrackingEntry.bulkDuration = `${duration}s`;
			bulkTrackingEntry.bulkBatches = batches.length;
			if (totalSent > 0) {
				bulkTrackingEntry.sentAt = new Date();
			}
			await bulkTrackingEntry.save();

			return {
				success: true,
				bulkId,
				trackingId: bulkTrackingEntry._id,
				message: `Sent ${totalSent}/${uniqueRecipients.length} email(s)`,
				sentCount: totalSent,
				failedCount: totalFailed,
				totalRecipients: uniqueRecipients.length,
				duration: `${duration}s`,
				batches: batches.length,
				results: allResults,
			};
		} catch (error) {
			// Update tracking entry with error
			if (bulkTrackingEntry) {
				bulkTrackingEntry.bulkProgress = "failed";
				bulkTrackingEntry.status = "failed";
				bulkTrackingEntry.error = error.message;
				bulkTrackingEntry.bulkCompletedAt = new Date();
				const duration = ((Date.now() - startTime) / 1000).toFixed(2);
				bulkTrackingEntry.bulkDuration = `${duration}s`;
				try {
					await bulkTrackingEntry.save();
				} catch (saveError) {
					logger.error("Failed to update bulk tracking entry:", saveError);
				}
			}
			logger.error("Error sending bulk email:", error);
			throw new ApiError(500, "Failed to send bulk email");
		}
	}

	async _sendBatchWithRetry({
		batch,
		from,
		cc,
		bcc,
		subject,
		text,
		html,
		contentType,
		sentBy,
		bulkId,
		maxRetries,
		batchIndex,
	}) {
		const results = [];

		for (const recipient of batch) {
			let lastError = null;
			let sent = false;

			// Retry logic
			for (let attempt = 1; attempt <= maxRetries; attempt++) {
				try {
					const result = await this._sendSingleEmail({
						from,
						recipient,
						cc,
						bcc,
						subject,
						text,
						html,
						contentType,
						sentBy,
						bulkId,
						attempt,
					});

					results.push(result);
					sent = true;
					break; // Success, exit retry loop
				} catch (error) {
					lastError = error;

					if (attempt < maxRetries) {
						// Exponential backoff: 1s, 2s, 4s
						const delay = Math.pow(2, attempt - 1) * 1000;
						logger.warn(
							`Retry ${attempt}/${maxRetries} for ${recipient} after ${delay}ms`,
							{
								bulkId,
								error: error.message,
							},
						);
						await this._sleep(delay);
					}
				}
			}

			// If all retries failed
			if (!sent) {
				const emailRecord = new Email({
					from: from || process.env.SMTP_FROM || emailConfig.auth.user,
					to: [recipient],
					cc: cc || [],
					bcc: bcc || [],
					subject,
					text: contentType === "text" ? text : undefined,
					html: contentType === "html" ? html : undefined,
					contentType: contentType || (html ? "html" : "text"),
					status: "failed",
					sentBy,
					isBulk: true,
					bulkId,
					error: `Failed after ${maxRetries} retries: ${lastError?.message}`,
					failedAt: new Date(),
				});
				await emailRecord.save();

				results.push({
					success: false,
					recipient,
					error: lastError?.message || "Unknown error",
					emailId: emailRecord._id,
					retriesExhausted: true,
				});

				logger.error(
					`Failed to send to ${recipient} after ${maxRetries} retries`,
					{
						bulkId,
						error: lastError?.message,
					},
				);
			}
		}

		return results;
	}

	async _sendSingleEmail({
		from,
		recipient,
		cc,
		bcc,
		subject,
		text,
		html,
		contentType,
		sentBy,
		bulkId,
		attempt,
	}) {
		const emailRecord = new Email({
			from: from || process.env.SMTP_FROM || emailConfig.auth.user,
			to: [recipient],
			cc: cc || [],
			bcc: bcc || [],
			subject,
			text: contentType === "text" ? text : undefined,
			html: contentType === "html" ? html : undefined,
			contentType: contentType || (html ? "html" : "text"),
			status: "pending",
			sentBy,
			isBulk: true,
			bulkId,
		});

		try {
			const mailOptions = {
				from: emailRecord.from,
				to: recipient,
				subject,
			};

			if (cc && cc.length > 0) {
				mailOptions.cc = Array.isArray(cc) ? cc.join(", ") : cc;
			}

			if (bcc && bcc.length > 0) {
				mailOptions.bcc = Array.isArray(bcc) ? bcc.join(", ") : bcc;
			}

			if (contentType === "html" || html) {
				mailOptions.html = html;
			} else {
				mailOptions.text = text;
			}

			const info =
				process.env.NODE_ENV === "development"
					? {
							messageId: `dev-${Date.now()}@example.com`,
							accepted: [recipient],
							rejected: [],
					  }
					: await transporter.sendMail(mailOptions);

			emailRecord.status = "sent";
			emailRecord.messageId = info.messageId;
			emailRecord.sentAt = new Date();
			emailRecord.accepted = info.accepted;
			emailRecord.rejected = info.rejected;
			await emailRecord.save();

			return {
				success: true,
				recipient,
				messageId: info.messageId,
				emailId: emailRecord._id,
				attempt,
			};
		} catch (error) {
			// Don't save failed record yet - let retry logic handle it
			throw error;
		}
	}

	_sleep(ms) {
		return new Promise((resolve) => setTimeout(resolve, ms));
	}

	async verifyConnection() {
		try {
			if (!transporter) {
				return {
					connected: false,
					message: "Email service not configured",
				};
			}

			await transporter.verify();

			logger.info("Email connection verified");
			return {
				connected: true,
				message: "Email service ready",
				config: {
					host: emailConfig.host,
					port: emailConfig.port,
					user: emailConfig.auth.user,
				},
			};
		} catch (error) {
			logger.error("Email connection verification failed:", error);
			return {
				connected: false,
				message: `Connection failed: ${error.message}`,
			};
		}
	}

	async getEmailHistory({ status, limit = 50, page = 1 }) {
		try {
			const filter = {};
			if (status) filter.status = status;

			const skip = (page - 1) * limit;

			const emails = await Email.find(filter)
				.sort({ createdAt: -1 })
				.limit(limit)
				.skip(skip);

			const total = await Email.countDocuments(filter);

			return {
				emails,
				total,
				page,
				limit,
				totalPages: Math.ceil(total / limit),
			};
		} catch (error) {
			logger.error("Error getting email history:", error);
			throw new ApiError(500, "Failed to retrieve email history");
		}
	}

	async getEmailById(id) {
		try {
			const email = await Email.findById(id);

			if (!email) {
				throw new ApiError(404, "Email not found");
			}

			return email;
		} catch (error) {
			logger.error("Error getting email:", error);
			throw new ApiError(500, "Failed to retrieve email");
		}
	}

	async getBulkEmailStatus(bulkId) {
		try {
			// First try to find the tracking entry
			const trackingEntry = await Email.findOne({
				bulkId,
				bulkJobId: { $exists: true },
			}).sort({ createdAt: -1 });

			if (trackingEntry) {
				// Get individual email records for this bulk
				const emails = await Email.find({
					bulkId,
					bulkJobId: { $exists: false },
				}).sort({ createdAt: -1 });

				return {
					trackingId: trackingEntry._id,
					bulkId,
					bulkJobId: trackingEntry.bulkJobId,
					progress: trackingEntry.bulkProgress,
					status: trackingEntry.status,
					stats: {
						total: trackingEntry.bulkRecipientCount,
						sent: trackingEntry.bulkSentCount,
						failed: trackingEntry.bulkFailedCount,
						pending:
							trackingEntry.bulkRecipientCount -
							trackingEntry.bulkSentCount -
							trackingEntry.bulkFailedCount,
					},
					startedAt: trackingEntry.bulkStartedAt,
					completedAt: trackingEntry.bulkCompletedAt,
					duration: trackingEntry.bulkDuration,
					batches: trackingEntry.bulkBatches,
					error: trackingEntry.error,
					subject: trackingEntry.subject,
					sentBy: trackingEntry.sentBy,
					emails: emails.slice(0, 100), // Limit to first 100 individual emails
					totalEmailRecords: emails.length,
				};
			}

			// Fallback to old behavior for legacy bulk emails
			const emails = await Email.find({ bulkId }).sort({ createdAt: -1 });

			if (emails.length === 0) {
				throw new ApiError(404, "Bulk email batch not found");
			}

			const stats = {
				bulkId,
				total: emails.length,
				sent: emails.filter((e) => e.status === "sent").length,
				failed: emails.filter((e) => e.status === "failed").length,
				pending: emails.filter((e) => e.status === "pending").length,
			};

			return {
				stats,
				emails,
			};
		} catch (error) {
			logger.error("Error getting bulk email status:", error);
			throw new ApiError(500, "Failed to retrieve bulk email status");
		}
	}
}

module.exports = new EmailService();
