const pushSubscriptionService = require("./pushSubscription.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const { enqueueJob } = require("../../queues/job.queue");
const { v4: uuidv4 } = require("uuid");

class PushSubscriptionController {
	async getAllSubscriptions(req, res, next) {
		try {
			const { username, isActive } = req.query;

			const subscriptions = await pushSubscriptionService.getAllSubscriptions({
				username,
				isActive: isActive === "true" ? true : isActive === "false" ? false : undefined,
			});

			res
				.status(200)
				.json(
					new ApiResponse(200, subscriptions, "Push subscriptions retrieved successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	async getSubscriptionsByUsername(req, res, next) {
		try {
			const { username } = req.params;

			const subscriptions = await pushSubscriptionService.getSubscriptionsByUsername(username);

			res
				.status(200)
				.json(
					new ApiResponse(200, subscriptions, "User subscriptions retrieved successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	async getSubscriptionById(req, res, next) {
		try {
			const { id } = req.params;

			const subscription = await pushSubscriptionService.getSubscriptionById(id);

			res
				.status(200)
				.json(
					new ApiResponse(200, subscription, "Subscription retrieved successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	async createOrUpdateSubscription(req, res, next) {
		try {
			const subscription = await pushSubscriptionService.createOrUpdateSubscription(req.body);

			res
				.status(201)
				.json(new ApiResponse(201, subscription, "Subscription saved successfully"));
		} catch (error) {
			next(error);
		}
	}

	async updateSubscriptionStatus(req, res, next) {
		try {
			const { id } = req.params;
			const { isActive } = req.body;

			if (isActive === undefined) {
				throw new ApiError(400, "isActive field is required");
			}

			const subscription = await pushSubscriptionService.updateSubscriptionStatus(id, isActive);

			res
				.status(200)
				.json(
					new ApiResponse(200, subscription, "Subscription status updated successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	async deleteSubscription(req, res, next) {
		try {
			const { id } = req.params;

			const result = await pushSubscriptionService.deleteSubscription(id);

			res.status(200).json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	async deleteSubscriptionByEndpoint(req, res, next) {
		try {
			const { endpoint } = req.body;

			if (!endpoint) {
				throw new ApiError(400, "Endpoint is required");
			}

			const result = await pushSubscriptionService.deleteSubscriptionByEndpoint(endpoint);

			res.status(200).json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	async deleteUserSubscriptions(req, res, next) {
		try {
			const { username } = req.params;

			const result = await pushSubscriptionService.deleteUserSubscriptions(username);

			res.status(200).json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	async cleanupInactiveSubscriptions(req, res, next) {
		try {
			const { daysInactive } = req.query;

			const result = await pushSubscriptionService.cleanupInactiveSubscriptions(
				daysInactive ? parseInt(daysInactive) : 90
			);

			res.status(200).json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	async sendNotificationToUser(req, res, next) {
		try {
			const { username } = req.params;
			const payload = req.body;

			if (!payload.title || !payload.body) {
				throw new ApiError(400, "Notification title and body are required");
			}

			const sentBy = req.user?.username || req.user?.email;
			const result = await pushSubscriptionService.sendNotificationToUser(username, payload, sentBy);

			return res
				.status(200)
				.json(new ApiResponse(200, result, result.message));
		} catch (error) {
			next(error);
		}
	}

	async sendNotificationToAll(req, res, next) {
		try {
			const payload = req.body;
			const { batchSize, delayBetweenBatches, maxRetries } = req.query;

			if (!payload.title || !payload.body) {
				throw new ApiError(400, "Notification title and body are required");
			}

			const bulkNotificationId = uuidv4();
			const sentBy = req.user?.username || req.user?.email;

			// Create job for async processing
			const job = await enqueueJob(
				"send-bulk-push-notification",
				{
					payload,
					bulkNotificationId,
					batchSize: batchSize ? parseInt(batchSize) : undefined,
					delayBetweenBatches: delayBetweenBatches ? parseInt(delayBetweenBatches) : undefined,
					maxRetries: maxRetries ? parseInt(maxRetries) : undefined,
					sentBy,
				},
				{
					priority: 5,
					attempts: 3,
					backoff: { type: "exponential", delay: 5000 },
				}
			);

			return res
				.status(202)
				.json(new ApiResponse(202, {
					bulkNotificationId,
					status: "processing",
					message: "Bulk push notification job created and processing in background",
					statusUrl: `/api/push-subscriptions/bulk/${bulkNotificationId}`,
				}, "Bulk notification job created successfully. Use the bulkNotificationId to track progress."));
		} catch (error) {
			next(error);
		}
	}

	async getBulkNotificationStatus(req, res, next) {
		try {
			const { bulkId } = req.params;

			const result = await pushSubscriptionService.getBulkNotificationStatus(bulkId);

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Bulk notification status retrieved"));
		} catch (error) {
			next(error);
		}
	}

	async getNotificationHistory(req, res, next) {
		try {
			const { status, sentBy, limit, page } = req.query;

			const result = await pushSubscriptionService.getNotificationHistory({
				status,
				sentBy,
				limit: limit ? parseInt(limit) : undefined,
				page: page ? parseInt(page) : undefined,
			});

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Notification history retrieved successfully"));
		} catch (error) {
			next(error);
		}
	}
}

module.exports = new PushSubscriptionController();
