const { PushSubscription } = require("./pushSubscription.model");
const { PushNotification } = require("./pushNotification.model");
const { ApiError } = require("../../utils/ApiError");
const webpush = require("web-push");
const { logger } = require("../../core/logger/logger");

// Configure VAPID keys
const vapidPublicKey = process.env.VAPID_PUBLIC_KEY || "";
const vapidPrivateKey = process.env.VAPID_PRIVATE_KEY || "";
const vapidSubject = process.env.VAPID_SUBJECT || "mailto:support@node-meta.com";

if (vapidPublicKey && vapidPrivateKey) {
	webpush.setVapidDetails(vapidSubject, vapidPublicKey, vapidPrivateKey);
	logger.info("Web push VAPID configured", {
		subject: vapidSubject,
		publicKeyLength: vapidPublicKey.length,
		privateKeyLength: vapidPrivateKey.length,
	});
} else {
	logger.error("VAPID keys not configured - push notifications will not work", {
		hasPublicKey: !!vapidPublicKey,
		hasPrivateKey: !!vapidPrivateKey,
		publicKeyLength: vapidPublicKey.length,
		privateKeyLength: vapidPrivateKey.length,
	});
}

class PushSubscriptionService {
	async getAllSubscriptions({ username, isActive } = {}) {
		try {
			const filter = {};

			if (username) {
				filter.username = username;
			}

			if (isActive !== undefined) {
				filter.isActive = isActive;
			}

			const subscriptions = await PushSubscription.find(filter).sort({
				createdAt: -1,
			});

			logger.info(`Retrieved ${subscriptions.length} push subscriptions`);
			return subscriptions;
		} catch (error) {
			logger.error("Error getting push subscriptions:", error);
			throw new ApiError(500, "Failed to retrieve push subscriptions");
		}
	}

	async getSubscriptionsByUsername(username) {
		try {
			const subscriptions = await PushSubscription.find({
				username,
				isActive: true,
			}).sort({ lastUsed: -1 });

			logger.info(
				`Retrieved ${subscriptions.length} subscriptions for user: ${username}`,
			);
			return subscriptions;
		} catch (error) {
			logger.error("Error getting subscriptions by username:", error);
			throw new ApiError(500, "Failed to retrieve user subscriptions");
		}
	}

	async getSubscriptionById(id) {
		try {
			const subscription = await PushSubscription.findById(id);

			if (!subscription) {
				throw new ApiError(404, "Push subscription not found");
			}

			logger.info(`Retrieved subscription: ${id}`);
			return subscription;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error getting subscription by ID:", error);
			throw new ApiError(500, "Failed to retrieve subscription");
		}
	}

	// Upsert subscription by endpoint
	async createOrUpdateSubscription(subscriptionData) {
		try {
			const { endpoint, username, userId, subscription, deviceInfo } =
				subscriptionData;

			// Validate
			if (!endpoint || !username || !subscription) {
				throw new ApiError(
					400,
					"Endpoint, username, and subscription object are required",
				);
			}

			if (
				!subscription.endpoint ||
				!subscription.keys?.p256dh ||
				!subscription.keys?.auth
			) {
				throw new ApiError(400, "Invalid subscription object format");
			}

			const existing = await PushSubscription.findOne({ endpoint });

			if (existing) {
				existing.username = username;
				existing.userId = userId;
				existing.subscription = subscription;
				existing.deviceInfo = deviceInfo || existing.deviceInfo;
				existing.isActive = true;
				existing.lastUsed = new Date();

				await existing.save();

				logger.info(`Updated push subscription: ${existing._id}`);
				return existing;
			}

			const newSubscription = new PushSubscription({
				endpoint,
				username,
				userId,
				subscription,
				deviceInfo,
				isActive: true,
				lastUsed: new Date(),
			});

			await newSubscription.save();

			logger.info(`Created push subscription: ${newSubscription._id}`);
			return newSubscription;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error creating/updating subscription:", error);
			throw new ApiError(500, "Failed to create/update subscription");
		}
	}

	async updateSubscriptionStatus(id, isActive) {
		try {
			const subscription = await PushSubscription.findById(id);

			if (!subscription) {
				throw new ApiError(404, "Push subscription not found");
			}

			subscription.isActive = isActive;
			subscription.lastUsed = new Date();
			await subscription.save();

			logger.info(`Updated subscription status: ${id} -> ${isActive}`);
			return subscription;
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error updating subscription status:", error);
			throw new ApiError(500, "Failed to update subscription status");
		}
	}

	async deleteSubscription(id) {
		try {
			const subscription = await PushSubscription.findById(id);

			if (!subscription) {
				throw new ApiError(404, "Push subscription not found");
			}

			await subscription.deleteOne();

			logger.info(`Deleted push subscription: ${id}`);
			return { message: "Push subscription deleted successfully" };
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error deleting subscription:", error);
			throw new ApiError(500, "Failed to delete subscription");
		}
	}

	async deleteSubscriptionByEndpoint(endpoint) {
		try {
			const result = await PushSubscription.deleteOne({ endpoint });

			if (result.deletedCount === 0) {
				throw new ApiError(404, "Push subscription not found");
			}

			logger.info(`Deleted subscription by endpoint: ${endpoint}`);
			return { message: "Push subscription deleted successfully" };
		} catch (error) {
			if (error instanceof ApiError) {
				throw error;
			}
			logger.error("Error deleting subscription by endpoint:", error);
			throw new ApiError(500, "Failed to delete subscription");
		}
	}

	async deleteUserSubscriptions(username) {
		try {
			const result = await PushSubscription.deleteMany({ username });

			logger.info(
				`Deleted ${result.deletedCount} subscriptions for user: ${username}`,
			);
			return {
				message: `Deleted ${result.deletedCount} subscription(s)`,
				deletedCount: result.deletedCount,
			};
		} catch (error) {
			logger.error("Error deleting user subscriptions:", error);
			throw new ApiError(500, "Failed to delete user subscriptions");
		}
	}

	async cleanupInactiveSubscriptions(daysInactive = 90) {
		try {
			const cutoffDate = new Date();
			cutoffDate.setDate(cutoffDate.getDate() - daysInactive);

			const result = await PushSubscription.deleteMany({
				isActive: false,
				lastUsed: { $lt: cutoffDate },
			});

			logger.info(`Cleaned up ${result.deletedCount} inactive subscriptions`);
			return {
				message: `Cleaned up ${result.deletedCount} inactive subscription(s)`,
				deletedCount: result.deletedCount,
			};
		} catch (error) {
			logger.error("Error cleaning up subscriptions:", error);
			throw new ApiError(500, "Failed to cleanup subscriptions");
		}
	}

	async sendNotificationToUser(username, payload, sentBy = null) {
		const notificationId = `user_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
		
		// Create notification tracking entry
		const notificationEntry = new PushNotification({
			bulkNotificationId: notificationId,
			title: payload.title,
			body: payload.body,
			payload,
			icon: payload.icon,
			badge: payload.badge,
			image: payload.image,
			url: payload.url,
			actions: payload.actions,
			status: "processing",
			targetType: "user",
			targetUsers: [username],
			sentBy,
			startedAt: new Date(),
		});

		try {
			const subscriptions = await PushSubscription.find({
				username,
				isActive: true,
			});

			notificationEntry.totalUsers = 1;
			notificationEntry.totalSubscriptions = subscriptions.length;
			await notificationEntry.save();

			if (subscriptions.length === 0) {
				notificationEntry.status = "completed";
				notificationEntry.completedAt = new Date();
				notificationEntry.duration = "0s";
				await notificationEntry.save();

				logger.warn(`No active subscriptions found for user: ${username}`);
				return {
					success: false,
					message: "No active subscriptions found for user",
					sentCount: 0,
					notificationId,
				};
			}

			const results = await Promise.allSettled(
				subscriptions.map(async (sub) => {
					try {
						await webpush.sendNotification(
							{
								endpoint: sub.endpoint,
								keys: {
									p256dh: sub.subscription.keys.p256dh,
									auth: sub.subscription.keys.auth,
								},
							},
							JSON.stringify(payload),
						);

						await PushSubscription.findByIdAndUpdate(sub._id, {
							lastUsed: new Date(),
						});

						return { success: true, endpoint: sub.endpoint };
					} catch (error) {
						logger.error(`Failed to send to ${sub.endpoint}:`, error);

						if (error.statusCode === 410) {
							await PushSubscription.findByIdAndDelete(sub._id);
						}

						return {
							success: false,
							endpoint: sub.endpoint,
							error: error.message,
						};
					}
				}),
			);

			const sentCount = results.filter((r) => r.value?.success).length;
			const failedCount = results.filter((r) => !r.value?.success).length;

			// Update notification entry with results
			notificationEntry.status = "completed";
			notificationEntry.sentCount = sentCount;
			notificationEntry.failedCount = failedCount;
			notificationEntry.completedAt = new Date();
			const duration = (Date.now() - notificationEntry.startedAt.getTime()) / 1000;
			notificationEntry.duration = `${duration.toFixed(2)}s`;
			await notificationEntry.save();

			logger.info(
				`Sent notification to ${sentCount}/${subscriptions.length} subscriptions for user: ${username}`,
			);
			return {
				success: true,
				message: `Notification sent to ${sentCount}/${subscriptions.length} device(s)`,
				sentCount,
				failedCount,
				totalSubscriptions: subscriptions.length,
				notificationId,
				results: results.map((r) => r.value),
			};
		} catch (error) {
			// Update notification entry with error
			if (notificationEntry) {
				notificationEntry.status = "failed";
				notificationEntry.error = error.message;
				notificationEntry.completedAt = new Date();
				const duration = (Date.now() - notificationEntry.startedAt.getTime()) / 1000;
				notificationEntry.duration = `${duration.toFixed(2)}s`;
				try {
					await notificationEntry.save();
				} catch (saveError) {
					logger.error("Failed to update notification entry:", saveError);
				}
			}
			logger.error("Error sending notification to user:", error);
			throw new ApiError(500, "Failed to send notification to user");
		}
	}

	async sendNotificationToAll(
		payload,
		{ batchSize = 20, delayBetweenBatches = 1000, maxRetries = 2, bulkJobId = null, bulkNotificationId = null, sentBy = null } = {}
	) {
		// Use provided bulkNotificationId or generate one
		const notificationId = bulkNotificationId || `notif_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
		const startTime = Date.now();

		// Create notification tracking entry
		const notificationEntry = new PushNotification({
			bulkNotificationId: notificationId,
			bulkJobId,
			title: payload.title,
			body: payload.body,
			payload,
			icon: payload.icon,
			badge: payload.badge,
			image: payload.image,
			url: payload.url,
			actions: payload.actions,
			status: "processing",
			targetType: "all",
			batchSize,
			sentBy,
			startedAt: new Date(),
		});

		try {
			const subscriptions = await PushSubscription.find({ 
				isActive: true,
			});

			notificationEntry.totalSubscriptions = subscriptions.length;
			await notificationEntry.save();

			if (subscriptions.length === 0) {
				notificationEntry.status = "completed";
				notificationEntry.completedAt = new Date();
				notificationEntry.duration = "0s";
				await notificationEntry.save();

				logger.warn("No active subscriptions found");
				return {
					success: false,
					bulkNotificationId: notificationId,
					trackingId: notificationEntry._id,
					message: "No active subscriptions found",
					sentCount: 0,
				};
			}

			logger.info(`Starting bulk push notification send`, {
				bulkNotificationId: notificationId,
				totalSubscriptions: subscriptions.length,
				batches: Math.ceil(subscriptions.length / batchSize),
				batchSize,
				delayBetweenBatches,
			});

			// Group by unique users (username)
			const userSubscriptions = new Map();
			subscriptions.forEach(sub => {
				if (!userSubscriptions.has(sub.username)) {
					userSubscriptions.set(sub.username, []);
				}
				userSubscriptions.get(sub.username).push(sub);
			});

			const uniqueUsers = Array.from(userSubscriptions.keys());
			
			// Split into batches
			const batches = [];
			for (let i = 0; i < uniqueUsers.length; i += batchSize) {
				batches.push(uniqueUsers.slice(i, i + batchSize));
			}

			const allResults = [];
			let totalSent = 0;
			let totalFailed = 0;

			// Process each batch
			for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
				const batch = batches[batchIndex];
				
				logger.info(`Processing batch ${batchIndex + 1}/${batches.length}`, {
					bulkNotificationId: notificationId,
					batchSize: batch.length,
					progress: `${totalSent + totalFailed}/${uniqueUsers.length}`,
				});

				// Send to users in current batch
				const batchResults = await this._sendBatchWithRetry({
					batch,
					userSubscriptions,
					payload,
					bulkNotificationId: notificationId,
					maxRetries,
				});

				// Collect results
				allResults.push(...batchResults);
				totalSent += batchResults.filter(r => r.success).length;
				totalFailed += batchResults.filter(r => !r.success).length;

				logger.info(`Batch ${batchIndex + 1} completed`, {
					bulkNotificationId: notificationId,
					batchSent: batchResults.filter(r => r.success).length,
					batchFailed: batchResults.filter(r => !r.success).length,
					totalSent,
					totalFailed,
				});

				// Add delay between batches (except after last batch)
				if (batchIndex < batches.length - 1) {
					await this._sleep(delayBetweenBatches);
				}
			}

			const duration = ((Date.now() - startTime) / 1000).toFixed(2);

			logger.info(`Bulk push notification completed`, {
				bulkNotificationId: notificationId,
				totalSent,
				totalFailed,
				duration: `${duration}s`,
				successRate: `${((totalSent / uniqueUsers.length) * 100).toFixed(1)}%`,
			});

			// Update tracking entry with final results
			notificationEntry.status = "completed";
			notificationEntry.totalUsers = uniqueUsers.length;
			notificationEntry.sentCount = totalSent;
			notificationEntry.failedCount = totalFailed;
			notificationEntry.completedAt = new Date();
			notificationEntry.duration = `${duration}s`;
			notificationEntry.batches = batches.length;
			await notificationEntry.save();

			return {
				success: true,
				bulkNotificationId: notificationId,
				trackingId: notificationEntry._id,
				message: `Notification sent to ${totalSent}/${uniqueUsers.length} user(s)`,
				sentCount: totalSent,
				failedCount: totalFailed,
				totalUsers: uniqueUsers.length,
				totalSubscriptions: subscriptions.length,
				duration: `${duration}s`,
				batches: batches.length,
				results: allResults,
			};
		} catch (error) {
			// Update tracking entry with error
			if (notificationEntry) {
				notificationEntry.status = "failed";
				notificationEntry.error = error.message;
				notificationEntry.completedAt = new Date();
				const duration = ((Date.now() - startTime) / 1000).toFixed(2);
				notificationEntry.duration = `${duration}s`;
				try {
					await notificationEntry.save();
				} catch (saveError) {
					logger.error("Failed to update notification tracking entry:", saveError);
				}
			}
			logger.error("Error sending notification to all:", error);
			throw new ApiError(500, "Failed to send notification to all users");
		}
	}

	async _sendBatchWithRetry({ batch, userSubscriptions, payload, bulkNotificationId, maxRetries }) {
		const results = [];

		for (const username of batch) {
			const subs = userSubscriptions.get(username);
			let lastError = null;
			let sent = false;

			// Retry logic for this user's devices
			for (let attempt = 1; attempt <= maxRetries; attempt++) {
				try {
					const subResults = await Promise.allSettled(
						subs.map(async (sub) => {
							try {
								await webpush.sendNotification(
									{
										endpoint: sub.endpoint,
										keys: {
											p256dh: sub.subscription.keys.p256dh,
											auth: sub.subscription.keys.auth,
										},
									},
									JSON.stringify(payload),
								);

								await PushSubscription.findByIdAndUpdate(sub._id, {
									lastUsed: new Date(),
								});

								return { success: true, endpoint: sub.endpoint };
							} catch (error) {
								logger.error(`Failed to send to ${sub.endpoint}:`, error);

								// Remove expired subscriptions
								if (error.statusCode === 410 || error.statusCode === 404) {
									await PushSubscription.findByIdAndDelete(sub._id);
								}

								throw error;
							}
						})
					);

					const successCount = subResults.filter(r => r.status === "fulfilled").length;
					
					if (successCount > 0) {
						results.push({
							success: true,
							username,
							devicesReached: successCount,
							totalDevices: subs.length,
							attempt,
						});
						sent = true;
						break; // Success, exit retry loop
					} else {
						throw new Error("All devices failed");
					}
				} catch (error) {
					lastError = error;
					
					if (attempt < maxRetries) {
						const delay = Math.pow(2, attempt - 1) * 500; // 500ms, 1s
						logger.warn(`Retry ${attempt}/${maxRetries} for user ${username} after ${delay}ms`, {
							bulkNotificationId,
							error: error.message,
						});
						await this._sleep(delay);
					}
				}
			}

			// If all retries failed
			if (!sent) {
				results.push({
					success: false,
					username,
					devicesReached: 0,
					totalDevices: subs.length,
					error: lastError?.message || "Unknown error",
					retriesExhausted: true,
				});

				logger.error(`Failed to send to user ${username} after ${maxRetries} retries`, {
					bulkNotificationId,
					error: lastError?.message,
				});
			}
		}

		return results;
	}

	_sleep(ms) {
		return new Promise(resolve => setTimeout(resolve, ms));
	}

	async getBulkNotificationStatus(bulkNotificationId) {
		try {
			const notification = await PushNotification.findOne({ 
				bulkNotificationId 
			});

			if (!notification) {
				throw new ApiError(404, "Bulk notification not found");
			}

			return {
				trackingId: notification._id,
				bulkNotificationId: notification.bulkNotificationId,
				bulkJobId: notification.bulkJobId,
				status: notification.status,
				title: notification.title,
				body: notification.body,
				targetType: notification.targetType,
				stats: {
					totalUsers: notification.totalUsers,
					totalSubscriptions: notification.totalSubscriptions,
					sent: notification.sentCount,
					failed: notification.failedCount,
					pending: notification.totalUsers - notification.sentCount - notification.failedCount,
				},
				payload: notification.payload,
				startedAt: notification.startedAt,
				completedAt: notification.completedAt,
				duration: notification.duration,
				batches: notification.batches,
				sentBy: notification.sentBy,
				error: notification.error,
				createdAt: notification.createdAt,
			};
		} catch (error) {
			if (error instanceof ApiError) throw error;
			logger.error("Error getting bulk notification status:", error);
			throw new ApiError(500, "Failed to retrieve bulk notification status");
		}
	}

	async getNotificationHistory({ status, sentBy, limit = 50, page = 1 } = {}) {
		try {
			const filter = {};
			if (status) filter.status = status;
			if (sentBy) filter.sentBy = sentBy;

			const skip = (page - 1) * limit;

			const notifications = await PushNotification.find(filter)
				.sort({ createdAt: -1 })
				.limit(limit)
				.skip(skip);

			const total = await PushNotification.countDocuments(filter);

			return {
				notifications,
				total,
				page,
				limit,
				totalPages: Math.ceil(total / limit),
			};
		} catch (error) {
			logger.error("Error getting notification history:", error);
			throw new ApiError(500, "Failed to retrieve notification history");
		}
	}
}

module.exports = new PushSubscriptionService();
