const { pancakeswapService } = require("./pancakeswapTransaction.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const { AdminConfig } = require("../admin/adminConfig.model");
const bcrypt = require("bcryptjs");

// Helper to verify master password
async function verifyMasterPassword(masterPassword) {
	if (!masterPassword) {
		throw new ApiError(401, "Admin master password required");
	}

	const config = await AdminConfig.findOne({ type: "master_password" });
	if (!config || !config.passwordHash) {
		throw new ApiError(
			500,
			"Admin configuration not found. Please contact support.",
		);
	}

	const isPasswordValid = await bcrypt.compare(
		masterPassword.trim(),
		config.passwordHash,
	);
	if (!isPasswordValid) {
		throw new ApiError(401, "Invalid admin master password");
	}

	return true;
}

const pancakeswapController = {
	// Get PancakeSwap statistics
	async getStats(req, res, next) {
		try {
			const { timeRange = "24h" } = req.query;

			const stats = await pancakeswapService.getStats(timeRange);

			res.json(
				new ApiResponse(200, { stats }, "Statistics retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get recent transactions
	async getTransactions(req, res, next) {
		try {
			const {
				limit = 50,
				type = null,
				search = null,
				page = 1,
				pageSize = 15,
				sortBy = "time-desc",
				startDate = null,
				endDate = null,
				myTrades = null,
			} = req.query;

			const isAdmin = req.user?.role === "admin";
			const currentUsername = req.user?.username || null;

			const result = await pancakeswapService.getRecentTransactions({
				limit: parseInt(limit),
				type,
				search,
				page: parseInt(page),
				pageSize: parseInt(pageSize),
				sortBy,
				isAdmin,
				currentUsername,
				startDate,
				endDate,
				myTrades: myTrades === "true" ? true : false,
			});

			res.json(
				new ApiResponse(
					200,
					{
						transactions: result.transactions,
						count: result.count,
						totalCount: result.totalCount,
						totalPages: result.totalPages,
						currentPage: result.currentPage,
						pageSize: result.pageSize,
						buyCount: result.buyCount,
						sellCount: result.sellCount,
						transferCount: result.transferCount,
					},
					"Transactions retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get summary (stats + recent transactions)
	async getSummary(req, res, next) {
		try {
			const { timeRange = "24h" } = req.query;
			const isAdmin = req.user?.role === "admin";
			const currentUsername = req.user?.username || null;

			const result = await pancakeswapService.getSummary(
				timeRange,
				isAdmin,
				currentUsername,
			);

			res.json(new ApiResponse(200, result, "Summary retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Track transactions (admin only)
	async trackTransactions(req, res, next) {
		try {
			const result = await pancakeswapService.trackTransactions();

			res.json(
				new ApiResponse(
					200,
					result,
					"PancakeSwap tracking completed successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Repair transactions (admin only)
	async repairTransactions(req, res, next) {
		try {
			const result = await pancakeswapService.repairTransactions();

			res.json(
				new ApiResponse(
					200,
					result,
					"Transaction repair completed successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get OHLC data for charts
	async getOHLC(req, res, next) {
		try {
			const { timeRange = "24h", interval = null } = req.query;

			const ohlcData = await pancakeswapService.getOHLC(timeRange, interval);

			res.json(
				new ApiResponse(
					200,
					{ ohlc: ohlcData },
					"OHLC data retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// ==========================================
	// Master Password Protected Routes (no JWT)
	// ==========================================

	// Get all transactions with master password (no JWT required)
	async getAllTransactions(req, res, next) {
		try {
			const masterPassword = req.headers["x-admin-master-password"];
			await verifyMasterPassword(masterPassword);

			const {
				limit = 50,
				type = null,
				search = null,
				page = 1,
				pageSize = 15,
				sortBy = "time-desc",
				startDate = null,
				endDate = null,
			} = req.query;

			// Master password = admin access
			const result = await pancakeswapService.getRecentTransactions({
				limit: parseInt(limit),
				type,
				search,
				page: parseInt(page),
				pageSize: parseInt(pageSize),
				sortBy,
				isAdmin: true,
				currentUsername: null,
				startDate,
				endDate,
			});

			res.json(
				new ApiResponse(
					200,
					{
						transactions: result.transactions,
						count: result.count,
						totalCount: result.totalCount,
						totalPages: result.totalPages,
						currentPage: result.currentPage,
						pageSize: result.pageSize,
						buyCount: result.buyCount,
						sellCount: result.sellCount,
						transferCount: result.transferCount,
					},
					"Transactions retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get stats with master password (no JWT required)
	async getAllStats(req, res, next) {
		try {
			const masterPassword = req.headers["x-admin-master-password"];
			await verifyMasterPassword(masterPassword);

			const { timeRange = "24h" } = req.query;

			const stats = await pancakeswapService.getStats(timeRange);

			res.json(
				new ApiResponse(200, { stats }, "Statistics retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get summary with master password (no JWT required)
	async getAllSummary(req, res, next) {
		try {
			const masterPassword = req.headers["x-admin-master-password"];
			await verifyMasterPassword(masterPassword);

			const { timeRange = "24h" } = req.query;

			const result = await pancakeswapService.getSummary(
				timeRange,
				true, // isAdmin
				null, // no username filter
			);

			res.json(new ApiResponse(200, result, "Summary retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Get OHLC data with master password (no JWT required)
	async getAllOHLC(req, res, next) {
		try {
			const masterPassword = req.headers["x-admin-master-password"];
			await verifyMasterPassword(masterPassword);

			const { timeRange = "24h", interval = null } = req.query;

			const ohlcData = await pancakeswapService.getOHLC(timeRange, interval);

			res.json(
				new ApiResponse(
					200,
					{ ohlc: ohlcData },
					"OHLC data retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { pancakeswapController };
