const { PancakeswapTransaction } = require("./pancakeswapTransaction.model");
const { User } = require("../users/user.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { sha256 } = require("../../utils/crypto.util");
const { createPublicClient, http, parseAbiItem } = require("viem");
const { bsc } = require("viem/chains");
const { ethers } = require("ethers");

// Environment variables
const LIQUIDITY_PAIR_ADDRESS = process.env.TOKEN_BNB_PAIR;
const TOKEN_CONTRACT_ADDRESS = process.env.TOKEN_CONTRACT_ADDRESS;
const ANKR_BSC = process.env.ANKR_BSC;

// PancakeSwap Router Addresses
const PANCAKESWAP_ROUTERS = [
	"0x10ed43c718714eb63d5aa57b78b54704e256024e",
	"0x13f4ea83d0bd40e75c8222255bc855a974568dd4",
	"0x1b81d678ffb9c0263b24a97847620c99d213eb14",
].map((addr) => addr.toLowerCase());

// Custom transaction types
const PURCHASE_ADDRESSES = {
	"0xf1fc0bd2be1a1288e68ea9e14a53a24101a91d28": "core_node_purchase",
	"0x922fb56f472cac1b40d436f7932f3a2e06d790df": "elite_node_purchase",
	"0xfe0f48e0dd00b49aaaa0130e5c676c2d4b2221a5": "meta_pulse_purchase",
	"0x2a5631061364ef781766b04db5c9f96cc17bf3e7": "staking_purchase",
	"0xcad5192315231187ee48d366653bb2ba29c9bcc2": "dbe_purchase",
};

const REWARD_ADDRESSES = {
	"0xb8bd75021c4c81a08417a0d5140d1339f7e720e7": "core_reward",
	"0x7a6b8e9a3f8a3e1ef16f1d6b891688417abf7b77": "elite_reward",
	"0x1452b879d0391cac43f7a49e4884bc9cf6ec04c4": "staking_others_reward",
	"0x27e2895b1b25fce2e4e93a999c76fbf105256141": "meta_pulse_reward",
	"0x44ed6f76e3d8e217e8091b62c6b6929fa003cc89": "dbe_reward",
};

// Create wallet hash for address lookup
const createWalletHash = (address) => {
	return sha256(address.toLowerCase());
};

// Get public client for BSC
function getPublicClient() {
	return createPublicClient({
		chain: bsc,
		transport: http(ANKR_BSC, {
			timeout: 60_000,
			batch: false,
			retryCount: 3,
		}),
	});
}

// Fetch transfer logs from RPC
async function fetchTransfersDirectRPC(fromBlock, toBlock) {
	const publicClient = getPublicClient();

	let transferLogs = [];
	let swapLogs = [];

	try {
		transferLogs = await publicClient.getLogs({
			address: TOKEN_CONTRACT_ADDRESS,
			event: parseAbiItem(
				"event Transfer(address indexed from, address indexed to, uint256 value)",
			),
			fromBlock: BigInt(fromBlock),
			toBlock: BigInt(toBlock),
		});
	} catch (error) {
		logger.warn("Failed to fetch transfer logs:", error.message);
	}

	try {
		swapLogs = await publicClient.getLogs({
			address: LIQUIDITY_PAIR_ADDRESS,
			event: parseAbiItem(
				"event Swap(address indexed sender, uint256 amount0In, uint256 amount1In, uint256 amount0Out, uint256 amount1Out, address indexed to)",
			),
			fromBlock: BigInt(fromBlock),
			toBlock: BigInt(toBlock),
		});
	} catch (error) {
		logger.warn("Failed to fetch swap logs:", error.message);
	}

	const swapsByTxHash = {};
	for (const swap of swapLogs) {
		swapsByTxHash[swap.transactionHash] = {
			amount0In: swap.args.amount0In?.toString() || "0",
			amount1In: swap.args.amount1In?.toString() || "0",
			amount0Out: swap.args.amount0Out?.toString() || "0",
			amount1Out: swap.args.amount1Out?.toString() || "0",
		};
	}

	const blockNumbers = [...new Set(transferLogs.map((log) => log.blockNumber))];
	const blockTimestamps = {};

	for (const num of blockNumbers) {
		try {
			const block = await publicClient.getBlock({ blockNumber: num });
			blockTimestamps[num.toString()] = Number(block.timestamp);
		} catch (error) {
			logger.warn(`Failed to fetch block ${num}:`, error.message);
			blockTimestamps[num.toString()] = Date.now() / 1000;
		}
	}

	return transferLogs.map((log) => ({
		transactionHash: log.transactionHash,
		blockNumber: Number(log.blockNumber),
		timestamp: blockTimestamps[log.blockNumber.toString()],
		topics: [log.topics[0], log.topics[1], log.topics[2]],
		data: log.data,
		swapData: swapsByTxHash[log.transactionHash] || null,
	}));
}

// Fetch DexScreener data
async function fetchDexScreenerData(pairAddress) {
	try {
		const url = `https://api.dexscreener.com/latest/dex/pairs/bsc/${pairAddress?.toLowerCase()}`;
		const response = await fetch(url);
		const data = await response.json();

		if (!data?.pair) throw new Error("Invalid response from DexScreener API");

		const pair = data.pair;
		const priceUsd = Number(pair.priceUsd ?? pair.price?.usd ?? 0) || 0;
		const priceNative =
			Number(pair.priceNative ?? pair.price?.native ?? 0) || 0;
		const rawTxns = pair.txns || {};
		const txns = {
			m5: rawTxns.m5 || rawTxns["5m"] || {},
			h1: rawTxns.h1 || rawTxns["1h"] || {},
			h6: rawTxns.h6 || rawTxns["6h"] || {},
			h24: rawTxns.h24 || rawTxns["24h"] || {},
		};
		const volume = Object.assign({}, pair.volume || {});
		const priceChange = Object.assign({}, pair.priceChange || {});

		return {
			success: true,
			data: {
				priceUsd,
				priceNative,
				price: {
					usd: priceUsd,
					native: priceNative,
				},
				txns: txns,
				volume: {
					h24: Number(volume.h24 ?? volume["24h"] ?? 0) || 0,
					h6: Number(volume.h6 ?? 0) || 0,
					h1: Number(volume.h1 ?? 0) || 0,
					m5: Number(volume.m5 ?? 0) || 0,
					"24h": Number(volume["24h"] ?? volume.h24 ?? 0) || 0,
				},
				priceChange: {
					h24: Number(priceChange.h24 ?? priceChange["24h"] ?? 0) || 0,
					h6: Number(priceChange.h6 ?? 0) || 0,
					h1: Number(priceChange.h1 ?? 0) || 0,
					m5: Number(priceChange.m5 ?? 0) || 0,
				},
				liquidity: {
					usd: Number(pair.liquidity?.usd ?? 0) || 0,
					base: Number(pair.liquidity?.base ?? 0) || 0,
					quote: Number(pair.liquidity?.quote ?? 0) || 0,
				},
				fdv: Number(pair.fdv ?? 0) || 0,
				marketCap: Number(pair.marketCap ?? 0) || 0,
				pairCreatedAt: pair.pairCreatedAt
					? new Date(pair.pairCreatedAt).toISOString()
					: null,
				rawData: data,
			},
		};
	} catch (error) {
		logger.error("Error fetching DexScreener data:", error);
		return { success: false, error: error.message };
	}
}

const pancakeswapService = {
	// Get transaction statistics from DexScreener data
	async getStats(timeRange = "24h") {
		try {
			const dexResult = await fetchDexScreenerData(LIQUIDITY_PAIR_ADDRESS);
			if (!dexResult.success) {
				throw new ApiError(500, dexResult.error);
			}
			const pair = dexResult.data;

			let txnData = null,
				volumeData = 0,
				priceChangeData = 0;

			if (pair.txns) {
				switch (timeRange) {
					case "5m":
						txnData = pair.txns.m5 || {};
						volumeData = pair.volume?.m5 || 0;
						priceChangeData = pair.priceChange?.m5 || 0;
						break;
					case "1h":
						txnData = pair.txns.h1 || {};
						volumeData = pair.volume?.h1 || 0;
						priceChangeData = pair.priceChange?.h1 || 0;
						break;
					case "6h":
						txnData = pair.txns.h6 || {};
						volumeData = pair.volume?.h6 || 0;
						priceChangeData = pair.priceChange?.h6 || 0;
						break;
					default:
						txnData = pair.txns.h24 || {};
						volumeData = pair.volume?.["24h"] || pair.volume?.h24 || 0;
						priceChangeData =
							pair.priceChange?.["24h"] || pair.priceChange?.h24 || 0;
				}
			}

			return {
				currentPrice: Number(pair.price?.usd ?? pair.priceUsd ?? 0) || 0,
				priceChange: priceChangeData,
				liquidity: {
					usd: pair.liquidity?.usd || 0,
					base: pair.liquidity?.base || 0,
					quote: pair.liquidity?.quote || 0,
				},
				marketCap: pair.marketCap || 0,
				fdv: pair.fdv || 0,
				buys: txnData?.buys || 0,
				sells: txnData?.sells || 0,
				selectedTimeRange: timeRange,
				txns: txnData || {},
				volume: volumeData || 0,
			};
		} catch (error) {
			logger.error("Error getting PancakeSwap stats:", error);
			throw error;
		}
	},

	// Get recent transactions
	async getRecentTransactions(options) {
		const {
			limit = 100,
			type = null,
			search = null,
			page = 1,
			pageSize = 15,
			sortBy = "time-desc",
			isAdmin = false,
			currentUsername = null,
			startDate = null,
			endDate = null,
			myTrades,
		} = options;

		try {
			// Build the initial query
			const query = { processed: true };

			// Filter by user if not admin
			/* if (!isAdmin && currentUsername) {
				query.$or = [
					{ usernameFrom: currentUsername },
					{ usernameTo: currentUsername },
				];
			} */


			// Filter for the user
			if (myTrades) {
				query.$or = [
					{ usernameFrom: currentUsername },
					{ usernameTo: currentUsername },
				];
			}

			const allowedTypes = [
				"buy",
				"sell",
				"transfer",
				"core_node_purchase",
				"elite_node_purchase",
				"meta_pulse_purchase",
				"staking_purchase",
				"dbe_purchase",
				"core_reward",
				"elite_reward",
				"staking_others_reward",
				"meta_pulse_reward",
				"dbe_reward",
			];

			if (type && allowedTypes.includes(type)) {
				query.type = type;
			}

			// Add date range filtering
			if (startDate || endDate) {
				query.timestamp = {};
				if (startDate) {
					query.timestamp.$gte = new Date(startDate);
				}
				if (endDate) {
					query.timestamp.$lt = new Date(endDate);
				}
			}

			const totalCount = await PancakeswapTransaction.countDocuments(query);

			if (search && search.trim()) {
				const searchRegex = new RegExp(search.trim(), "i");
				const orConditions = [
					{ transactionHash: searchRegex },
					{ from: searchRegex },
					{ to: searchRegex },
					{ type: searchRegex },
				];

				if (isAdmin) {
					orConditions.push(
						{ usernameFrom: searchRegex },
						{ usernameTo: searchRegex },
					);
				}

				// If user is not admin, combine user filter with search
				if (!isAdmin && currentUsername && query.$or) {
					query.$and = [{ $or: query.$or }, { $or: orConditions }];
					delete query.$or;
				} else {
					query.$or = orConditions;
				}
			}

			// Get type counts
			const typeCounts = await PancakeswapTransaction.aggregate([
				{ $match: query },
				{ $group: { _id: "$type", count: { $sum: 1 } } },
			]);

			const buyCount = typeCounts.find((c) => c._id === "buy")?.count || 0;
			const sellCount = typeCounts.find((c) => c._id === "sell")?.count || 0;
			const transferCount =
				typeCounts.find((c) => c._id === "transfer")?.count || 0;

			const skip = (page - 1) * pageSize;

			let sort = {};
			const [field, direction] = sortBy.split("-");
			const sortDirection = direction === "desc" ? -1 : 1;

			switch (field) {
				case "amount":
					sort = { valueFormatted: sortDirection };
					break;
				case "price":
					sort = { priceUSD: sortDirection };
					break;
				case "value":
					sort = { valueUSD: sortDirection };
					break;
				case "time":
				default:
					sort = { timestamp: sortDirection };
					break;
			}

			const transactions = await PancakeswapTransaction.find(query)
				.sort(sort)
				.skip(skip)
				.limit(pageSize)
				.lean();
			console.log("🐞 ~ pancakeswapTransaction.service.js:387 ~ transactions:", transactions)

			const totalPages = Math.ceil(totalCount / pageSize);

			return {
				transactions: transactions.map((tx) => {
					const base = {
						...tx,
						_id: tx._id.toString(),
					};
					if (!isAdmin) {
						delete base.usernameFrom;
						delete base.usernameTo;
					}
					return base;
				}),
				count: transactions.length,
				totalCount,
				totalPages,
				currentPage: page,
				pageSize,
				buyCount,
				sellCount,
				transferCount,
			};
		} catch (error) {
			logger.error("Error getting recent transactions:", error);
			throw error;
		}
	},

	// Get summary (stats + recent transactions)
	async getSummary(timeRange, isAdmin, currentUsername) {
		try {
			const [stats, txResult] = await Promise.all([
				this.getStats(timeRange),
				this.getRecentTransactions({
					limit: 10,
					page: 1,
					pageSize: 10,
					sortBy: "time-desc",
					isAdmin,
					currentUsername,
				}),
			]);

			return {
				stats,
				recentTransactions: txResult.transactions,
			};
		} catch (error) {
			logger.error("Error getting summary:", error);
			throw error;
		}
	},

	// Track PancakeSwap transactions
	async trackTransactions() {
		if (!LIQUIDITY_PAIR_ADDRESS || !TOKEN_CONTRACT_ADDRESS) {
			throw new ApiError(500, "Missing pair or token address in env");
		}

		logger.info("Starting PancakeSwap transaction tracking...");

		try {
			const publicClient = getPublicClient();
			const ethersProvider = new ethers.JsonRpcProvider(ANKR_BSC);
			let currentBlock;

			try {
				currentBlock = await publicClient.getBlockNumber();
			} catch (error) {
				logger.error("Failed to get current block number:", error);
				throw new ApiError(500, "Unable to connect to RPC provider");
			}

			logger.info(`Current BSC block number: ${currentBlock}`);

			const lastTx = await PancakeswapTransaction.findOne()
				.sort({ blockNumber: -1 })
				.lean();

			logger.info(
				`Last processed block number: ${lastTx ? lastTx.blockNumber : "none"}`,
			);

			let fromBlock;
			if (lastTx) {
				fromBlock = BigInt(lastTx.blockNumber) + 1n;
			} else {
				fromBlock = currentBlock - 900n;
			}

			logger.info(
				`Fetching logs from block ${fromBlock} to ${currentBlock}...`,
			);

			if (fromBlock >= currentBlock) {
				return {
					buys: 0,
					sells: 0,
					transfers: 0,
					totalTransactions: 0,
					blocksProcessed: 0,
					fromBlock: Number(fromBlock),
					toBlock: Number(currentBlock),
				};
			}

			fromBlock = fromBlock > 200n ? fromBlock - 200n : 0n;

			logger.info(`Adjusted fromBlock for overlap: ${fromBlock}`);

			const allTransferLogs = [];
			const maxRange = 1000n;

			for (let fb = fromBlock; fb <= currentBlock; fb += maxRange + 1n) {
				const to = fb + maxRange > currentBlock ? currentBlock : fb + maxRange;
				const logs = await fetchTransfersDirectRPC(fb, to);
				allTransferLogs.push(...logs);
			}

			logger.info(`Fetched ${allTransferLogs.length} transfer logs`);

			if (allTransferLogs.length === 0) {
				return {
					buys: 0,
					sells: 0,
					transfers: 0,
					totalTransactions: 0,
					blocksProcessed: Number(currentBlock - fromBlock + 1n),
					fromBlock: Number(fromBlock),
					toBlock: Number(currentBlock),
				};
			}

			const dexResult = await fetchDexScreenerData(LIQUIDITY_PAIR_ADDRESS);
			if (!dexResult.success) throw new Error(dexResult.error);
			const priceUSD = Number(dexResult.data.price?.usd ?? 0) || 0;
			const priceBNB = Number(dexResult.data.price?.native ?? 0) || 0;

			const pairAddress = LIQUIDITY_PAIR_ADDRESS.toLowerCase();

			const transfersByTxHash = {};
			for (const log of allTransferLogs) {
				transfersByTxHash[log.transactionHash] ||= [];
				transfersByTxHash[log.transactionHash].push(log);
			}

			let buys = 0,
				sells = 0,
				transfers = 0;
			const transactions = [];
			const addressSet = new Set();

			let isToken0 = null;
			try {
				const pairContract = new ethers.Contract(
					LIQUIDITY_PAIR_ADDRESS,
					[
						"function token0() view returns (address)",
						"function token1() view returns (address)",
					],
					ethersProvider,
				);
				const token0Addr = (await pairContract.token0()).toLowerCase();
				const token1Addr = (await pairContract.token1()).toLowerCase();
				const normalizedToken = TOKEN_CONTRACT_ADDRESS.toLowerCase();
				if (token0Addr === normalizedToken) isToken0 = true;
				else if (token1Addr === normalizedToken) isToken0 = false;
			} catch (err) {
				logger.warn("Failed to read token0/token1 from pair contract");
			}

			let tokenDecimals = 18;
			try {
				const tokenContract = new ethers.Contract(
					TOKEN_CONTRACT_ADDRESS,
					["function decimals() view returns (uint8)"],
					ethersProvider,
				);
				const decimals = await tokenContract.decimals();
				if (decimals !== undefined && decimals !== null)
					tokenDecimals = Number(decimals);
			} catch (err) {
				logger.warn("Failed to determine token decimals; defaulting to 18");
			}

			for (const [txHash, txLogs] of Object.entries(transfersByTxHash)) {
				const allAddresses = new Set();
				const transferDetails = txLogs.map((log) => {
					const from = ("0x" + log.topics[1].slice(-40)).toLowerCase();
					const to = ("0x" + log.topics[2].slice(-40)).toLowerCase();
					const value = BigInt(log.data);
					allAddresses.add(from);
					allAddresses.add(to);
					addressSet.add(from);
					addressSet.add(to);
					return { from, to, value, log };
				});

				logger.debug(
					`Processing tx ${txHash} with ${transferDetails.length} transfers`,
				);

				const involvesPair = allAddresses.has(pairAddress);
				const involvesRouter = PANCAKESWAP_ROUTERS.some((r) =>
					allAddresses.has(r),
				);

				let type = null;
				let relevantTransfers = [];

				if (involvesPair || involvesRouter) {
					const fromPair = transferDetails.filter(
						(t) => t.from === pairAddress,
					);
					const toPair = transferDetails.filter((t) => t.to === pairAddress);

					if (fromPair.length > 0 && toPair.length === 0) {
						type = "buy";
						relevantTransfers = fromPair;
						buys++;
					} else if (toPair.length > 0 && fromPair.length === 0) {
						type = "sell";
						relevantTransfers = toPair;
						sells++;
					} else continue;
				} else {
					const amountsByFrom = {};
					for (const t of transferDetails) {
						amountsByFrom[t.from] ||= 0n;
						amountsByFrom[t.from] += t.value;
					}
					let chosenFrom = null;
					let largest = 0n;
					for (const [addr, sum] of Object.entries(amountsByFrom)) {
						const curSum = sum;
						if (curSum > largest) {
							largest = curSum;
							chosenFrom = addr;
						}
					}
					if (!chosenFrom) {
						const max = transferDetails.reduce((a, b) =>
							a.value > b.value ? a : b,
						);
						relevantTransfers = [max];
					} else {
						relevantTransfers = transferDetails.filter(
							(t) => t.from === chosenFrom,
						);
					}
					type = "transfer";
					transfers++;
				}

				if (!relevantTransfers.length) continue;

				const totalValue = relevantTransfers.reduce(
					(sum, t) => sum + t.value,
					0n,
				);
				const mainLog = relevantTransfers[0].log;
				const from = "0x" + mainLog.topics[1].slice(-40);
				const to = "0x" + mainLog.topics[2].slice(-40);
				const tokenAmount = Number(
					ethers.formatUnits(totalValue, tokenDecimals),
				);

				if (PURCHASE_ADDRESSES[to.toLowerCase()])
					type = PURCHASE_ADDRESSES[to.toLowerCase()];
				if (REWARD_ADDRESSES[from.toLowerCase()])
					type = REWARD_ADDRESSES[from.toLowerCase()];

				const valueUSD = tokenAmount * priceUSD;
				const valueBNB = tokenAmount * priceBNB;

				let displayAmount = tokenAmount;
				let displayUSD = valueUSD;
				let displayBNB = valueBNB;

				if (mainLog.swapData) {
					try {
						const { amount0In, amount1In, amount0Out, amount1Out } =
							mainLog.swapData;
						const a0in = amount0In
							? Number(
									ethers.formatUnits(amount0In, isToken0 ? tokenDecimals : 18),
							  )
							: 0;
						const a1in = amount1In
							? Number(
									ethers.formatUnits(amount1In, isToken0 ? 18 : tokenDecimals),
							  )
							: 0;
						const a0out = amount0Out
							? Number(
									ethers.formatUnits(amount0Out, isToken0 ? tokenDecimals : 18),
							  )
							: 0;
						const a1out = amount1Out
							? Number(
									ethers.formatUnits(amount1Out, isToken0 ? 18 : tokenDecimals),
							  )
							: 0;

						let bnbAmount = 0;

						if (type === "buy") {
							if (isToken0 === true) {
								bnbAmount = a1in || a1out || 0;
							} else if (isToken0 === false) {
								bnbAmount = a0in || a0out || 0;
							} else {
								bnbAmount = a0in || a1in || a0out || a1out || 0;
							}
						} else if (type === "sell") {
							if (isToken0 === true) {
								bnbAmount = a1out || a1in || 0;
							} else if (isToken0 === false) {
								bnbAmount = a0out || a0in || 0;
							} else {
								bnbAmount = a0out || a1out || a0in || a1in || 0;
							}
						} else {
							bnbAmount = Math.max(a0in, a1in, a0out, a1out, 0);
						}

						let bnbToUsd = 0;
						if (priceBNB && priceBNB !== 0) bnbToUsd = priceUSD / priceBNB;

						displayBNB = bnbAmount;
						displayUSD = bnbAmount * bnbToUsd || displayUSD || 0;
					} catch (err) {
						logger.warn("Failed to parse swapData amounts", {
							txHash: mainLog.transactionHash,
						});
					}
				}

				if ((displayBNB === 0 || displayUSD === 0) && tokenAmount > 0) {
					logger.warn(
						"Computed zero monetary value for non-zero token amount",
						{
							tx: mainLog.transactionHash,
							tokenAmount,
							priceUSD,
							priceBNB,
							swapData: !!mainLog.swapData,
						},
					);
				}

				transactions.push({
					transactionHash: mainLog.transactionHash,
					blockNumber: mainLog.blockNumber,
					timestamp: new Date(mainLog.timestamp * 1000),
					from,
					to,
					value: totalValue.toString(),
					valueFormatted: displayAmount,
					valueUSD: displayUSD,
					valueBNB: displayBNB,
					tokenAmount: tokenAmount,
					type,
					tokenAddress: TOKEN_CONTRACT_ADDRESS,
					pairAddress: LIQUIDITY_PAIR_ADDRESS,
					priceUSD,
					priceBNB,
					swapData: mainLog.swapData || null,
					processed: true,
				});
			}

			if (transactions.length > 0) {
				const bulkOps = transactions.map((tx) => ({
					updateOne: {
						filter: { transactionHash: tx.transactionHash },
						update: { $set: tx },
						upsert: true,
					},
				}));
				await PancakeswapTransaction.bulkWrite(bulkOps);
			}

			logger.info(`Inserted/Updated ${transactions.length} transactions`);

			// Batch lookup usernames for all unique addresses
			const addressArray = Array.from(addressSet);
			const walletHashes = addressArray.map((addr) => createWalletHash(addr));
			const users = await User.find({
				walletHash: { $in: walletHashes },
			}).lean();
			const userMap = {};
			users.forEach((user) => {
				userMap[user.walletHash] = user.username;
			});

			transactions.forEach((tx) => {
				tx.usernameFrom = userMap[createWalletHash(tx.from)] || null;
				if (!tx.usernameFrom) {
					const addr = tx.from.toLowerCase();
					if (REWARD_ADDRESSES[addr]) {
						tx.usernameFrom = REWARD_ADDRESSES[addr];
					} else if (PANCAKESWAP_ROUTERS.includes(addr)) {
						tx.usernameFrom = "PancakeSwap Router";
					} else if (addr === LIQUIDITY_PAIR_ADDRESS.toLowerCase()) {
						tx.usernameFrom = "PancakeSwap";
					}
				}
				tx.usernameTo = userMap[createWalletHash(tx.to)] || null;
				if (!tx.usernameTo) {
					const addr = tx.to.toLowerCase();
					if (PURCHASE_ADDRESSES[addr]) {
						tx.usernameTo = PURCHASE_ADDRESSES[addr];
					} else if (PANCAKESWAP_ROUTERS.includes(addr)) {
						tx.usernameTo = "PancakeSwap Router";
					} else if (addr === LIQUIDITY_PAIR_ADDRESS.toLowerCase()) {
						tx.usernameTo = "PancakeSwap";
					}
				}
			});

			if (transactions.length > 0) {
				const updateOps = transactions.map((tx) => ({
					updateOne: {
						filter: { transactionHash: tx.transactionHash },
						update: {
							$set: {
								usernameFrom: tx.usernameFrom,
								usernameTo: tx.usernameTo,
							},
						},
					},
				}));
				await PancakeswapTransaction.bulkWrite(updateOps);
			}

			logger.info(
				`Processed ${transactions.length} transactions: ${buys} buys, ${sells} sells, ${transfers} transfers`,
			);

			return {
				buys,
				sells,
				transfers,
				totalTransactions: transactions.length,
				blocksProcessed: Number(currentBlock - fromBlock + 1n),
				fromBlock: Number(fromBlock),
				toBlock: Number(currentBlock),
				message: "Success — fully synced with Ankr",
			};
		} catch (error) {
			logger.error("trackPancakeSwapTransactions error:", error);
			throw error;
		}
	},

	// Repair existing transactions
	async repairTransactions() {
		try {
			const ethersProvider = new ethers.JsonRpcProvider(ANKR_BSC);
			const publicClient = getPublicClient();

			const cursor = PancakeswapTransaction.find({
				$or: [{ valueFormatted: 0 }, { tokenAmount: { $exists: false } }],
			}).cursor();

			const ops = [];

			for await (const tx of cursor) {
				try {
					let tokenDecimals = 18;
					try {
						const tokenContract = new ethers.Contract(
							TOKEN_CONTRACT_ADDRESS,
							["function decimals() view returns (uint8)"],
							ethersProvider,
						);
						const decimals = await tokenContract.decimals();
						if (decimals !== undefined && decimals !== null)
							tokenDecimals = Number(decimals);
					} catch (err) {
						logger.warn(
							"repair failed to read token decimals; defaulting to 18",
						);
					}

					let tokenAmount = Number(ethers.formatUnits(tx.value, tokenDecimals));
					const priceUSD = Number(tx.priceUSD || 0);
					const priceBNB = Number(tx.priceBNB || 0);

					let computedValueBNB = tokenAmount * priceBNB;
					let computedValueUSD = tokenAmount * priceUSD;

					// Handle swapData if exists (like Next.js version)
					if (
						tx.swapData &&
						(tx.swapData.amount0In ||
							tx.swapData.amount1In ||
							tx.swapData.amount0Out ||
							tx.swapData.amount1Out)
					) {
						try {
							const a0in = tx.swapData.amount0In
								? Number(ethers.formatUnits(tx.swapData.amount0In, 18))
								: 0;
							const a1in = tx.swapData.amount1In
								? Number(ethers.formatUnits(tx.swapData.amount1In, 18))
								: 0;
							const a0out = tx.swapData.amount0Out
								? Number(ethers.formatUnits(tx.swapData.amount0Out, 18))
								: 0;
							const a1out = tx.swapData.amount1Out
								? Number(ethers.formatUnits(tx.swapData.amount1Out, 18))
								: 0;
							const bnbAmount = Math.max(a0in, a1in, a0out, a1out, 0);

							if (bnbAmount > 0) {
								computedValueBNB = bnbAmount;
								if (priceBNB && priceBNB !== 0)
									computedValueUSD = bnbAmount * (priceUSD / priceBNB);
							} else {
								// Fallback: re-fetch transfer logs from blockchain
								try {
									const transferLogs = await publicClient.getLogs({
										address: TOKEN_CONTRACT_ADDRESS,
										event: parseAbiItem(
											"event Transfer(address indexed from, address indexed to, uint256 value)",
										),
										fromBlock: BigInt(tx.blockNumber),
										toBlock: BigInt(tx.blockNumber),
									});
									const txLogs = transferLogs.filter(
										(l) => l.transactionHash === tx.transactionHash,
									);
									if (txLogs && txLogs.length > 0) {
										const amountsByFrom = {};
										for (const log of txLogs) {
											const fromAddr = (
												"0x" + log.topics[1].slice(-40)
											).toLowerCase();
											const value = BigInt(log.data);
											amountsByFrom[fromAddr] ||= 0n;
											amountsByFrom[fromAddr] += value;
										}
										let chosenFrom = null;
										let largest = 0n;
										for (const [addr, sum] of Object.entries(amountsByFrom)) {
											const cur = sum;
											if (cur > largest) {
												largest = cur;
												chosenFrom = addr;
											}
										}
										if (largest > 0n) {
											tokenAmount = Number(
												ethers.formatUnits(largest, tokenDecimals),
											);
										}
									}
								} catch (err) {
									logger.warn(
										"repairPancakeSwapTransactions: failed to refetch logs for tx",
										tx.transactionHash,
									);
								}
							}
						} catch (err) {
							logger.warn(
								"repair failed parsing swapData for tx",
								tx.transactionHash,
							);
						}
					}

					ops.push({
						updateOne: {
							filter: { _id: tx._id },
							update: {
								$set: {
									tokenAmount,
									valueFormatted: tokenAmount,
									valueBNB: computedValueBNB,
									valueUSD: computedValueUSD,
								},
							},
						},
					});
				} catch (err) {
					logger.warn("Failed to repair tx", tx.transactionHash);
				}
			}

			if (ops.length > 0) {
				const res = await PancakeswapTransaction.bulkWrite(ops);
				return { modified: res.modifiedCount || 0 };
			}

			return { modified: 0 };
		} catch (err) {
			logger.error("repairPancakeSwapTransactions error:", err);
			throw err;
		}
	},

	// Get OHLC data for charts
	async getOHLC(timeRange = "24h", interval = null) {
		try {
			const defaultMap = {
				"1h": "1m",
				"6h": "5m",
				"24h": "1h",
				"7d": "4h",
			};
			const chosen = interval || defaultMap[timeRange] || "1h";

			const msMap = {
				"1m": 60 * 1000,
				"5m": 5 * 60 * 1000,
				"15m": 15 * 60 * 1000,
				"1h": 60 * 60 * 1000,
				"4h": 4 * 60 * 60 * 1000,
				"1d": 24 * 60 * 60 * 1000,
			};

			const intervalMs = msMap[chosen] || msMap["1h"];

			let hoursBack;
			switch (timeRange) {
				case "1h":
					hoursBack = 1;
					break;
				case "6h":
					hoursBack = 6;
					break;
				case "24h":
					hoursBack = 24;
					break;
				case "7d":
					hoursBack = 7 * 24;
					break;
				default:
					hoursBack = 24;
			}

			const startTime = new Date(Date.now() - hoursBack * 60 * 60 * 1000);

			const txResult = await this.getRecentTransactions({
				page: 1,
				pageSize: 1000,
				sortBy: "time-asc",
				isAdmin: true,
			});

			const pricePoints = txResult.transactions
				.filter((tx) => new Date(tx.timestamp) >= startTime)
				.map((tx) => ({
					time: tx.timestamp,
					price: tx.priceUSD,
				}));

			if (pricePoints.length === 0) {
				return [];
			}

			const ohlcData = [];
			let bucketStart = new Date(pricePoints[0].time).getTime();
			let open = pricePoints[0].price;
			let high = open;
			let low = open;
			let close = open;

			pricePoints.forEach((point) => {
				const time = new Date(point.time).getTime();
				if (time >= bucketStart + intervalMs) {
					ohlcData.push({
						time: new Date(bucketStart).toISOString(),
						open,
						high,
						low,
						close,
					});
					bucketStart = time;
					open = point.price;
					high = point.price;
					low = point.price;
				}
				close = point.price;
				high = Math.max(high, point.price);
				low = Math.min(low, point.price);
			});

			ohlcData.push({
				time: new Date(bucketStart).toISOString(),
				open,
				high,
				low,
				close,
			});

			return ohlcData;
		} catch (error) {
			logger.error("Error getting OHLC data:", error);
			throw error;
		}
	},
};

module.exports = { pancakeswapService };
