const { problemReportService } = require("./problemReport.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const problemReportController = {
	// List user's own reports only
	async listUserReports(req, res, next) {
		try {
			const {
				page = 1,
				limit = 10,
				search = "",
				status = "",
				priority = "",
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				status,
				priority,
				userOnly: true,
				currentUsername: req.user?.username,
			};

			const result = await problemReportService.listReports(options);
			res.json(
				new ApiResponse(200, result, "Your reports retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// List all reports with advanced features (admin only)
	async listReportsAdmin(req, res, next) {
		try {
			const {
				page = 1,
				limit = 10,
				search = "",
				status = "",
				priority = "",
				username = "",
				startDate = null,
				endDate = null,
				sortBy = "createdAt",
				sortOrder = "desc",
				hasUnreadComments = null,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				status,
				priority,
				username,
				startDate,
				endDate,
				sortBy,
				sortOrder,
				hasUnreadComments: hasUnreadComments === "true",
				isAdmin: true,
			};

			const result = await problemReportService.listReportsAdmin(options);
			res.json(
				new ApiResponse(200, result, "All reports retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get statistics (admin only)
	async getStats(req, res, next) {
		try {
			const stats = await problemReportService.getStats();
			res.json(
				new ApiResponse(200, stats, "Statistics retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// List all problem reports
	async listReports(req, res, next) {
		try {
			const {
				page = 1,
				limit = 10,
				search = "",
				status = "",
				priority = "",
				username = "",
				startDate = null,
				endDate = null,
				userOnly = false,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				status,
				priority,
				username,
				startDate,
				endDate,
				userOnly: userOnly === "true",
				currentUsername: req.user?.username,
			};

			const result = await problemReportService.listReports(options);
			res.json(new ApiResponse(200, result, "Reports retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Get single report by ID
	async getReportById(req, res, next) {
		try {
			const { id } = req.params;
			const username = req.user?.username;
			const isAdmin = req.user?.role === "admin";

			const report = await problemReportService.getReportById(
				id,
				username,
				isAdmin,
			);

			res.json(
				new ApiResponse(200, { report }, "Report retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Create new problem report
	async createReport(req, res, next) {
		try {
			const { title, details, priority } = req.body;
			console.log("🐞 ~ problemReport.controller.js:149 ~  req.body:",  req.body)
			const username = req.user?.username;
			const userEmail = req.user?.email;

			if (!username) {
				throw new ApiError(401, "User authentication required");
			}

			if (!title || !details) {
				throw new ApiError(400, "title and details are required");
			}

			const reportData = {
				username,
				userEmail,
				title: title.trim(),
				details: details.trim(),
				priority: priority || "normal",
			};

			const files = req.files || [];
			console.log("🐞 ~ problemReport.controller.js:170 ~ files:", files)
			const uploadConfig = req.uploadConfig;
			const report = await problemReportService.createReport(
				reportData,
				files,
				uploadConfig,
			);

			res
				.status(201)
				.json(
					new ApiResponse(
						201,
						{ report },
						"Problem report submitted successfully",
					),
				);
		} catch (err) {
			next(err);
		}
	},

	// Update report (admin only)
	async updateReport(req, res, next) {
		try {
			const { id } = req.params;
			const { status, priority, assignedTo } = req.body;
			const username = req.user?.username;
			const isAdmin = req.user?.role === "admin";

			const updateData = {};
			if (status !== undefined) updateData.status = status;
			if (priority !== undefined) updateData.priority = priority;
			if (assignedTo !== undefined) updateData.assignedTo = assignedTo;

			const report = await problemReportService.updateReport(
				id,
				updateData,
				username,
				isAdmin,
			);

			res.json(new ApiResponse(200, { report }, "Report updated successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Delete report (admin only)
	async deleteReport(req, res, next) {
		try {
			const { id } = req.params;
			const isAdmin = req.user?.role === "admin";

			const result = await problemReportService.deleteReport(id, isAdmin);

			res.json(new ApiResponse(200, result, "Report deleted successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Get comments for a report
	async getComments(req, res, next) {
		try {
			const { id } = req.params;
			const username = req.user?.username;
			const isAdmin = req.user?.role === "admin";

			const comments = await problemReportService.getComments(
				id,
				username,
				isAdmin,
			);

			res.json(
				new ApiResponse(200, { comments }, "Comments retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Add comment to report
	async addComment(req, res, next) {
		try {
			const { id } = req.params;
			const { message } = req.body;
			const username = req.user?.username;
			const role = req.user?.role || "user";
			const isAdmin = req.user?.role === "admin";

			if (!message || !message.trim()) {
				throw new ApiError(400, "message is required");
			}

			const commentData = {
				username,
				role,
				message: message.trim(),
				isAdmin,
			};

			const files = req.files || [];
			const uploadConfig = req.uploadConfig;
			const result = await problemReportService.addComment(
				id,
				commentData,
				files,
				uploadConfig,
			);

			res
				.status(201)
				.json(new ApiResponse(201, result, "Comment added successfully"));
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { problemReportController };
