const express = require('express');
const { problemReportController } = require('./problemReport.controller');
const { secure } = require('../../middlewares/security.middleware');
const { uploadManager } = require('../../utils/upload.util');

const router = express.Router();

// Middleware to set where files should be uploaded
const setUploadConfig = (folder, prefix) => (req, res, next) => {
  req.uploadConfig = { folder, prefix };
  next();
};

// Configure upload for problem reports (max 3 files, 5MB each)
const uploadReportAttachments = uploadManager.array('attachments', 3);

// Get stats (admin only)
router.get('/stats', ...secure('AR', { role: 'admin' }), problemReportController.getStats);

// List all reports with advanced features (admin only)
router.get('/admin/list', ...secure('AR', { role: 'admin' }), problemReportController.listReportsAdmin);

// List user's own reports only
router.get('/', ...secure('AR'), problemReportController.listUserReports);

// Get single report by ID
router.get('/:id', ...secure('AR'), problemReportController.getReportById);

// Create new problem report with attachments
router.post(
  '/',
  ...secure('AR'),
  setUploadConfig('problem-reports', 'report'),
  uploadReportAttachments,
  problemReportController.createReport
);

// Update report (admin only)
router.patch('/:id', ...secure('AR', { role: 'admin' }), problemReportController.updateReport);

// Delete report (admin only)
router.delete('/:id', ...secure('AR', { role: 'admin' }), problemReportController.deleteReport);

// Get comments for a report
router.get('/:id/comments', ...secure('AR'), problemReportController.getComments);

// Add comment to report with attachments
router.post(
  '/:id/comments',
  ...secure('AR'),
  setUploadConfig('problem-reports', 'comment'),
  uploadReportAttachments,
  problemReportController.addComment
);

module.exports = router;
