const { ProblemReport } = require("./problemReport.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { uploadManager } = require("../../utils/upload.util");
const systemNotificationService = require("../../utils/system-notification.util");
const generateEmailFromTemplate = require("../../templates/generateEmailFromTemplate");

const SERVER_URL = process.env.SERVER_URL || "http://localhost:5000";

// Helper to prepend server URL to attachment paths
const getFullAttachmentUrl = (attachmentPath) => {
	if (!attachmentPath) return "";
	if (
		attachmentPath.startsWith("http://") ||
		attachmentPath.startsWith("https://")
	) {
		return attachmentPath;
	}
	return `${SERVER_URL}${attachmentPath}`;
};

// Helper to add full URLs to attachments array
const addFullUrlsToAttachments = (attachments) => {
	if (!attachments || !Array.isArray(attachments)) return [];
	return attachments.map((att) => ({
		...att,
		path: getFullAttachmentUrl(att.path),
	}));
};

const problemReportService = {
	// List problem reports with filters and pagination (user-only)
	async listReports(options) {
		const {
			page = 1,
			limit = 10,
			search = "",
			status = "",
			priority = "",
			userOnly = false,
			currentUsername = null,
		} = options;

		const filter = {};

		// User-specific filter
		if (userOnly && currentUsername) {
			filter.username = currentUsername;
		}

		// Status filter
		if (status && status !== "all") {
			filter.status = status;
		}

		// Priority filter
		if (priority && priority !== "all") {
			filter.priority = priority;
		}

		// Search filter (only in title and details for users)
		if (search) {
			filter.$or = [
				{ title: { $regex: search, $options: "i" } },
				{ details: { $regex: search, $options: "i" } },
			];
		}

		const skip = (page - 1) * limit;
		const sort = { createdAt: -1 };

		const [reports, total] = await Promise.all([
			ProblemReport.find(filter).sort(sort).skip(skip).limit(limit).lean(),
			ProblemReport.countDocuments(filter),
		]);

		// Add server URL to attachment paths
		const reportsWithFullUrls = reports.map((report) => ({
			...report,
			attachments: addFullUrlsToAttachments(report.attachments),
			comments:
				report.comments?.map((comment) => ({
					...comment,
					attachments: addFullUrlsToAttachments(comment.attachments),
				})) || [],
		}));

		return {
			reports: reportsWithFullUrls,
			pagination: {
				page,
				limit,
				total,
				totalPages: Math.ceil(total / limit),
			},
		};
	},

	// Get a single report by ID
	async getReportById(id, username = null, isAdmin = false) {
		const report = await ProblemReport.findById(id).lean();
		if (!report) {
			throw new ApiError(404, "Problem report not found");
		}

		// Check permissions - user can only see their own reports unless admin
		if (!isAdmin && username && report.username !== username) {
			throw new ApiError(403, "You can only view your own reports");
		}

		// Add full URLs to attachments
		report.attachments = addFullUrlsToAttachments(report.attachments);
		report.comments =
			report.comments?.map((comment) => ({
				...comment,
				attachments: addFullUrlsToAttachments(comment.attachments),
			})) || [];

		return report;
	},

	// Create new problem report
	async createReport(reportData, files = [], uploadConfig) {
		const {
			username,
			userEmail,
			title,
			details,
			priority = "normal",
		} = reportData;

		// Check for existing open report
		const existingReport = await ProblemReport.findOne({
			username,
			status: { $in: ["open", "in_progress"] },
		});

		if (existingReport) {
			throw new ApiError(
				409,
				"You already have an open problem report. Please wait for it to be resolved before submitting a new one.",
			);
		}

		// Validate attachments
		if (files.length > 3) {
			throw new ApiError(400, "Maximum 3 attachments allowed");
		}

		// Process attachments
		const attachments = [];
		for (const file of files) {
			if (file.size > 5 * 1024 * 1024) {
				throw new ApiError(400, `File ${file.originalname} exceeds 5MB limit`);
			}

			const uploadedFile = await uploadManager.uploadImage(
				file,
				uploadConfig.folder,
				uploadConfig.prefix,
			);

			attachments.push({
				filename: file.originalname,
				path: uploadedFile,
				contentType: file.mimetype,
				size: file.size,
			});
		}

		const report = new ProblemReport({
			username,
			userEmail,
			title,
			details,
			priority,
			attachments,
			status: "open",
			comments: [],
			assignedTo: null,
		});

		await report.save();

		logger.info("Problem report created", {
			reportId: report._id,
			username,
			priority,
			attachmentCount: attachments.length,
		});

		// Send confirmation email to user
		try {
			const priorityColors = {
				low: "#3b82f6",
				normal: "#10b981",
				high: "#f59e0b",
				urgent: "#ef4444",
			};
			const priorityColor = priorityColors[priority] || "#10b981";

			const userEmailBody = `
        <div style="color: #e1e4eb; line-height: 1.8;">
          <h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">✅ Problem Report Submitted</h2>
          
          <p style="margin-bottom: 15px;">Hello <strong>${username}</strong>,</p>
          
          <p style="margin-bottom: 20px;">Your problem report has been successfully submitted to our support team. We'll review it and get back to you as soon as possible.</p>
          
          <div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
            <h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Report Details</h3>
            <table style="width: 100%; border-collapse: collapse;">
              <tr>
                <td style="padding: 8px 0; color: #9aa3b2;">Report ID:</td>
                <td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace;">#${report._id
									.toString()
									.slice(-8)
									.toUpperCase()}</td>
              </tr>
              <tr>
                <td style="padding: 8px 0; color: #9aa3b2;">Title:</td>
                <td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${title}</td>
              </tr>
              <tr>
                <td style="padding: 8px 0; color: #9aa3b2;">Priority:</td>
                <td style="padding: 8px 0; color: ${priorityColor}; text-align: right; font-weight: 600; text-transform: uppercase;">${priority}</td>
              </tr>
              <tr>
                <td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
                <td style="padding: 8px 0; color: #3b82f6; text-align: right; font-weight: 600;">Open</td>
              </tr>
              <tr>
                <td style="padding: 8px 0; color: #9aa3b2;">Submitted:</td>
                <td style="padding: 8px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td>
              </tr>
              <tr>
                <td style="padding: 8px 0; color: #9aa3b2;">Attachments:</td>
                <td style="padding: 8px 0; color: #fff; text-align: right;">${
									attachments.length
								} file(s)</td>
              </tr>
            </table>
          </div>
          
          <div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
            <p style="margin: 0; color: #3b82f6;"><strong>📝 Your Message</strong></p>
            <p style="margin: 10px 0 0 0; color: #e1e4eb; white-space: pre-wrap;">${details}</p>
          </div>
          
          <div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
            <p style="margin: 0; color: #eab308;"><strong>⏱️ What's Next?</strong></p>
            <p style="margin: 10px 0 0 0; color: #e1e4eb;">Our support team will review your report and respond within 24-48 hours. You'll receive an email notification when there's an update.</p>
          </div>
          
          <div style="text-align: center; margin-top: 30px;">
            <a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View Report Status</a>
          </div>
        </div>
      `;

			const userEmailHtml = generateEmailFromTemplate(userEmailBody);

			systemNotificationService.sendToUser({
				username,
				email: userEmail,
				subject: `Problem Report Submitted - #${report._id
					.toString()
					.slice(-8)
					.toUpperCase()}`,
				title: "Report Submitted",
				body: `Your problem report "${title}" has been submitted successfully. Priority: ${priority}.`,
				html: userEmailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: `/support/reports/${report._id}`,
						reportId: report._id.toString(),
					},
				},
				sendEmail: true,
				sendPush: false,
			});
		} catch (notificationError) {
			logger.error("Failed to send problem report confirmation email:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				reportId: report._id,
				username,
			});
		}

		// Send admin notification
		try {
			const priorityColors = {
				low: "#3b82f6",
				normal: "#10b981",
				high: "#f59e0b",
				urgent: "#ef4444",
			};
			const priorityColor = priorityColors[priority] || "#10b981";

			const adminEmailBody = `
        <div style="color: #e1e4eb; line-height: 1.8;">
          <h2 style="color: ${priorityColor}; margin-bottom: 20px; font-size: 22px;">🆘 New Problem Report</h2>
          <p style="margin-bottom: 15px;">A new problem report has been submitted and requires support team attention:</p>

          <div style="background: linear-gradient(135deg, rgba(0, 229, 255, 0.15), rgba(0, 229, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 20px 0; border-radius: 8px;">
            <h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Report Information</h3>
            <table style="width: 100%; border-collapse: collapse;">
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Report ID:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-family: monospace;">${
								report._id
							}</td></tr>
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Short ID:</td><td style="padding: 6px 0; color: #00e5ff; text-align: right; font-family: monospace; font-weight: 600;">#${report._id
								.toString()
								.slice(-8)
								.toUpperCase()}</td></tr>
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Title:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-weight: 600;">${title}</td></tr>
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Priority:</td><td style="padding: 6px 0; color: ${priorityColor}; text-align: right; font-weight: 600; text-transform: uppercase;">${priority}</td></tr>
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Status:</td><td style="padding: 6px 0; color: #3b82f6; text-align: right; font-weight: 600;">Open</td></tr>
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Submitted:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td></tr>
            </table>
          </div>

          <div style="background: linear-gradient(135deg, rgba(139, 92, 246, 0.15), rgba(139, 92, 246, 0.05)); border-left: 4px solid #8b5cf6; padding: 20px; margin: 20px 0; border-radius: 8px;">
            <h3 style="color: #8b5cf6; margin-top: 0; margin-bottom: 15px; font-size: 18px;">User Information</h3>
            <table style="width: 100%; border-collapse: collapse;">
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Username:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-weight: 600;">${username}</td></tr>
              <tr><td style="padding: 6px 0; color: #9aa3b2;">Email:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${userEmail}</td></tr>
            </table>
          </div>

          <div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
            <p style="margin: 0; color: #3b82f6;"><strong>📝 Problem Details</strong></p>
            <p style="margin: 10px 0 0 0; color: #e1e4eb; white-space: pre-wrap;">${details}</p>
          </div>

          <div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
            <p style="margin: 0; color: #eab308;"><strong>📎 Attachments</strong></p>
            <p style="margin: 10px 0 0 0; color: #e1e4eb;">${
							attachments.length > 0
								? `${attachments.length} file(s) attached`
								: "No attachments"
						}</p>
          </div>

          <div style="text-align: center; margin-top: 30px;">
            <a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 12px 24px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 6px; font-weight: 600; margin-right: 10px;">View Report</a>
            <a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 12px 24px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 6px; font-weight: 600;">All Reports</a>
          </div>
        </div>
      `;

			const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

			systemNotificationService.sendToAdmin({
				subject: `🆘 New ${priority.toUpperCase()} Priority Problem Report - ${username}`,
				title: "New Problem Report",
				body: `${username} submitted a ${priority} priority problem report: "${title}"`,
				html: adminEmailHtml,
				metadata: {
					reportId: report._id.toString(),
					username,
					userEmail,
					title,
					priority,
					attachmentCount: attachments.length,
					submittedAt: new Date().toISOString(),
				},
			});
		} catch (adminNotificationError) {
			logger.error("Failed to send admin problem report notification:", {
				errorMessage: adminNotificationError.message,
				errorStack: adminNotificationError.stack,
				reportId: report._id,
			});
		}

		const reportObject = report.toObject();
		reportObject.attachments = addFullUrlsToAttachments(
			reportObject.attachments,
		);
		return reportObject;
	},

	// Update report status/priority/assignment
	async updateReport(id, updateData, username = null, isAdmin = false) {
		const { status, priority, assignedTo } = updateData;

		const report = await ProblemReport.findById(id);
		if (!report) {
			throw new ApiError(404, "Problem report not found");
		}

		// Only admins can update reports
		if (!isAdmin) {
			throw new ApiError(403, "Only admins can update reports");
		}

		// Validate status
		if (status) {
			const validStatuses = ["open", "in_progress", "resolved", "closed"];
			if (!validStatuses.includes(status)) {
				throw new ApiError(400, "Invalid status");
			}
			report.status = status;
		}

		// Validate priority
		if (priority) {
			const validPriorities = ["low", "normal", "high", "urgent"];
			if (!validPriorities.includes(priority)) {
				throw new ApiError(400, "Invalid priority");
			}
			report.priority = priority;
		}

		// Update assignment
		if (assignedTo !== undefined) {
			report.assignedTo = assignedTo;
		}

		await report.save();

		logger.info("Problem report updated", {
			reportId: id,
			status: report.status,
			priority: report.priority,
		});

		// Send status update email to user
		if (status) {
			try {
				const statusColors = {
					open: "#3b82f6",
					in_progress: "#f59e0b",
					resolved: "#10b981",
					closed: "#9aa3b2",
				};
				const statusColor = statusColors[status] || "#3b82f6";
				const statusEmoji = {
					open: "🔵",
					in_progress: "🔄",
					resolved: "✅",
					closed: "⚫",
				};

				const userEmailBody = `
          <div style="color: #e1e4eb; line-height: 1.8;">
            <h2 style="color: ${statusColor}; margin-bottom: 20px; font-size: 24px;">${
					statusEmoji[status] || "📢"
				} Report Status Updated</h2>
            
            <p style="margin-bottom: 15px;">Hello <strong>${
							report.username
						}</strong>,</p>
            
            <p style="margin-bottom: 20px;">The status of your problem report has been updated by our support team.</p>
            
            <div style="background: linear-gradient(135deg, rgba(0, 229, 255, 0.15), rgba(0, 229, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 25px 0; border-radius: 8px;">
              <h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Report Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Report ID:</td>
                  <td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace;">#${id
										.toString()
										.slice(-8)
										.toUpperCase()}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Title:</td>
                  <td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										report.title
									}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">New Status:</td>
                  <td style="padding: 8px 0; color: ${statusColor}; text-align: right; font-weight: 600; text-transform: uppercase;">${status.replace(
					"_",
					" ",
				)}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Priority:</td>
                  <td style="padding: 8px 0; color: #f59e0b; text-align: right; font-weight: 600; text-transform: uppercase;">${
										report.priority
									}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Updated:</td>
                  <td style="padding: 8px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td>
                </tr>
              </table>
            </div>
            
            ${
							status === "resolved"
								? `
            <div style="background: rgba(16, 185, 129, 0.1); border-left: 4px solid #10b981; padding: 15px; margin: 20px 0; border-radius: 8px;">
              <p style="margin: 0; color: #10b981;"><strong>✅ Report Resolved</strong></p>
              <p style="margin: 10px 0 0 0; color: #e1e4eb;">Your issue has been resolved! If you're satisfied with the resolution, the report will be automatically closed. If you still need assistance, please reply to the report.</p>
            </div>
            `
								: ""
						}
            
            ${
							status === "in_progress"
								? `
            <div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
              <p style="margin: 0; color: #eab308;"><strong>🔄 In Progress</strong></p>
              <p style="margin: 10px 0 0 0; color: #e1e4eb;">Our support team is actively working on your issue. We'll keep you updated on the progress.</p>
            </div>
            `
								: ""
						}
            
            <div style="text-align: center; margin-top: 30px;">
              <a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View Report Details</a>
            </div>
          </div>
        `;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username: report.username,
					email: report.userEmail,
					subject: `Report Status Updated: ${status
						.replace("_", " ")
						.toUpperCase()} - #${id.toString().slice(-8).toUpperCase()}`,
					title: "Report Updated",
					body: `Your problem report "${
						report.title
					}" status has been updated to: ${status.replace("_", " ")}.`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: `/support/reports/${id}`,
							reportId: id.toString(),
						},
					},
					sendEmail: true,
					sendPush: false,
				});
			} catch (notificationError) {
				logger.error("Failed to send status update email:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					reportId: id,
				});
			}
		}

		const reportObject = report.toObject();
		reportObject.attachments = addFullUrlsToAttachments(
			reportObject.attachments,
		);
		reportObject.comments =
			reportObject.comments?.map((comment) => ({
				...comment,
				attachments: addFullUrlsToAttachments(comment.attachments),
			})) || [];
		return reportObject;
	},

	// Delete report (admin only)
	async deleteReport(id, isAdmin = false) {
		if (!isAdmin) {
			throw new ApiError(403, "Only admins can delete reports");
		}

		const report = await ProblemReport.findById(id);
		if (!report) {
			throw new ApiError(404, "Problem report not found");
		}

		// Delete attachments
		for (const attachment of report.attachments) {
			try {
				await uploadManager.deleteImage(attachment.path, "problem-reports");
			} catch (err) {
				logger.error("Failed to delete attachment", { path: attachment.path });
			}
		}

		// Delete comment attachments
		for (const comment of report.comments) {
			for (const attachment of comment.attachments) {
				try {
					await uploadManager.deleteImage(attachment.path, "problem-reports");
				} catch (err) {
					logger.error("Failed to delete comment attachment", {
						path: attachment.path,
					});
				}
			}
		}

		await ProblemReport.deleteOne({ _id: id });

		logger.info("Problem report deleted", { reportId: id });

		return { deletedId: id };
	},

	// Add comment to report
	async addComment(reportId, commentData, files = [], uploadConfig) {
		const { username, role, message, isAdmin } = commentData;

		const report = await ProblemReport.findById(reportId);
		if (!report) {
			throw new ApiError(404, "Problem report not found");
		}

		// Check permissions
		const isOwner = username === report.username;
		if (!isAdmin && !isOwner) {
			throw new ApiError(403, "You can only comment on your own reports");
		}

		// Check if user needs to wait for admin response
		if (!isAdmin) {
			if (report.comments.length === 0) {
				throw new ApiError(
					400,
					"Please wait for the support team to respond first",
				);
			}

			const lastComment = report.comments[report.comments.length - 1];
			if (lastComment.role !== "admin") {
				throw new ApiError(
					400,
					"Please wait for the support team to reply before sending another message",
				);
			}
		}

		// Validate attachments
		if (files.length > 3) {
			throw new ApiError(400, "Maximum 3 attachments allowed per comment");
		}

		// Process attachments
		const attachments = [];
		for (const file of files) {
			if (file.size > 5 * 1024 * 1024) {
				throw new ApiError(400, `File ${file.originalname} exceeds 5MB limit`);
			}

			const uploadedFile = await uploadManager.uploadImage(
				file,
				uploadConfig.folder,
				uploadConfig.prefix,
			);

			attachments.push({
				filename: file.originalname,
				path: uploadedFile,
				contentType: file.mimetype,
				size: file.size,
			});
		}

		const comment = {
			role: role || "user",
			message: message.trim(),
			attachments,
			createdAt: new Date(),
		};

		report.comments.push(comment);
		await report.save();

		logger.info("Comment added to report", {
			reportId,
			commentBy: username,
			role: comment.role,
			attachmentCount: attachments.length,
		});

		// Send email notification to user when admin comments
		if (isAdmin && report.userEmail) {
			try {
				const userEmailBody = `
          <div style="color: #e1e4eb; line-height: 1.8;">
            <h2 style="color: #3b82f6; margin-bottom: 20px; font-size: 24px;">💬 New Response from Support Team</h2>
            
            <p style="margin-bottom: 15px;">Hello <strong>${
							report.username
						}</strong>,</p>
            
            <p style="margin-bottom: 20px;">Our support team has responded to your problem report.</p>
            
            <div style="background: linear-gradient(135deg, rgba(0, 229, 255, 0.15), rgba(0, 229, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 25px 0; border-radius: 8px;">
              <h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Report Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Report ID:</td>
                  <td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace;">#${reportId
										.toString()
										.slice(-8)
										.toUpperCase()}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Title:</td>
                  <td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
										report.title
									}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
                  <td style="padding: 8px 0; color: #3b82f6; text-align: right; font-weight: 600; text-transform: uppercase;">${report.status.replace(
										"_",
										" ",
									)}</td>
                </tr>
                <tr>
                  <td style="padding: 8px 0; color: #9aa3b2;">Response Time:</td>
                  <td style="padding: 8px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td>
                </tr>
              </table>
            </div>
            
            <div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
              <p style="margin: 0; color: #3b82f6;"><strong>💬 Support Team Message</strong></p>
              <p style="margin: 10px 0 0 0; color: #e1e4eb; white-space: pre-wrap;">${message.trim()}</p>
              ${
								attachments.length > 0
									? `<p style="margin: 10px 0 0 0; color: #9aa3b2; font-size: 14px;">📎 ${attachments.length} file(s) attached</p>`
									: ""
							}
            </div>
            
            <div style="background: rgba(16, 185, 129, 0.1); border-left: 4px solid #10b981; padding: 15px; margin: 20px 0; border-radius: 8px;">
              <p style="margin: 0; color: #10b981;"><strong>💡 You Can Reply</strong></p>
              <p style="margin: 10px 0 0 0; color: #e1e4eb;">You can now reply to this message through your support dashboard. Our team is here to help!</p>
            </div>
            
            <div style="text-align: center; margin-top: 30px;">
              <a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View & Reply</a>
            </div>
          </div>
        `;

				const userEmailHtml = generateEmailFromTemplate(userEmailBody);

				systemNotificationService.sendToUser({
					username: report.username,
					email: report.userEmail,
					subject: `New Response to Your Report - #${reportId
						.toString()
						.slice(-8)
						.toUpperCase()}`,
					title: "Support Response",
					body: `The support team has responded to your problem report: "${report.title}".`,
					html: userEmailHtml,
					pushPayload: {
						icon: "/icons/icon-192x192.png",
						badge: "/icons/badge-72x72.png",
						data: {
							url: `/support/reports/${reportId}`,
							reportId: reportId.toString(),
						},
					},
					sendEmail: true,
					sendPush: false,
				});
			} catch (notificationError) {
				logger.error("Failed to send comment notification email:", {
					errorMessage: notificationError.message,
					errorStack: notificationError.stack,
					reportId,
				});
			}
		}

		// Send email notification to admin when user comments
		if (!isAdmin) {
			try {
				const priorityColors = {
					low: "#3b82f6",
					normal: "#10b981",
					high: "#f59e0b",
					urgent: "#ef4444",
				};
				const priorityColor = priorityColors[report.priority] || "#10b981";

				const adminEmailBody = `
          <div style="color: #e1e4eb; line-height: 1.8;">
            <h2 style="color: #00e5ff; margin-bottom: 20px; font-size: 22px;">💬 User Reply on Problem Report</h2>
            <p style="margin-bottom: 15px;">A user has replied to their problem report and is waiting for support team response:</p>

            <div style="background: linear-gradient(135deg, rgba(0, 229, 255, 0.15), rgba(0, 229, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 20px 0; border-radius: 8px;">
              <h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Report Information</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Report ID:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-family: monospace;">${reportId}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Short ID:</td><td style="padding: 6px 0; color: #00e5ff; text-align: right; font-family: monospace; font-weight: 600;">#${reportId
									.toString()
									.slice(-8)
									.toUpperCase()}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Title:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-weight: 600;">${
									report.title
								}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Priority:</td><td style="padding: 6px 0; color: ${priorityColor}; text-align: right; font-weight: 600; text-transform: uppercase;">${
					report.priority
				}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Status:</td><td style="padding: 6px 0; color: #3b82f6; text-align: right; font-weight: 600; text-transform: uppercase;">${report.status.replace(
									"_",
									" ",
								)}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Total Comments:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-weight: 600;">${
									report.comments.length
								}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Reply Time:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td></tr>
              </table>
            </div>

            <div style="background: linear-gradient(135deg, rgba(139, 92, 246, 0.15), rgba(139, 92, 246, 0.05)); border-left: 4px solid #8b5cf6; padding: 20px; margin: 20px 0; border-radius: 8px;">
              <h3 style="color: #8b5cf6; margin-top: 0; margin-bottom: 15px; font-size: 18px;">User Information</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Username:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-weight: 600;">${
									report.username
								}</td></tr>
                <tr><td style="padding: 6px 0; color: #9aa3b2;">Email:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${
									report.userEmail
								}</td></tr>
              </table>
            </div>

            <div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
              <p style="margin: 0; color: #3b82f6;"><strong>💬 User's Reply</strong></p>
              <p style="margin: 10px 0 0 0; color: #e1e4eb; white-space: pre-wrap;">${message.trim()}</p>
              ${
								attachments.length > 0
									? `<p style="margin: 10px 0 0 0; color: #9aa3b2; font-size: 14px;">📎 ${attachments.length} file(s) attached</p>`
									: ""
							}
            </div>

            <div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
              <p style="margin: 0; color: #eab308;"><strong>⏱️ Action Required</strong></p>
              <p style="margin: 10px 0 0 0; color: #e1e4eb;">This user is waiting for a response from the support team. Please review and reply as soon as possible.</p>
            </div>

            <div style="text-align: center; margin-top: 30px;">
              <a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 12px 24px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 6px; font-weight: 600; margin-right: 10px;">View & Reply</a>
              <a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 12px 24px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 6px; font-weight: 600;">All Reports</a>
            </div>
          </div>
        `;

				const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

				systemNotificationService.sendToAdmin({
					subject: `💬 User Reply on ${report.priority.toUpperCase()} Priority Report - ${
						report.username
					}`,
					title: "User Reply",
					body: `${report.username} replied to problem report: "${report.title}"`,
					html: adminEmailHtml,
					metadata: {
						reportId: reportId.toString(),
						username: report.username,
						userEmail: report.userEmail,
						title: report.title,
						priority: report.priority,
						status: report.status,
						attachmentCount: attachments.length,
						replyTime: new Date().toISOString(),
						totalComments: report.comments.length,
					},
				});
			} catch (adminNotificationError) {
				logger.error("Failed to send admin user reply notification:", {
					errorMessage: adminNotificationError.message,
					errorStack: adminNotificationError.stack,
					reportId,
				});
			}
		}

		const reportObject = report.toObject();
		reportObject.attachments = addFullUrlsToAttachments(
			reportObject.attachments,
		);
		reportObject.comments =
			reportObject.comments?.map((c) => ({
				...c,
				attachments: addFullUrlsToAttachments(c.attachments),
			})) || [];

		return {
			...reportObject,
			newComment: {
				...comment,
				attachments: addFullUrlsToAttachments(comment.attachments),
			},
		};
	},

	// Get comments for a report
	async getComments(reportId, username = null, isAdmin = false) {
		const report = await ProblemReport.findById(reportId).lean();
		if (!report) {
			throw new ApiError(404, "Problem report not found");
		}

		// Check permissions
		if (!isAdmin && username && report.username !== username) {
			throw new ApiError(403, "You can only view comments on your own reports");
		}

		const comments = (report.comments || []).map((comment) => ({
			...comment,
			attachments: addFullUrlsToAttachments(comment.attachments),
		}));

		return comments;
	},

	// List reports for admin with advanced features
	async listReportsAdmin(options) {
		const {
			page = 1,
			limit = 10,
			search = "",
			status = "",
			priority = "",
			username = "",
			startDate = null,
			endDate = null,
			sortBy = "createdAt",
			sortOrder = "desc",
			hasUnreadComments = false,
		} = options;

		const filter = {};

		// Status filter
		if (status && status !== "all") {
			filter.status = status;
		}

		// Priority filter
		if (priority && priority !== "all") {
			filter.priority = priority;
		}

		// Username filter
		if (username) {
			filter.username = { $regex: username, $options: "i" };
		}

		// Date filter
		if (startDate || endDate) {
			filter.createdAt = {};
			if (startDate) filter.createdAt.$gte = new Date(startDate);
			if (endDate) {
				const end = new Date(endDate);
				end.setHours(23, 59, 59, 999);
				filter.createdAt.$lte = end;
			}
		}

		// Search filter
		if (search) {
			filter.$or = [
				{ username: { $regex: search, $options: "i" } },
				{ title: { $regex: search, $options: "i" } },
				{ details: { $regex: search, $options: "i" } },
			];
		}

		const skip = (page - 1) * limit;

		// Build sort object
		const sort = {};
		if (sortBy === "latestComment") {
			// Sort by latest comment date
			sort["comments.0.createdAt"] = sortOrder === "asc" ? 1 : -1;
		} else if (sortBy === "priority") {
			// Custom priority sorting: urgent > high > normal > low
			const priorityOrder = { urgent: 1, high: 2, normal: 3, low: 4 };
			sort.priority = sortOrder === "asc" ? 1 : -1;
		} else if (sortBy === "status") {
			sort.status = sortOrder === "asc" ? 1 : -1;
		} else {
			// Default: createdAt
			sort.createdAt = sortOrder === "asc" ? 1 : -1;
		}

		const [reports, total] = await Promise.all([
			ProblemReport.find(filter).sort(sort).skip(skip).limit(limit).lean(),
			ProblemReport.countDocuments(filter),
		]);

		// Process reports with enriched data
		const reportsWithEnrichedData = reports.map((report) => {
			const lastComment =
				report.comments?.length > 0
					? report.comments[report.comments.length - 1]
					: null;

			const lastUserComment = report.comments
				?.slice()
				.reverse()
				.find((c) => c.role === "user");
			const hasUnreadUserComment =
				lastUserComment && lastComment?.role === "user";

			return {
				...report,
				attachments: addFullUrlsToAttachments(report.attachments),
				comments:
					report.comments?.map((comment) => ({
						...comment,
						attachments: addFullUrlsToAttachments(comment.attachments),
					})) || [],
				// Enriched data
				lastCommentDate: lastComment?.createdAt || report.createdAt,
				lastCommentBy: lastComment?.role || null,
				hasUnreadUserComment,
				totalComments: report.comments?.length || 0,
			};
		});

		// Filter by unread comments if requested
		let filteredReports = reportsWithEnrichedData;
		if (hasUnreadComments) {
			filteredReports = reportsWithEnrichedData.filter(
				(r) => r.hasUnreadUserComment,
			);
		}

		// Re-sort if sorting by latestComment
		if (sortBy === "latestComment") {
			filteredReports.sort((a, b) => {
				const dateA = new Date(a.lastCommentDate);
				const dateB = new Date(b.lastCommentDate);
				return sortOrder === "asc" ? dateA - dateB : dateB - dateA;
			});
		}

		return {
			reports: filteredReports,
			pagination: {
				page,
				limit,
				total: hasUnreadComments ? filteredReports.length : total,
				totalPages: Math.ceil(
					(hasUnreadComments ? filteredReports.length : total) / limit,
				),
			},
		};
	},

	// Get statistics for admin dashboard
	async getStats() {
		const [
			totalReports,
			openReports,
			inProgressReports,
			resolvedReports,
			closedReports,
			urgentReports,
			highReports,
			reportsLast7Days,
			reportsLast30Days,
			avgResolutionTime,
		] = await Promise.all([
			ProblemReport.countDocuments(),
			ProblemReport.countDocuments({ status: "open" }),
			ProblemReport.countDocuments({ status: "in_progress" }),
			ProblemReport.countDocuments({ status: "resolved" }),
			ProblemReport.countDocuments({ status: "closed" }),
			ProblemReport.countDocuments({ priority: "urgent" }),
			ProblemReport.countDocuments({ priority: "high" }),
			ProblemReport.countDocuments({
				createdAt: { $gte: new Date(Date.now() - 7 * 24 * 60 * 60 * 1000) },
			}),
			ProblemReport.countDocuments({
				createdAt: { $gte: new Date(Date.now() - 30 * 24 * 60 * 60 * 1000) },
			}),
			this.calculateAvgResolutionTime(),
		]);

		// Get reports awaiting admin response
		const reportsAwaitingResponse = await ProblemReport.countDocuments({
			$or: [{ comments: { $size: 0 } }, { "comments.0.role": "user" }],
			status: { $in: ["open", "in_progress"] },
		});

		// Get top users with most reports
		const topReporters = await ProblemReport.aggregate([
			{
				$group: {
					_id: "$username",
					count: { $sum: 1 },
				},
			},
			{ $sort: { count: -1 } },
			{ $limit: 5 },
		]);

		return {
			total: totalReports,
			byStatus: {
				open: openReports,
				in_progress: inProgressReports,
				resolved: resolvedReports,
				closed: closedReports,
			},
			byPriority: {
				urgent: urgentReports,
				high: highReports,
				normal: totalReports - urgentReports - highReports,
			},
			recent: {
				last7Days: reportsLast7Days,
				last30Days: reportsLast30Days,
			},
			awaitingResponse: reportsAwaitingResponse,
			avgResolutionTimeHours: avgResolutionTime,
			topReporters: topReporters.map((r) => ({
				username: r._id,
				count: r.count,
			})),
		};
	},

	// Helper to calculate average resolution time
	async calculateAvgResolutionTime() {
		const resolvedReports = await ProblemReport.find({
			status: { $in: ["resolved", "closed"] },
			updatedAt: { $exists: true },
		}).lean();

		if (resolvedReports.length === 0) return 0;

		const totalHours = resolvedReports.reduce((sum, report) => {
			const created = new Date(report.createdAt);
			const resolved = new Date(report.updatedAt);
			const hours = (resolved - created) / (1000 * 60 * 60);
			return sum + hours;
		}, 0);

		return Math.round(totalHours / resolvedReports.length);
	},
};

module.exports = { problemReportService };
