const productService = require("./product.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

class ProductController {
	/**
	 * Get all products
	 */
	async getAllProducts(req, res, next) {
		try {
			const activeOnly = req.query.activeOnly === "true";
			const includeSellCount = req.query.includeSellCount === "true";
			const packageType = req.query.packageType;

			const products = await productService.getAllProducts({
				activeOnly,
				includeSellCount,
				packageType,
			});

			res
				.status(200)
				.json(
					new ApiResponse(200, products, "Products retrieved successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get product by ID
	 */
	async getProductById(req, res, next) {
		try {
			const { id } = req.params;

			const product = await productService.getProductById(id);

			res
				.status(200)
				.json(
					new ApiResponse(200, product, "Product retrieved successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Create a new product
	 */
	async createProduct(req, res, next) {
		try {
			const product = await productService.createProduct(req.body);

			res
				.status(201)
				.json(new ApiResponse(201, product, "Product created successfully"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Update a product
	 */
	async updateProduct(req, res, next) {
		try {
			const { id } = req.params;

			const product = await productService.updateProduct(id, req.body);

			res
				.status(200)
				.json(new ApiResponse(200, product, "Product updated successfully"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Toggle product status
	 */
	async toggleProductStatus(req, res, next) {
		try {
			const { id } = req.params;
			const { isActive } = req.body;

			const product = await productService.toggleProductStatus(id, isActive);

			res
				.status(200)
				.json(
					new ApiResponse(200, product, "Product status updated successfully")
				);
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Delete a product
	 */
	async deleteProduct(req, res, next) {
		try {
			const { id } = req.params;

			await productService.deleteProduct(id);

			res
				.status(200)
				.json(new ApiResponse(200, null, "Product deleted successfully"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get user counts for all packages
	 */
	async getPackageUserCounts(req, res, next) {
		try {
			const activeOnly = req.query.activeOnly === "true";
			const packageType = req.query.packageType;

			const packageUserCounts = await productService.getPackageUserCounts({
				activeOnly,
				packageType,
			});

			res
				.status(200)
				.json(
					new ApiResponse(
						200,
						packageUserCounts,
						"Package user counts retrieved successfully"
					)
				);
		} catch (error) {
			next(error);
		}
	}
}

module.exports = new ProductController();
