const { Product } = require("./product.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

class ProductService {
	/**
	 * Get all products with optional filters
	 */
	async getAllProducts({
		activeOnly = false,
		includeSellCount = false,
		packageType = null,
	} = {}) {
		try {
			let pipeline = [];

			// Filter only active products
			if (activeOnly) {
				pipeline.push({ $match: { isActive: true } });
			}

			// Filter by package type
			if (packageType) {
				pipeline.push({ $match: { packageType } });
			}

			// Include sell count if requested
			if (includeSellCount) {
				pipeline.push(
					{
						$lookup: {
							from: "userPackages",
							localField: "_id",
							foreignField: "packageId",
							as: "sales",
						},
					},
					{
						$addFields: {
							soldCount: { $size: "$sales" },
							remainingCount: {
								$cond: [
									{ $ifNull: ["$maxSellsCount", false] },
									{
										$max: [
											{ $subtract: ["$maxSellsCount", { $size: "$sales" }] },
											0,
										],
									},
									null,
								],
							},
							sellPercentage: {
								$cond: [
									{
										$and: [
											{ $gt: ["$maxSellsCount", 0] },
											{ $ifNull: ["$maxSellsCount", false] },
										],
									},
									{
										$min: [
											100,
											{
												$round: [
													{
														$multiply: [
															{
																$divide: [
																	{ $size: "$sales" },
																	"$maxSellsCount",
																],
															},
															100,
														],
													},
													0,
												],
											},
										],
									},
									0,
								],
							},
						},
					},
					{ $project: { sales: 0 } },
				);
			}

			// If no pipeline stages, use find() instead of aggregate()
			let products;
			if (pipeline.length === 0) {
				products = await Product.find();
			} else {
				products = await Product.aggregate(pipeline);
			}

			logger.info(`Retrieved ${products.length} products`);
			return products;
		} catch (error) {
			logger.error("Error getting all products:", error);
			throw new ApiError(500, "Failed to retrieve products");
		}
	}

	/**
	 * Get product by ID
	 */
	async getProductById(productId) {
		try {
			const product = await Product.findById(productId);

			if (!product) {
				throw new ApiError(404, "Product not found");
			}

			logger.info(`Retrieved product: ${productId}`);
			return product;
		} catch (error) {
			if (error instanceof ApiError) throw error;
			logger.error("Error getting product by ID:", error);
			throw new ApiError(500, "Failed to retrieve product");
		}
	}

	/**
	 * Create a new product
	 */
	async createProduct(productData) {
		try {
			// Validate required fields
			if (!productData.name || productData.name.trim() === "") {
				throw new ApiError(400, "Name is required");
			}

			// Check if product name already exists
			const existingProduct = await Product.findOne({
				name: productData.name,
			});
			if (existingProduct) {
				throw new ApiError(409, "Product with this name already exists");
			}

			// Basic validation for referralBonusImmediate
			if (
				productData.referralBonusImmediate?.firstGenPercent !== undefined &&
				(productData.referralBonusImmediate.firstGenPercent < 0 ||
					typeof productData.referralBonusImmediate.firstGenPercent !==
						"number")
			) {
				throw new ApiError(
					400,
					"referralBonusImmediate.firstGenPercent must be a non-negative number",
				);
			}

			const product = new Product({
				...productData,
				isActive:
					productData.isActive !== undefined ? productData.isActive : true,
			});

			await product.save();

			logger.info(`Created new product: ${product._id}`);
			return product;
		} catch (error) {
			if (error instanceof ApiError) throw error;
			logger.error("Error creating product:", error);

			if (error.name === "ValidationError") {
				const messages = Object.values(error.errors).map((err) => err.message);
				throw new ApiError(400, `Validation error: ${messages.join(", ")}`);
			}

			throw new ApiError(500, "Failed to create product");
		}
	}

	/**
	 * Update a product
	 */
	async updateProduct(productId, updateData) {
		try {
			// Remove _id from updateData if it exists
			delete updateData._id;

			// Check if name conflicts with another product
			if (updateData.name) {
				const existingProduct = await Product.findOne({
					name: updateData.name,
					_id: { $ne: productId },
				});
				if (existingProduct) {
					throw new ApiError(409, "Product with this name already exists");
				}
			}

			const product = await Product.findByIdAndUpdate(
				productId,
				{ $set: updateData },
				{ new: true, runValidators: true },
			);

			if (!product) {
				throw new ApiError(404, "Product not found");
			}

			logger.info(`Updated product: ${productId}`);
			return product;
		} catch (error) {
			if (error instanceof ApiError) throw error;
			logger.error("Error updating product:", error);

			if (error.name === "ValidationError") {
				const messages = Object.values(error.errors).map((err) => err.message);
				throw new ApiError(400, `Validation error: ${messages.join(", ")}`);
			}

			throw new ApiError(500, "Failed to update product");
		}
	}

	/**
	 * Toggle product active status
	 */
	async toggleProductStatus(productId, isActive) {
		try {
			if (typeof isActive !== "boolean") {
				throw new ApiError(400, "Invalid status value");
			}

			const product = await Product.findByIdAndUpdate(
				productId,
				{ $set: { isActive } },
				{ new: true },
			);

			if (!product) {
				throw new ApiError(404, "Product not found");
			}

			logger.info(
				`Toggled product status: ${productId} - isActive: ${isActive}`,
			);
			return product;
		} catch (error) {
			if (error instanceof ApiError) throw error;
			logger.error("Error toggling product status:", error);
			throw new ApiError(500, "Failed to update product status");
		}
	}

	/**
	 * Delete a product
	 */
	async deleteProduct(productId) {
		try {
			const product = await Product.findByIdAndDelete(productId);

			if (!product) {
				throw new ApiError(404, "Product not found");
			}

			logger.info(`Deleted product: ${productId}`);
			return product;
		} catch (error) {
			if (error instanceof ApiError) throw error;
			logger.error("Error deleting product:", error);
			throw new ApiError(500, "Failed to delete product");
		}
	}

	/**
	 * Get user counts for all packages
	 */
	async getPackageUserCounts({ activeOnly = false, packageType = null } = {}) {
		try {
			let matchConditions = {};

			// Filter only active products
			if (activeOnly) {
				matchConditions.isActive = true;
			}

			// Filter by package type
			if (packageType) {
				matchConditions.packageType = packageType;
			}

			const pipeline = [
				...(Object.keys(matchConditions).length > 0
					? [{ $match: matchConditions }]
					: []),
				{
					$lookup: {
						from: "userPackages",
						localField: "_id",
						foreignField: "packageId",
						as: "users",
					},
				},
				{
					$project: {
						_id: 1,
						name: 1,
						packageType: 1,
						price: 1,
						isActive: 1,
						userCount: { $size: "$users" },
						maxSellsCount: 1,
						remainingCount: {
							$cond: [
								{ $ifNull: ["$maxSellsCount", false] },
								{
									$max: [
										{ $subtract: ["$maxSellsCount", { $size: "$users" }] },
										0,
									],
								},
								null,
							],
						},
					},
				},
				{
					$sort: { userCount: -1 },
				},
			];

			const packageUserCounts = await Product.aggregate(pipeline);

			logger.info(
				`Retrieved user counts for ${packageUserCounts.length} packages`,
			);
			return packageUserCounts;
		} catch (error) {
			logger.error("Error getting package user counts:", error);
			throw new ApiError(500, "Failed to retrieve package user counts");
		}
	}
}

module.exports = new ProductService();
