const { purchaseService } = require("./purchase.service");
const { ApiResponse } = require("../../utils/ApiResponse");

const purchaseController = {
	// Overview stats across all collections
	async getOverview(req, res, next) {
		try {
			const stats = await purchaseService.getOverviewStats();
			res.json(
				new ApiResponse(
					200,
					stats,
					"Overview statistics retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// List analytics with filters
	async getAnalytics(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "timestamp",
				sortOrder = "desc",
				username = "",
				packageId = "",
				status = "",
				startDate = "",
				endDate = "",
				purchaseId = "",
				search = "",
				timezone = 0,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				sortBy,
				sortOrder,
				username,
				packageId,
				status,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone: parseInt(timezone),
			};

			const result = await purchaseService.getAnalytics(options);
			res.json(
				new ApiResponse(
					200,
					result,
					"Purchase analytics retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// List failures with error details
	async getFailures(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "timestamp",
				sortOrder = "desc",
				username = "",
				packageId = "",
				errorType = "",
				startDate = "",
				endDate = "",
				purchaseId = "",
				search = "",
				timezone = 0,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				sortBy,
				sortOrder,
				username,
				packageId,
				errorType,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone: parseInt(timezone),
			};

			const result = await purchaseService.getFailures(options);
			res.json(
				new ApiResponse(
					200,
					result,
					"Purchase failures retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// List locks with status info
	async getLocks(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "createdAt",
				sortOrder = "desc",
				username = "",
				packageId = "",
				status = "",
				startDate = "",
				endDate = "",
				purchaseId = "",
				search = "",
				timezone = 0,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				sortBy,
				sortOrder,
				username,
				packageId,
				status,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone: parseInt(timezone),
			};

			const result = await purchaseService.getLocks(options);
			res.json(
				new ApiResponse(200, result, "Purchase locks retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Remove old expired locks
	async clearLocks(req, res, next) {
		try {
			const result = await purchaseService.clearExpiredLocks();
			res.json(
				new ApiResponse(
					200,
					result,
					`${result.deletedCount} expired/old purchase locks cleared successfully`,
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// List successful purchases
	async getSuccesses(req, res, next) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "timestamp",
				sortOrder = "desc",
				username = "",
				packageId = "",
				startDate = "",
				endDate = "",
				purchaseId = "",
				search = "",
				timezone = 0,
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				sortBy,
				sortOrder,
				username,
				packageId,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone: parseInt(timezone),
			};

			const result = await purchaseService.getSuccesses(options);
			res.json(
				new ApiResponse(
					200,
					result,
					"Purchase successes retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get filtered packages by type (locked/unlocked)
	async getFilteredPackages(req, res, next) {
		try {
			const {
				username,
				type: packageType,
				name,
				age,
				page = 1,
				limit = 10,
				search = "",
				sortBy = "purchasedAt",
				sortOrder = "desc",
			} = req.query;

			if (!packageType || !["locked", "unlocked"].includes(packageType)) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Invalid package type. Must be 'locked' or 'unlocked'",
						),
					);
			}

			const options = {
				username,
				packageType,
				name,
				age: age ? parseInt(age) : undefined,
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				sortBy,
				sortOrder,
			};

			const result = await purchaseService.getFilteredPackages(options);
			res.json(
				new ApiResponse(
					200,
					result,
					"Filtered packages retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Purchase package
	async purchasePackage(req, res, next) {
		const startTime = Date.now();
		try {
			const result = await purchaseService.processPurchase({
				body: {
					...req.body,
					userAgent: req.headers["user-agent"] || "unknown",
				},
				user: req.user,
				clientIP: req.clientIP,
				startTime,
			});

			res.status(201).json(new ApiResponse(201, result, result.message));
		} catch (err) {
			next(err);
		}
	},

	// Purchase status by transaction hash
	async getPurchaseStatus(req, res, next) {
		try {
			const { transactionHash } = req.query;

			const result = await purchaseService.getPurchaseStatus(transactionHash);
			res.json(
				new ApiResponse(200, result, "Purchase status retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { purchaseController };
