const {
	PurchaseAnalytics,
	PurchaseAnalyticsModel,
} = require("./purchaseAnalytics.model");
const { PurchaseFailure } = require("./purchaseFailure.model");
const { PurchaseLock } = require("./purchaseLock.model");
const { PurchaseSuccess } = require("./purchaseSuccess.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { processPurchase } = require("./services/purchaseProcessing.service");
const UserPackage = require("./userPackage.model");

// Convert local date to UTC with timezone
const parseLocalDateToUTC = (dateStr, offsetMinutes, endOfDay = false) => {
	const [year, month, day] = dateStr.split("-").map(Number);

	const utcDate = new Date(
		Date.UTC(
			year,
			month - 1,
			day,
			endOfDay ? 23 : 0,
			endOfDay ? 59 : 0,
			endOfDay ? 59 : 0,
			endOfDay ? 999 : 0,
		),
	);

	// Adjust for timezone
	const offsetMs = -offsetMinutes * 60 * 1000;
	return new Date(utcDate.getTime() - offsetMs);
};

const purchaseService = {
	// Stats from all purchase collections
	async getOverviewStats() {
		try {
			const [analyticsStats, successStats, failureStats, lockStats] =
				await Promise.all([
					// Total analytics attempts
					PurchaseAnalyticsModel.aggregate([
						{
							$group: {
								_id: null,
								totalAttempts: { $sum: 1 },
							},
						},
					]),

					// Success stats
					PurchaseSuccess.aggregate([
						{
							$group: {
								_id: null,
								totalSuccesses: { $sum: 1 },
								avgDuration: { $avg: "$duration" },
							},
						},
					]),

					// Failure stats
					PurchaseFailure.aggregate([
						{
							$group: {
								_id: null,
								totalFailures: { $sum: 1 },
							},
						},
					]),

					// Lock stats (active locks)
					PurchaseLock.aggregate([
						{
							$group: {
								_id: null,
								totalLocks: { $sum: 1 },
								activeLocks: {
									$sum: {
										$cond: [
											{
												$or: [
													{ $eq: ["$status", "processing"] },
													{ $eq: ["$status", "pending"] },
												],
											},
											1,
											0,
										],
									},
								},
							},
						},
					]),
				]);

			const totalAttempts = analyticsStats[0]?.totalAttempts || 0;
			const totalSuccesses = successStats[0]?.totalSuccesses || 0;
			const totalFailures = failureStats[0]?.totalFailures || 0;
			const activeLocks = lockStats[0]?.activeLocks || 0;
			const averageDuration = successStats[0]?.avgDuration || 0;

			const totalTransactions = totalAttempts || 1;
			const successRate =
				((totalSuccesses / totalTransactions) * 100).toFixed(1) + "%";
			const failureRate =
				((totalFailures / totalTransactions) * 100).toFixed(1) + "%";

			return {
				totalAttempts,
				totalSuccesses,
				totalFailures,
				activeLocks,
				successRate,
				failureRate,
				averageDuration: Math.round(averageDuration),
			};
		} catch (error) {
			logger.error("Error getting overview stats:", error);
			throw new ApiError(500, "Failed to retrieve overview statistics");
		}
	},

	// Analytics with search and filters
	async getAnalytics(options) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "timestamp",
				sortOrder = "desc",
				username,
				packageId,
				status,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone = 0,
			} = options;

			const skip = (page - 1) * limit;
			const query = {};

			// Username filter
			if (username && username.trim()) {
				query.username = { $regex: username.trim(), $options: "i" };
			}

			// Package ID filter
			if (packageId && packageId.trim()) {
				query.packageId = { $regex: packageId.trim(), $options: "i" };
			}

			// Status filter
			if (status && status !== "all") {
				query.status = status;
			}

			// Purchase ID filter
			if (purchaseId && purchaseId.trim()) {
				query.purchaseId = { $regex: purchaseId.trim(), $options: "i" };
			}

			// Date range filter with timezone
			if (startDate || endDate) {
				query.timestamp = {};
				if (startDate) {
					query.timestamp.$gte = parseLocalDateToUTC(
						startDate,
						timezone,
						false,
					);
				}
				if (endDate) {
					query.timestamp.$lte = parseLocalDateToUTC(endDate, timezone, true);
				}
			}

			// Search across multiple fields
			if (search && search.trim()) {
				const searchRegex = { $regex: search.trim(), $options: "i" };
				query.$or = [
					{ username: searchRegex },
					{ packageId: searchRegex },
					{ purchaseId: searchRegex },
					{ transactionHash: searchRegex },
					{ "metadata.traceId": searchRegex },
				];
			}

			const sortOptions = {};
			sortOptions[sortBy] = sortOrder === "desc" ? -1 : 1;

			const [analytics, total] = await Promise.all([
				PurchaseAnalyticsModel.find(query)
					.sort(sortOptions)
					.skip(skip)
					.limit(limit)
					.lean(),
				PurchaseAnalyticsModel.countDocuments(query),
			]);

			return {
				analytics,
				pagination: {
					total,
					page,
					limit,
					totalPages: Math.ceil(total / limit),
					hasNext: page < Math.ceil(total / limit),
					hasPrev: page > 1,
				},
				filters: {
					username,
					packageId,
					status,
					startDate,
					endDate,
					purchaseId,
					search,
				},
			};
		} catch (error) {
			logger.error("Error getting analytics:", error);
			throw new ApiError(500, "Failed to retrieve purchase analytics");
		}
	},

	// Failures with error breakdown
	async getFailures(options) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "timestamp",
				sortOrder = "desc",
				username,
				packageId,
				errorType,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone = 0,
			} = options;

			const skip = (page - 1) * limit;
			const query = {};

			if (username && username.trim()) {
				query.username = { $regex: username.trim(), $options: "i" };
			}

			if (packageId && packageId.trim()) {
				query.packageId = { $regex: packageId.trim(), $options: "i" };
			}

			if (errorType && errorType.trim()) {
				query.errorType = { $regex: errorType.trim(), $options: "i" };
			}

			if (purchaseId && purchaseId.trim()) {
				query.purchaseId = { $regex: purchaseId.trim(), $options: "i" };
			}

			if (startDate || endDate) {
				query.timestamp = {};
				if (startDate) {
					query.timestamp.$gte = parseLocalDateToUTC(
						startDate,
						timezone,
						false,
					);
				}
				if (endDate) {
					query.timestamp.$lte = parseLocalDateToUTC(endDate, timezone, true);
				}
			}

			if (search && search.trim()) {
				const searchRegex = { $regex: search.trim(), $options: "i" };
				query.$or = [
					{ username: searchRegex },
					{ packageId: searchRegex },
					{ purchaseId: searchRegex },
					{ errorType: searchRegex },
					{ errorDetails: searchRegex },
					{ transactionHash: searchRegex },
					{ "metadata.traceId": searchRegex },
				];
			}

			const sortOptions = {};
			sortOptions[sortBy] = sortOrder === "desc" ? -1 : 1;

			const [failures, total, summaryResult, errorBreakdown] =
				await Promise.all([
					PurchaseFailure.find(query)
						.sort(sortOptions)
						.skip(skip)
						.limit(limit)
						.lean(),
					PurchaseFailure.countDocuments(query),
					PurchaseFailure.aggregate([
						{ $match: query },
						{
							$group: {
								_id: null,
								totalFailures: { $sum: 1 },
								avgProcessingTime: { $avg: "$processingTime" },
							},
						},
					]),
					PurchaseFailure.aggregate([
						{ $match: query },
						{
							$group: {
								_id: "$errorType",
								count: { $sum: 1 },
							},
						},
						{ $sort: { count: -1 } },
					]),
				]);

			const summary = summaryResult[0] || {
				totalFailures: 0,
				avgProcessingTime: 0,
			};

			return {
				failures,
				summary: {
					...summary,
					errorBreakdown,
				},
				pagination: {
					total,
					page,
					limit,
					totalPages: Math.ceil(total / limit),
					hasNext: page < Math.ceil(total / limit),
					hasPrev: page > 1,
				},
				filters: {
					username,
					packageId,
					errorType,
					startDate,
					endDate,
					purchaseId,
					search,
				},
			};
		} catch (error) {
			logger.error("Error getting failures:", error);
			throw new ApiError(500, "Failed to retrieve purchase failures");
		}
	},

	// Locks with status breakdown
	async getLocks(options) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "createdAt",
				sortOrder = "desc",
				username,
				packageId,
				status,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone = 0,
			} = options;

			const skip = (page - 1) * limit;
			const query = {};

			if (username && username.trim()) {
				query.username = { $regex: username.trim(), $options: "i" };
			}

			if (packageId && packageId.trim()) {
				query.packageId = { $regex: packageId.trim(), $options: "i" };
			}

			if (status && status !== "all") {
				query.status = status;
			}

			if (purchaseId && purchaseId.trim()) {
				query.purchaseId = { $regex: purchaseId.trim(), $options: "i" };
			}

			if (startDate || endDate) {
				query.createdAt = {};
				if (startDate) {
					query.createdAt.$gte = parseLocalDateToUTC(
						startDate,
						timezone,
						false,
					);
				}
				if (endDate) {
					query.createdAt.$lte = parseLocalDateToUTC(endDate, timezone, true);
				}
			}

			if (search && search.trim()) {
				const searchRegex = { $regex: search.trim(), $options: "i" };
				query.$or = [
					{ username: searchRegex },
					{ packageId: searchRegex },
					{ purchaseId: searchRegex },
					{ transactionHash: searchRegex },
					{ status: searchRegex },
				];
			}

			const sortOptions = {};
			sortOptions[sortBy] = sortOrder === "desc" ? -1 : 1;

			const [locks, total, summaryResult, statusBreakdown] = await Promise.all([
				PurchaseLock.find(query)
					.sort(sortOptions)
					.skip(skip)
					.limit(limit)
					.lean(),
				PurchaseLock.countDocuments(query),
				PurchaseLock.aggregate([
					{ $match: query },
					{
						$group: {
							_id: null,
							totalLocks: { $sum: 1 },
							activeLocks: {
								$sum: {
									$cond: [
										{
											$or: [
												{ $eq: ["$status", "processing"] },
												{ $eq: ["$status", "pending"] },
											],
										},
										1,
										0,
									],
								},
							},
							completedLocks: {
								$sum: {
									$cond: [{ $eq: ["$status", "completed"] }, 1, 0],
								},
							},
							failedLocks: {
								$sum: {
									$cond: [{ $eq: ["$status", "failed"] }, 1, 0],
								},
							},
						},
					},
				]),
				PurchaseLock.aggregate([
					{ $match: query },
					{
						$group: {
							_id: "$status",
							count: { $sum: 1 },
						},
					},
					{ $sort: { count: -1 } },
				]),
			]);

			const summary = summaryResult[0] || {
				totalLocks: 0,
				activeLocks: 0,
				completedLocks: 0,
				failedLocks: 0,
			};

			return {
				locks,
				summary: {
					...summary,
					statusBreakdown,
				},
				pagination: {
					total,
					page,
					limit,
					totalPages: Math.ceil(total / limit),
					hasNext: page < Math.ceil(total / limit),
					hasPrev: page > 1,
				},
				filters: {
					username,
					packageId,
					status,
					startDate,
					endDate,
					purchaseId,
					search,
				},
			};
		} catch (error) {
			logger.error("Error getting locks:", error);
			throw new ApiError(500, "Failed to retrieve purchase locks");
		}
	},

	// Delete old and expired locks
	async clearExpiredLocks() {
		try {
			const oneHourAgo = new Date(Date.now() - 60 * 60 * 1000);

			const result = await PurchaseLock.deleteMany({
				$or: [
					{ expiresAt: { $lt: new Date() } },
					{ createdAt: { $lt: oneHourAgo }, status: "failed" },
					{ createdAt: { $lt: oneHourAgo }, status: "completed" },
				],
			});

			return {
				deletedCount: result.deletedCount,
			};
		} catch (error) {
			logger.error("Error clearing locks:", error);
			throw new ApiError(500, "Failed to clear expired locks");
		}
	},

	// Successful purchases with details
	async getSuccesses(options) {
		try {
			const {
				page = 1,
				limit = 20,
				sortBy = "timestamp",
				sortOrder = "desc",
				username,
				packageId,
				startDate,
				endDate,
				purchaseId,
				search,
				timezone = 0,
			} = options;

			const skip = (page - 1) * limit;
			const query = {};

			if (username && username.trim()) {
				query.username = { $regex: username.trim(), $options: "i" };
			}

			if (packageId && packageId.trim()) {
				query.packageId = { $regex: packageId.trim(), $options: "i" };
			}

			if (purchaseId && purchaseId.trim()) {
				query.purchaseId = { $regex: purchaseId.trim(), $options: "i" };
			}

			if (startDate || endDate) {
				query.timestamp = {};
				if (startDate) {
					query.timestamp.$gte = parseLocalDateToUTC(
						startDate,
						timezone,
						false,
					);
				}
				if (endDate) {
					query.timestamp.$lte = parseLocalDateToUTC(endDate, timezone, true);
				}
			}

			if (search && search.trim()) {
				const searchRegex = { $regex: search.trim(), $options: "i" };
				query.$or = [
					{ username: searchRegex },
					{ packageId: searchRegex },
					{ purchaseId: searchRegex },
					{ transactionHash: searchRegex },
					{ "metadata.traceId": searchRegex },
				];
			}

			const sortOptions = {};
			sortOptions[sortBy] = sortOrder === "desc" ? -1 : 1;

			const [successes, total, summaryResult] = await Promise.all([
				PurchaseSuccess.find(query)
					.sort(sortOptions)
					.skip(skip)
					.limit(limit)
					.lean(),
				PurchaseSuccess.countDocuments(query),
				PurchaseSuccess.aggregate([
					{ $match: query },
					{
						$group: {
							_id: null,
							totalSuccesses: { $sum: 1 },
							avgDuration: { $avg: "$duration" },
							totalPackagesCreated: { $sum: "$packagesCreated" },
							avgProcessingTime: { $avg: "$processingTime" },
						},
					},
				]),
			]);

			const summary = summaryResult[0] || {
				totalSuccesses: 0,
				avgDuration: 0,
				totalPackagesCreated: 0,
				avgProcessingTime: 0,
			};

			return {
				successes,
				summary,
				pagination: {
					total,
					page,
					limit,
					totalPages: Math.ceil(total / limit),
					hasNext: page < Math.ceil(total / limit),
					hasPrev: page > 1,
				},
				filters: {
					username,
					packageId,
					startDate,
					endDate,
					purchaseId,
					search,
				},
			};
		} catch (error) {
			logger.error("Error getting successes:", error);
			throw new ApiError(500, "Failed to retrieve purchase successes");
		}
	},

	// Process purchase (main purchase logic)
	processPurchase,

	// Get filtered packages by type (locked/unlocked)
	async getFilteredPackages(options) {
		try {
			const {
				username,
				packageType,
				name,
				age,
				page = 1,
				limit = 10,
				search = "",
				sortBy = "purchasedAt",
				sortOrder = "desc",
			} = options;

			const skip = (page - 1) * limit;
			const sortOrderValue = sortOrder === "desc" ? -1 : 1;

			// Build search filter
			const searchFilter = search
				? {
						$or: [
							{ transactionId: { $regex: search, $options: "i" } },
							{ status: { $regex: search, $options: "i" } },
						],
				  }
				: {};

			// Build sort object
			const sortObject = {};
			sortObject[sortBy || "purchasedAt"] = sortOrderValue;

			const pipeline = [
				{
					$match: {
						username,
						...searchFilter,
					},
				},
				{
					$lookup: {
						from: "products",
						localField: "packageId",
						foreignField: "_id",
						as: "packageDetails",
					},
				},
				{
					$match: {
						"packageDetails.packageType": packageType,
						...(name ? { "packageDetails.name": name } : {}),
						...(age
							? { "packageDetails.lockPeriod.value": { $gte: age } }
							: {}),
					},
				},
				{
					$addFields: {
						name: { $arrayElemAt: ["$packageDetails.name", 0] },
						packageType: { $arrayElemAt: ["$packageDetails.packageType", 0] },
					},
				},
				{
					$project: {
						packageDetails: 0,
					},
				},
				{
					$sort: sortObject,
				},
			];

			// Get total count for pagination
			const totalCount = await UserPackage.aggregate([
				...pipeline,
				{ $count: "total" },
			]);

			const total = totalCount[0]?.total || 0;
			const totalPages = Math.ceil(total / limit);

			// Add pagination to pipeline
			pipeline.push({ $skip: skip }, { $limit: limit });

			const packages = await UserPackage.aggregate(pipeline);

			return {
				success: true,
				packages,
				pagination: {
					total,
					totalPages,
					currentPage: page,
					limit,
					hasNext: page < totalPages,
					hasPrev: page > 1,
				},
			};
		} catch (error) {
			logger.error("Error getting filtered packages:", error);
			throw new ApiError(500, "Failed to retrieve filtered packages");
		}
	},

	async getPurchaseStatus(transactionHash) {
		try {
			// Check if packages exist for this transaction
			const userPackages = await UserPackage.find({
				$or: [
					{ blockchainTransactionHash: transactionHash },
					{ "blockchainTransaction.hash": transactionHash },
				],
			}).lean();
			if (userPackages.length > 0) {
				return {
					isComplete: true,
					packagesCreated: userPackages.length,
					packages: userPackages.map((pkg) => ({
						id: pkg._id.toString(),
						packageId: pkg.packageId.toString(),
						createdAt: pkg.createdAt,
						status: pkg.status || "active",
					})),
				};
			}
			// No packages found yet
			return {
				isComplete: false,
				packagesCreated: 0,
				message: "Purchase is still being processed",
			};
		} catch (error) {
			logger.error("Error getting purchase status:", error);
			throw new ApiError(500, "Failed to retrieve purchase status");
		}
	},
};

module.exports = { purchaseService };
