const mongoose = require("mongoose");
const { generateTraceId } = require("./services/transactionTracking.service");
const { PurchaseSuccess } = require("./purchaseSuccess.model");
const { PurchaseFailure } = require("./purchaseFailure.model");
const { logger } = require("../../core/logger/logger");

const purchaseAnalyticsSchema = new mongoose.Schema(
	{
		username: { type: String, required: true, trim: true, index: true },
		fromAddress: { type: String, required: true, trim: true },
		packageId: { type: String, required: true, trim: true, index: true },
		quantity: { type: Number, required: true },
		priceUSD: { type: mongoose.Schema.Types.Mixed, required: true },
		priceNTE: { type: mongoose.Schema.Types.Mixed, required: true },
		transactionHash: { type: String, required: true, trim: true },
		status: { type: String, required: true, trim: true },
		timestamp: { type: Date, required: true, index: true },
		validationErrors: { type: [String], default: [] },
		validationWarnings: { type: [String], default: [] },
		purchaseId: { type: String, trim: true },
		attemptType: { type: String, default: "initial_attempt" },
		metadata: {
			traceId: { type: String, trim: true },
		},
	},
	{ timestamps: false, collection: "purchase_analytics" },
);

purchaseAnalyticsSchema.index({ username: 1, timestamp: -1 });
purchaseAnalyticsSchema.index({ packageId: 1, timestamp: -1 });

const PurchaseAnalyticsModel = mongoose.model(
	"PurchaseAnalytics",
	purchaseAnalyticsSchema,
);

class PurchaseAnalytics {
	static async trackPurchaseAttempt(data) {
		try {
			const traceId = generateTraceId(
				"purchase_attempt",
				data.username || "ANONYMOUS",
			);

			await PurchaseAnalyticsModel.create({
				...data,
				timestamp: new Date(),
				attemptType: "initial_attempt",
				metadata: {
					...data.metadata,
					traceId,
				},
			});
		} catch (error) {
			if (logger && logger.error) {
				logger.error("Failed to track purchase attempt:", error);
			} else {
				console.error("Failed to track purchase attempt:", error);
			}
		}
	}

	static async trackPurchaseSuccess(data) {
		try {
			const traceId = generateTraceId(
				"purchase_success",
				data.username || "ANONYMOUS",
			);

			await PurchaseSuccess.create({
				...data,
				timestamp: new Date(),
				duration: data.duration || 0,
				metadata: {
					...data.metadata,
					traceId,
				},
			});
		} catch (error) {
			if (logger && logger.error) {
				logger.error("Failed to track purchase success:", error);
			} else {
				console.error("Failed to track purchase success:", error);
			}
		}
	}

	static async trackPurchaseFailure(data) {
		try {
			const traceId = generateTraceId(
				"purchase_failure",
				data.username || "ANONYMOUS",
			);

			await PurchaseFailure.create({
				...data,
				timestamp: new Date(),
				errorType: data.errorType || "unknown",
				metadata: {
					...data.metadata,
					traceId,
				},
			});
		} catch (error) {
			if (logger && logger.error) {
				logger.error("Failed to track purchase failure:", error);
			} else {
				console.error("Failed to track purchase failure:", error);
			}
		}
	}

	static async updateBonusResults(
		purchaseId,
		bonusSuccessCount,
		bonusFailureCount,
	) {
		try {
			await PurchaseSuccess.updateOne(
				{ purchaseId },
				{
					$set: {
						bonusSuccessCount,
						bonusFailureCount,
						bonusProcessingCompleted: true,
						bonusCompletedAt: new Date(),
					},
				},
			);
		} catch (error) {
			console.error("Failed to update bonus results:", error);
			throw error;
		}
	}
}

module.exports = { PurchaseAnalytics, PurchaseAnalyticsModel };
