const { logger } = require("../../../core/logger/logger");

const TOKEN_CONTRACT_ADDRESS = process.env.TOKEN_CONTRACT_ADDRESS;

// checkValidation: true = strict mode (1% tolerance), false = relaxed for admin overrides (5% tolerance)
async function verifyNTETransfer(
	transactionHash,
	fromAddress,
	toAddress,
	expectedAmount,
	checkValidation = true,
) {
	try {
		const rpcUrl = "https://bsc-dataseed1.binance.org/";

		const receiptResponse = await fetch(rpcUrl, {
			method: "POST",
			headers: {
				"Content-Type": "application/json",
			},
			body: JSON.stringify({
				jsonrpc: "2.0",
				method: "eth_getTransactionReceipt",
				params: [transactionHash],
				id: 1,
			}),
		});

		const receiptData = await receiptResponse.json();

		if (!receiptData.result) {
			return {
				isValid: false,
				error: "Transaction not found or not confirmed",
			};
		}

		const receipt = receiptData.result;

		if (receipt.status !== "0x1") {
			return {
				isValid: false,
				error: "Transaction failed on blockchain",
			};
		}

		const txResponse = await fetch(rpcUrl, {
			method: "POST",
			headers: {
				"Content-Type": "application/json",
			},
			body: JSON.stringify({
				jsonrpc: "2.0",
				method: "eth_getTransactionByHash",
				params: [transactionHash],
				id: 2,
			}),
		});

		const txData = await txResponse.json();

		if (!txData.result) {
			return {
				isValid: false,
				error: "Transaction details not found",
			};
		}

		const transaction = txData.result;

		// sender address check - admins can't bypass this either
		if (transaction.from.toLowerCase() !== fromAddress.toLowerCase()) {
			return {
				isValid: false,
				error: "Transaction sender mismatch",
				expected: fromAddress,
				actual: transaction.from,
			};
		}

		// parse recipient from transaction input data
		if (!transaction.input || transaction.input.length < 74) {
			return {
				isValid: false,
				error: "Transaction input data too short or missing",
				expected: "At least 74 characters (0x + 8 method + 64 address)",
				actual: transaction.input?.length || 0,
			};
		}
		
		const dataWithoutSelector = transaction.input.slice(10);
		const recipientHex = dataWithoutSelector.slice(0, 64);
		
		if (recipientHex.length !== 64) {
			return {
				isValid: false,
				error: "Invalid recipient hex length in transaction data",
				expected: 64,
				actual: recipientHex.length,
			};
		}
		
		const recipientAddress = "0x" + recipientHex.slice(24);

		if (recipientAddress.toLowerCase() !== toAddress.toLowerCase()) {
			return {
				isValid: false,
				error: "Transaction recipient is not the token contract",
				expected: toAddress,
				actual: recipientAddress,
			};
		}

		// find the Transfer event in transaction logs
		const transferEventTopic =
			"0xddf252ad1be2c89b69c2b068fc378daa952ba7f163c4a11628f55a4df523b3ef";

		const transferLog = receipt.logs.find(
			(log) =>
				log.topics[0] === transferEventTopic &&
				log.address.toLowerCase() === TOKEN_CONTRACT_ADDRESS.toLowerCase(),
		);

		if (!transferLog) {
			return {
				isValid: false,
				error: "Transfer event not found in transaction logs",
				debug: {
					totalLogs: receipt.logs.length,
					tokenContract: TOKEN_CONTRACT_ADDRESS,
					expectedTopic: transferEventTopic,
				},
			};
		}

		if (!transferLog.topics || transferLog.topics.length < 3) {
			return {
				isValid: false,
				error: "Invalid transfer event - missing required topics",
				debug: {
					actualTopics: transferLog.topics?.length || 0,
					requiredTopics: 3,
					transferLog,
				},
			};
		}
		
		if (!transferLog.topics[1] || transferLog.topics[1].length < 27) {
			return {
				isValid: false,
				error: "Invalid from address in transfer topics",
				debug: { topicLength: transferLog.topics[1]?.length || 0 },
			};
		}

		const logFromAddress = "0x" + transferLog.topics[1].slice(26);

		if (logFromAddress.toLowerCase() !== fromAddress.toLowerCase()) {
			return {
				isValid: false,
				error: "Transfer from address mismatch",
				expected: fromAddress,
				actual: logFromAddress,
				debug: {
					transferLog: transferLog.topics,
				},
			};
		}

		// calculate total transfer amount (including tax transfers)
		if (!transferLog.data) {
			return {
				isValid: false,
				error: "Transfer amount data missing from event log",
				debug: {
					transferLog,
				},
			};
		}

		const transferLogs = receipt.logs.filter(
			(log) => log.topics[0] === transferEventTopic,
		);

		let transferAmount = 0;

		for (const log of transferLogs) {
			if (!log.topics || log.topics.length < 3) {
				logger.warn('Transfer log missing required topics:', log);
				continue;
			}
			
			if (!log.topics[1] || log.topics[1].length < 27 || !log.topics[2] || log.topics[2].length < 27) {
				logger.warn('Transfer log topics have invalid length:', log);
				continue;
			}
			
			const logFrom = "0x" + log.topics[1].slice(26);
			const logTo = "0x" + log.topics[2].slice(26);

			if (
				logFrom.toLowerCase() === fromAddress.toLowerCase() &&
				(logTo.toLowerCase() === toAddress.toLowerCase() ||
					logTo.toLowerCase() ===
						process.env.TOKEN_TAX_ADDRESS.toLowerCase())
			) {
				try {
					if (!log.data || log.data === '0x') {
						logger.warn('Empty or invalid transfer log data:', log);
						continue;
					}
					transferAmount += Number(BigInt(log.data)) / 1e18;
				} catch (error) {
					logger.error('Failed to parse transfer amount from log:', error, log);
					return {
						isValid: false,
						error: 'Invalid transfer amount data in blockchain logs',
					};
				}
			}
		}

		// amount validation - always enforced but tolerance varies
		const amountDifference = Math.abs(transferAmount - expectedAmount);
		const tolerance = checkValidation
			? Math.max(0.01 * expectedAmount, 0.02) // strict: 1% or $0.02
			: Math.max(0.05 * expectedAmount, 0.05); // relaxed: 5% or $0.05

		logger.info(`Verifying tx ${transactionHash.slice(0, 10)}... - expected: ${expectedAmount}, actual: ${transferAmount}, diff: ${amountDifference.toFixed(4)}, tolerance: ${tolerance.toFixed(4)}, mode: ${checkValidation ? "strict" : "admin"}`);

		if (amountDifference > tolerance) {
			return {
				isValid: false,
				error: "Transfer amount mismatch",
				expected: expectedAmount,
				actual: transferAmount,
				difference: amountDifference,
				tolerance: tolerance,
				mode: checkValidation ? "strict_mode" : "admin_mode_relaxed",
				debug: {
					transferLogData: transferLog.data,
					calculatedAmount: transferAmount,
				},
			};
		}

		// get confirmations
		const blockResponse = await fetch(rpcUrl, {
			method: "POST",
			headers: {
				"Content-Type": "application/json",
			},
			body: JSON.stringify({
				jsonrpc: "2.0",
				method: "eth_blockNumber",
				params: [],
				id: 3,
			}),
		});

		const blockData = await blockResponse.json();
		const currentBlock = parseInt(blockData.result, 16);
		const txBlock = parseInt(receipt.blockNumber, 16);
		const confirmations = currentBlock - txBlock;

		return {
			isValid: true,
			transactionHash,
			from: fromAddress,
			to: toAddress,
			amount: transferAmount,
			blockNumber: txBlock,
			gasUsed: parseInt(receipt.gasUsed, 16).toString(),
			confirmations: confirmations,
		};
	} catch (error) {
		logger.error(`Blockchain verification failed for tx ${transactionHash?.slice(0, 10) || 'unknown'}:`, error.message);
		return {
			isValid: false,
			error: error.message || "Blockchain verification failed",
		};
	}
}

module.exports = {
	verifyNTETransfer,
};
