const { logger } = require("../../../core/logger/logger");
const { User } = require("../../users/user.model");
const UserPackage = require("../userPackage.model");
const { Product } = require("../../products/product.model");
const { DbeProgram } = require("../../dbe/dbeProgram.model");
const { DbeProgramAttendee } = require("../../dbe/dbeProgramAttendee.model");
const { recordReferralBonus } = require("./transactionTracking.service");
const { getUserUpline } = require("../../users/user.service");
const { convertNTEToUSD } = require("../../../utils/productCalculations");

function calculateImmediateReferralBonusNTE(productData, generation = 1) {
	const data = productData._doc || productData;

	if (!productData?.priceNTE || !data?.referralBonusImmediate) return 0;

	let percentage = 0;
	if (generation === 1) {
		percentage = data.referralBonusImmediate.firstGenPercent;
	} else if (generation === 2 && data.referralBonusImmediate.secondGenPercent) {
		percentage = data.referralBonusImmediate.secondGenPercent;
	} else if (generation === 3 && data.referralBonusImmediate.othersGenPercent) {
		percentage = data.referralBonusImmediate.othersGenPercent;
	}

	if (!percentage || percentage < 0 || percentage > 100) return 0;

	// Convert to wei (18 decimals) using BigInt
	const priceWei = BigInt(Math.round(productData.priceNTE * 1e18));
	const percentageBigInt = BigInt(Math.round(percentage * 100));

	// Calculate: (priceWei * percentage) / 10000
	const bonusWei = (priceWei * percentageBigInt) / BigInt(10000);

	// Convert back to NTE
	const bonusNTE = Number(bonusWei) / 1e18;

	if (!Number.isFinite(bonusNTE) || bonusNTE < 0) {
		logger.error(
			`Invalid bonus calculation: priceNTE=${productData.priceNTE}, percentage=${percentage}`,
		);
		return 0;
	}

	return bonusNTE;
}

async function processImmediateReferralBonuses(
	username,
	userPackageId,
	purchaseDistributionId = null,
	type = "user_package",
) {
	try {
		logger.info(
			`Processing immediate referrals for ${username}, package: ${userPackageId}`,
		);

		let distributionId = null;
		if (purchaseDistributionId) {
			distributionId =
				typeof purchaseDistributionId === "string"
					? purchaseDistributionId
					: purchaseDistributionId.toString();
		}

		let userPackage;
		let product;

		if (type === "dbe_package") {
			userPackage = await DbeProgramAttendee.findById(userPackageId);
			if (!userPackage) {
				logger.error(`DBE Attendee package not found: ${userPackageId}`);
				return { success: false, message: "DBE Attendee package not found" };
			}

			product = await DbeProgram.findById(userPackage.programId);
			if (!product) {
				logger.error(
					`DBE Program not found for attendee package: ${userPackage.programId}`,
				);
				return { success: false, message: "DBE Program not found" };
			}
		} else {
			userPackage = await UserPackage.findById(userPackageId);
			if (!userPackage) {
				logger.error(`User package not found: ${userPackageId}`);
				return { success: false, message: "User package not found" };
			}

			product = await Product.findById(userPackage.packageId);
			if (!product) {
				logger.error(`Product not found for package: ${userPackage.packageId}`);
				return { success: false, message: "Product not found" };
			}
		}

		if (!product.referralBonusImmediate?.firstGenPercent) {
			logger.info(
				`IMMEDIATE REFERRAL | Skipped for ${username} | No immediate referral bonuses configured for product ${product.name}`,
			);
			return {
				success: true,
				message: "No immediate referral bonuses configured",
			};
		}

		logger.info(
			`IMMEDIATE REFERRAL | Product ${product.name} settings | First gen: ${
				product.referralBonusImmediate.firstGenPercent || 0
			}% | Second gen: ${
				product.referralBonusImmediate.secondGenPercent || 0
			}% | Others gen: ${
				product.referralBonusImmediate.othersGenPercent || 0
			}%`,
		);

		// Get upline referrers
		const allUplineReferrers = await getUserUpline(username);

		logger.info(
			`IMMEDIATE REFERRAL | Found ${allUplineReferrers.length} upline referrers for ${username}`,
		);

		// Always max 3 level for immediate referral bonuses
		const maxGenerations = 3;
		const uplineReferrers = allUplineReferrers.slice(0, maxGenerations);

		if (uplineReferrers.length > 0) {
			uplineReferrers.slice(0, 3).forEach((ref, idx) => {
				logger.info(
					`${
						idx === 0 ? "1st" : idx === 1 ? "2nd" : "Others"
					} IMMEDIATE REFERRAL | Referrer ${idx + 1}: ${
						ref.username
					} | Level: ${ref.level}`,
				);
			});
		}

		if (uplineReferrers.length === 0) {
			logger.info(`No referrers found for ${username}`);
			return { success: true, message: "No referrers found" };
		}

		const results = [];

		// Process immediate referral bonuses for each generation
		for (
			let generation = 1;
			generation <= uplineReferrers.length;
			generation++
		) {
			const referrer = uplineReferrers[generation - 1];
			if (!referrer) {
				logger.info(
					`IMMEDIATE REFERRAL | Missing referrer for generation ${generation}`,
				);
				break;
			}

			// Get percentage for this generation
			let percentage = 0;
			if (generation === 1) {
				percentage = product.referralBonusImmediate.firstGenPercent || 0;
			} else if (generation === 2) {
				percentage = product.referralBonusImmediate.secondGenPercent || 0;
			} else {
				percentage = product.referralBonusImmediate.othersGenPercent || 0;
			}

			// Calculate immediate referral bonus using user package's actual purchase amount
			const bonusNTE = calculateImmediateReferralBonusNTE(
				{
					...product,
					priceNTE:
						type === "dbe_package" ? userPackage.feeNTE : userPackage.priceNTE,
				},
				generation,
			);

			logger.info(
				`${
					generation === 1 ? "1st" : generation === 2 ? "2nd" : "Others"
				} IMMEDIATE REFERRAL | Generation ${generation} | Referrer: ${
					referrer.username
				} | Percentage: ${percentage}% | Calculated bonus: ${bonusNTE} NTE (${await convertNTEToUSD(
					bonusNTE,
				)}) | Base value: ${
					type === "dbe_package" ? userPackage.feeNTE : userPackage.priceNTE
				} NTE`,
			);

			if (bonusNTE > 0) {
				try {
					// Check if referrer has at least one active "unlocked" package
					const referrerActiveUnlockedPackages = await UserPackage.aggregate([
						{
							$match: {
								username: referrer.username,
								status: "active",
							},
						},
						{
							$lookup: {
								from: "products",
								localField: "packageId",
								foreignField: "_id",
								as: "product",
							},
						},
						{
							$unwind: "$product",
						},
						{
							$match: {
								"product.packageType": "unlocked",
							},
						},
					]);

					if (referrerActiveUnlockedPackages.length === 0) {
						logger.info(
							`IMMEDIATE REFERRAL | Skipped referrer ${referrer.username} | Reason: No active unlocked packages`,
						);
						continue;
					}

					// Get referrer user details
					const referrerUser = await User.findOne({
						username: referrer.username,
					});
					if (!referrerUser) {
						logger.error(`Referrer user not found: ${referrer.username}`);
						continue;
					}

					// Record the referral bonus with package tracking
					const result = await recordReferralBonus({
						username: referrer.username,
						userId: referrerUser._id,
						purchaserUsername: username,
						generation: generation,
						percentage: percentage,
						baseValue:
							type === "dbe_package"
								? userPackage.feeNTE
								: userPackage.priceNTE,
						amountNTE: bonusNTE,
						amountUSD: await convertNTEToUSD(bonusNTE),
						packageDetails: {
							packageId:
								type === "dbe_package"
									? userPackage.programId.toString()
									: userPackage.packageId.toString(),
							name: product.name,
							packageType: product.packageType,
							userPackageId: userPackage._id.toString(),
						},
						walletBalanceBefore: referrerUser.walletBalance || 0,
						purchaseDistributionId: distributionId,
					});

					if (result.success) {
						results.push({
							referrerUsername: referrer.username,
							generation: generation,
							bonusNTE: bonusNTE,
							bonusUSD: await convertNTEToUSD(bonusNTE),
							percentage: percentage,
							packageName: product.name,
							userPackageId: userPackage._id.toString(),
							packageId:
								type === "dbe_package"
									? userPackage.programId.toString()
									: userPackage.packageId.toString(),
							distributionId: result.distributionId,
							bonusTransactionId: result.bonusTransactionId,
							walletTransactionId: result.walletTransactionId,
						});

						logger.info(
							`${
								generation === 1 ? "1st" : generation === 2 ? "2nd" : "Others"
							} IMMEDIATE REFERRAL | SUCCESS | Referrer: ${
								referrer.username
							} | Generation: ${generation} | Bonus: ${bonusNTE.toFixed(
								4,
							)} NTE | TxID: ${result.bonusTransactionId}`,
						);
					} else {
						logger.info(
							`${
								generation === 1 ? "1st" : generation === 2 ? "2nd" : "Others"
							} IMMEDIATE REFERRAL | FAILED | Referrer: ${
								referrer.username
							} | Generation: ${generation} | Error: ${
								result.error || "Unknown error"
							}`,
						);
						logger.error(
							`Failed to record referral bonus for ${referrer.username}:`,
							result.error,
						);
					}
				} catch (error) {
					logger.error(
						`Error processing immediate referral bonus for ${referrer.username}:`,
						error,
					);
				}
			}
		}

		return {
			success: true,
			data: {
				username,
				userPackageId: userPackage._id.toString(),
				packageId:
					type === "dbe_package"
						? userPackage.programId.toString()
						: userPackage.packageId.toString(),
				packageName: product.name,
				packageType: product.packageType,
				totalBonusesDistributed: results.length,
				referralBonuses: results,
				purchaseDistributionId: distributionId,
			},
			message: `Immediate referral bonuses processed for ${results.length} referrers`,
		};
	} catch (error) {
		logger.error(
			"Error processing immediate referral bonuses:",
			error.message,
			error.stack,
		);
		return { success: false, error: error.message };
	}
}

module.exports = {
	processImmediateReferralBonuses,
	calculateImmediateReferralBonusNTE,
};
