const { DbeProgramAttendee } = require('../../dbe/dbeProgramAttendee.model');
const { DbeProgram } = require('../../dbe/dbeProgram.model');
const { logger } = require('../../../core/logger/logger');
const { generateTraceId } = require('./userPackage.service');
const { getTokenValue } = require('../../config/globalConfig.service');

async function joinDBEProgram({
	programId,
	username,
	feeUSD,
	feeNTE,
	transactionId,
	blockchainTransactionHash,
	fromAddress,
	toAddress,
}) {
	try {
		if (!programId || typeof programId !== 'string') {
			throw new Error('Invalid programId');
		}
		
		if (!username || typeof username !== 'string') {
			throw new Error('Invalid username');
		}
		
		const parsedFeeUSD = parseFloat(feeUSD);
		const parsedFeeNTE = parseFloat(feeNTE);
		
		if (isNaN(parsedFeeUSD) || parsedFeeUSD < 0) {
			throw new Error('Invalid feeUSD value');
		}
		
		if (isNaN(parsedFeeNTE) || parsedFeeNTE < 0) {
			throw new Error('Invalid feeNTE value');
		}
		
		const program = await DbeProgram.findById(programId).lean();
		if (!program) {
			throw new Error('DBE program not found');
		}

		if (!program.isActive) {
			throw new Error('This DBE program is not active');
		}

		const startDate = new Date();
		const endDate = new Date(startDate);
		endDate.setDate(endDate.getDate() + program.durationDays);

		const traceId = generateTraceId('dbe_join', username);

		const tokenPriceAtJoining = await getTokenValue();

		const attendance = {
			programId,
			username: username,
			feeUSD: parsedFeeUSD,
			feeNTE: parsedFeeNTE,
			tokenPriceAtJoining,
			startDate,
			endDate,
			status: 'active',
			purchasedAt: startDate,
			createdAt: startDate,
			updatedAt: startDate,
			blockchainTransactionHash: blockchainTransactionHash,
			blockchainTransaction: blockchainTransactionHash
				? {
						hash: blockchainTransactionHash,
						fromAddress: fromAddress,
						toAddress: toAddress,
						network: 'BSC',
						tokenContract: process.env.TOKEN_CONTRACT_ADDRESS,
						verifiedAt: new Date(),
						traceId: traceId,
				  }
				: null,
			metadata: {
				traceId: traceId,
				creationMethod: 'dbe_join',
				transactionId: transactionId,
			},
		};

	const result = await DbeProgramAttendee.create(attendance);

	if (!result) {
		throw new Error('Failed to create DBE attendance record');
	}

	logger.info(
		`DBE join successful: ${username} -> ${program.name} (${result._id})`,
	);

	return {
		...attendance,
		_id: result._id.toString(),
		programId: result.programId.toString(),
	};
	} catch (error) {
		logger.error(`DBE join failed for ${username}: ${error.message}`);
		throw error;
	}
}

module.exports = {
  joinDBEProgram,
};
