const { Product } = require("../../products/product.model");
const UserPackage = require("../userPackage.model");
const { logger } = require("../../../core/logger/logger");
const mongoose = require("mongoose");
const { pricingCache } = require("./dynamicPricingCache.service");

async function initializePricingTracking(product) {
	if (
		!product.dynamicPricing ||
		!product.dynamicPricing.priceHistory ||
		product.dynamicPricing.priceHistory.length === 0
	) {
		const basePriceUSD = product.price?.startUSD || 0;

		const updateData = {
			"dynamicPricing.priceHistory": [
				{
					price: basePriceUSD,
					soldCount: 0,
					timestamp: new Date(),
					reason: "base_price_initialization",
					isBasePriceEntry: true,
				},
			],
			"dynamicPricing.basePriceUSD": basePriceUSD,
			"dynamicPricing.lastPriceUpdate": new Date(),
			"dynamicPricing.lastChecked": new Date(),
		};

		await Product.updateOne({ _id: product._id }, { $set: updateData });

		logger.info(
			`Price history initialized for ${product.name} at $${basePriceUSD}`,
		);

		return {
			...product.toObject(),
			dynamicPricing: {
				...product.dynamicPricing,
				priceHistory: updateData["dynamicPricing.priceHistory"],
				basePriceUSD: updateData["dynamicPricing.basePriceUSD"],
				lastPriceUpdate: updateData["dynamicPricing.lastPriceUpdate"],
				lastChecked: updateData["dynamicPricing.lastChecked"],
			},
		};
	}

	return product;
}

function calculateDynamicPrice(product, currentSoldCount) {
	const priceConfig = product.price;
	const dynamicPricing = product.dynamicPricing;

	if (
		!priceConfig?.startUSD ||
		priceConfig?.startUSD <= 0 ||
		!priceConfig?.countForPriceIncrease ||
		!priceConfig?.increaseSalePriceUSD
	) {
		return {
			shouldUpdate: false,
			reason: "Product does not qualify for dynamic pricing",
		};
	}

	if (priceConfig.fixedUSD && priceConfig.fixedUSD > 0) {
		return {
			shouldUpdate: false,
			reason: "Product uses fixed pricing",
		};
	}

	const basePrice =
		dynamicPricing.priceHistory[0]?.price || priceConfig.startUSD;
	const currentPrice = priceConfig.startUSD;

	const expectedIncreases = Math.floor(
		currentSoldCount / priceConfig.countForPriceIncrease,
	);
	const currentIncreases = dynamicPricing.priceIncreases || 0;

	if (expectedIncreases <= currentIncreases) {
		return {
			shouldUpdate: false,
			reason: "Price is already up to date",
			currentPrice: currentPrice,
			expectedIncreases,
			currentIncreases,
			basePrice,
		};
	}

	const totalIncreases = expectedIncreases;
	const newPrice =
		basePrice + totalIncreases * priceConfig.increaseSalePriceUSD;
	const priceIncreasesNeeded = expectedIncreases - currentIncreases;

	if (newPrice === currentPrice) {
		return {
			shouldUpdate: false,
			reason: "Calculated price matches current price - no update needed",
			currentPrice: currentPrice,
			expectedIncreases,
			currentIncreases,
			basePrice,
		};
	}

	return {
		shouldUpdate: true,
		oldPrice: currentPrice,
		newPrice: newPrice,
		basePrice: basePrice,
		priceIncreasesNeeded,
		totalIncreases,
		expectedIncreases,
		currentIncreases,
		priceIncreaseAmount: priceConfig.increaseSalePriceUSD,
		soldCount: currentSoldCount,
		reason: `Price increase from first history base $${basePrice} + ${totalIncreases} increases ($${priceConfig.increaseSalePriceUSD} each) due to ${currentSoldCount} sales`,
	};
}

// Write new price to DB
async function updateProductPrice(productId, priceCalculation, soldCount) {
	const updateData = {
		"price.startUSD": priceCalculation.newPrice,
		"dynamicPricing.lastPriceUpdate": new Date(),
		"dynamicPricing.lastChecked": new Date(),
	};

	let priceHistoryEntry = null;
	let pushOperation = {};

	if (priceCalculation.newPrice !== priceCalculation.oldPrice) {
		priceHistoryEntry = {
			price: priceCalculation.newPrice,
			soldCount: soldCount,
			timestamp: new Date(),
			reason: priceCalculation.reason,
			priceIncrease: priceCalculation.priceIncreaseAmount,
			increasesApplied: priceCalculation.priceIncreasesNeeded,
		};

		pushOperation = {
			$push: {
				"dynamicPricing.priceHistory": {
					$each: [priceHistoryEntry],
					$slice: -50, // Keep last 50 price history entries
				},
			},
		};
	}

	const result = await Product.updateOne(
		{ _id: new mongoose.Types.ObjectId(productId) },
		{
			$set: updateData,
			...pushOperation,
		},
	);

	return {
		success: result.modifiedCount > 0,
		modifiedCount: result.modifiedCount,
		priceHistoryEntry,
	};
}

async function processDynamicPricingForProduct(
	productId,
	triggeredBy = "cron",
	purchaseId = null,
) {
	try {
		logger.info(`Processing pricing for ${productId} (${triggeredBy})`);

		const cachedData = await pricingCache.get(productId);
		if (cachedData && triggeredBy === "purchase_completed") {
			try {
				const lastCheckedTime =
					typeof cachedData.lastChecked === "number"
						? cachedData.lastChecked
						: new Date(cachedData.lastChecked).getTime();

				if (!Number.isFinite(lastCheckedTime) || lastCheckedTime <= 0) {
					logger.warn(
						`Invalid cache timestamp for ${productId}, invalidating cache`,
					);
					await pricingCache.set(productId, null);
				} else if (Date.now() - lastCheckedTime < 30000) {
					return {
						success: true,
						updated: false,
						productId,
						productName: cachedData.productName,
						reason: "Recently processed via cache",
						currentPrice: cachedData.currentPrice,
						soldCount: cachedData.soldCount,
						triggeredBy,
						fromCache: true,
					};
				}
			} catch (dateError) {
				logger.error(
					`Cache data corrupted for ${productId}:`,
					dateError.message,
				);
				await pricingCache.set(productId, null);
			}
		}

		const lockId = purchaseId || `${triggeredBy}_${Date.now()}`;
		if (!pricingCache.acquireLock(productId, lockId)) {
			logger.info(`Product ${productId} locked, skipping`);
			return {
				success: true,
				updated: false,
				productId,
				reason: "Product locked by another process",
				triggeredBy,
				skipped: true,
			};
		}

		try {
			const product = await Product.findById(productId);
			if (!product) {
				logger.warn(`Product ${productId} not found`);
				return {
					success: false,
					error: "Product not found",
					productId,
					triggeredBy,
				};
			}

			let updatedProduct = await initializePricingTracking(product);

			const soldCount = await UserPackage.countDocuments({
				packageId: new mongoose.Types.ObjectId(productId),
			});

			logger.info(
				`${product.name}: sold ${soldCount}, price $${
					product.price?.startUSD || 0
				} (${triggeredBy})`,
			);

			const priceCalculation = calculateDynamicPrice(updatedProduct, soldCount);

			const cacheData = {
				productName: product.name,
				currentPrice: product.price?.startUSD || 0,
				soldCount,
				lastChecked: Date.now(),
				priceCalculation: {
					shouldUpdate: priceCalculation.shouldUpdate,
					reason: priceCalculation.reason,
				},
			};
			await pricingCache.set(productId, cacheData);

			if (!priceCalculation.shouldUpdate) {
				logger.info(
					`No price change for ${product.name}: ${priceCalculation.reason}`,
				);

				await Product.updateOne(
					{ _id: new mongoose.Types.ObjectId(productId) },
					{
						$set: {
							"dynamicPricing.lastChecked": new Date(),
						},
					},
				);

				return {
					success: true,
					updated: false,
					productId,
					productName: product.name,
					reason: priceCalculation.reason,
					currentPrice:
						priceCalculation.currentPrice || product.price?.startUSD || 0,
					soldCount,
					triggeredBy,
				};
			}

			logger.info(
				`Updating ${product.name}: $${priceCalculation.oldPrice} → $${priceCalculation.newPrice} (+${priceCalculation.priceIncreasesNeeded})`,
			);

			const updateResult = await updateProductPrice(
				productId,
				priceCalculation,
				soldCount,
			);

			if (updateResult.success) {
				logger.info(
					`Price updated: ${product.name} now $${priceCalculation.newPrice}, sold ${soldCount}`,
				);

				cacheData.currentPrice = priceCalculation.newPrice;
				cacheData.lastPriceUpdate = Date.now();
				await pricingCache.set(productId, cacheData);

				return {
					success: true,
					updated: true,
					productId,
					productName: product.name,
					oldPrice: priceCalculation.oldPrice,
					newPrice: priceCalculation.newPrice,
					priceIncreaseAmount: priceCalculation.priceIncreaseAmount,
					priceIncreasesApplied: priceCalculation.priceIncreasesNeeded,
					soldCount,
					reason: priceCalculation.reason,
					priceHistory: updateResult.priceHistoryEntry,
					triggeredBy,
				};
			} else {
				logger.error(`Failed to update price for ${product.name}`);
				return {
					success: false,
					error: "Failed to update product price",
					productId,
					productName: product.name,
					triggeredBy,
				};
			}
		} finally {
			pricingCache.releaseLock(productId, lockId);
		}
	} catch (error) {
		logger.error(`Error processing ${productId}:`, {
			message: error.message,
			stack: error.stack,
			triggeredBy,
			productId,
		});
		return {
			success: false,
			error: error.message,
			productId,
			triggeredBy,
		};
	}
}

module.exports = {
	processDynamicPricingForProduct,
};
