const { cacheService } = require("../../../core/cache/cache.service");
const { logger } = require("../../../core/logger/logger");

const CACHE_TTL = 30; // 30 seconds

class DynamicPricingCache {
	constructor() {
		this.locks = new Map();
	}

	async get(productId) {
		return await cacheService.get(`pricing:${productId}`);
	}

	async set(productId, data) {
		await cacheService.set(`pricing:${productId}`, data, CACHE_TTL);
	}

	isLocked(productId) {
		const lock = this.locks.get(productId);
		if (!lock) return false;

		// Lock too old? clear it (max 1 min)
		if (Date.now() - lock.timestamp > 60000) {
			this.locks.delete(productId);
			return false;
		}

		return true;
	}

	acquireLock(productId, lockId) {
		const existingLock = this.locks.get(productId);
		
		if (existingLock) {
			if (Date.now() - existingLock.timestamp > 60000) {
				this.locks.delete(productId);
			} else {
				return false;
			}
		}

		this.locks.set(productId, {
			lockId,
			timestamp: Date.now(),
		});
		return true;
	}

	releaseLock(productId, lockId) {
		const lock = this.locks.get(productId);
		if (lock && lock.lockId === lockId) {
			this.locks.delete(productId);
		}
	}

	cleanup() {
		const now = Date.now();
		const maxLocks = 10000; 

		// Dump old locks
		for (const [productId, lock] of this.locks.entries()) {
			if (now - lock.timestamp > 60000) {
				this.locks.delete(productId);
			}
		}

		if (this.locks.size > maxLocks) {
			const sortedLocks = Array.from(this.locks.entries())
				.sort((a, b) => a[1].timestamp - b[1].timestamp);
			
			const toDelete = sortedLocks.slice(0, this.locks.size - maxLocks);
			for (const [productId] of toDelete) {
				this.locks.delete(productId);
			}
			
			if (logger && logger.error) {
				logger.error(`Emergency lock cleanup: Removed ${toDelete.length} oldest locks (max: ${maxLocks})`);
			} else {
				console.error(`Emergency lock cleanup: Removed ${toDelete.length} oldest locks (max: ${maxLocks})`);
			}
		}
	}

	async getStats() {
		const cacheStats = cacheService.getStats();
		return {
			cacheSize: cacheStats.size,
			locksActive: this.locks.size,
		};
	}

	async clear() {
		this.locks.clear();
	}
}

const pricingCache = new DynamicPricingCache();

// Cleanup old locks every 5 min
const cleanupInterval = setInterval(() => {
	pricingCache.cleanup();
}, 300000);

if (cleanupInterval.unref) {
	cleanupInterval.unref();
}

process.on('SIGTERM', () => {
	clearInterval(cleanupInterval);
	pricingCache.clear();
});

process.on('SIGINT', () => {
	clearInterval(cleanupInterval);
	pricingCache.clear();
});

module.exports = { pricingCache };
