const { Product } = require("../../products/product.model");
const { logger } = require("../../../core/logger/logger");
const mongoose = require("mongoose");
const UserPackage = require("../userPackage.model");
const { cacheService } = require("../../../core/cache/cache.service");

const CACHE_TTL = 60 * 60; // 1 hour in seconds

async function getPackageOwners(packageId) {
	try {
		if (!packageId) {
			const products = await Product.find({});
			let totalCached = 0;

			for (const product of products) {
				const cacheKey = `package:owners:${product._id}`;
				const owners = await UserPackage.find({ packageId: product._id })
					.select("username transactionId -_id")
					.lean();

				await cacheService.set(cacheKey, owners, CACHE_TTL);
				totalCached++;
			}

			logger.info(`Package owners cache updated for ${totalCached} packages`);
			return { success: true, cachedPackages: totalCached };
		}

		// Check cache first
		const cacheKey = `package:owners:${packageId}`;
		const cached = await cacheService.get(cacheKey);

		if (cached) {
			return cached;
		}

		const owners = await UserPackage.find({
			packageId: new mongoose.Types.ObjectId(packageId),
		})
			.select("username transactionId -_id")
			.lean();

		await cacheService.set(cacheKey, owners, CACHE_TTL);

		return owners;
	} catch (error) {
		logger.error(`Error getting package owners:`, error);
		throw error;
	}
}

// Update cache for specific packages after purchase
async function updatePackageOwnersCache(packageId, purchaseId = null) {
	try {
		logger.info(`Updating cache for package ${packageId}`);
		const result = await getPackageOwners(packageId);
		logger.info(
			`Cache updated for ${packageId}, owners: ${result?.length || 0}`,
		);
		return { success: true, owners: result, count: result?.length || 0 };
	} catch (error) {
		logger.error(`Cache update failed for ${packageId}:`, error);
		return { success: false, error: error.message };
	}
}

module.exports = {
	getPackageOwners,
	updatePackageOwnersCache,
};
