﻿const { User } = require("../../users/user.model");
const { AdminConfig } = require("../../admin/adminConfig.model");
const { PurchaseLock } = require("../purchaseLock.model");
const { PurchaseAnalytics } = require("../purchaseAnalytics.model");
const { DbeProgramAttendee } = require("../../dbe/dbeProgramAttendee.model");
const { ApiError } = require("../../../utils/ApiError");
const bcrypt = require("bcryptjs");
const {
	validatePurchaseData,
	validateDBEJoinData,
	verifyPurchaseCompletion,
	verifyDBEJoinCompletion,
	logPurchaseAttempt,
} = require("./validation.service");
const {
	generateTransactionId,
	createUserPackage,
	getUserByUsername,
	getValidatedReserveWallet,
} = require("./userPackage.service");
const { joinDBEProgram } = require("./dbeAttendee.service");
const { recordPackagePurchase } = require("./transactionTracking.service");
const { notifyAdminsPurchase } = require("./notification.service");
const { logger } = require("../../../core/logger/logger");
const { enqueueJob } = require("../../../queues/job.queue");

async function processPurchase({ body, user, clientIP, startTime }) {
	let purchaseTimeout = null;
	const { type = "user_package" } = body;
	const username = body.username;
	const password = body.password;
	const masterPassword = body.masterPassword;
	const quantity =
		Math.max(1, Math.min(1000, parseInt(body.quantity) || 1)) || 1;
	const purchaseId = `${username}_${body.transactionHash}_${Date.now()}`;

	try {
		const dbUser = await User.findOne({ username: user.username }).select(
			"+password",
		);
		if (!dbUser) {
			logger.error(`User ${user.username} not found in database`);
			throw new ApiError(403, "Your account could not be verified");
		}

		const isAdmin = dbUser.role === "admin";

		if (type === "dbe_package") {
			const existingDBEPackage = await DbeProgramAttendee.findOne({
				username,
				status: { $in: ["active", "pending"] },
			});

			if (existingDBEPackage) {
				throw new ApiError(
					400,
					"You can only have one active DBE package at a time",
					{ error: "DUPLICATE_DBE_PACKAGE" },
				);
			}

			if (quantity > 1) {
				throw new ApiError(400, "Quantity for DBE Package can only be 1", {
					error: "INVALID_QUANTITY",
				});
			}
		}

		// Admins verify master password
		if (isAdmin) {
			if (
				!masterPassword ||
				typeof masterPassword !== "string" ||
				masterPassword.trim().length < 8
			) {
				throw new ApiError(400, "Admin master password is required");
			}

			const config = await AdminConfig.findOne({ type: "master_password" });
			if (!config || !config.passwordHash) {
				logger.error("Admin configuration not found in database");
				throw new ApiError(
					500,
					"Admin configuration not found. Please contact support.",
				);
			}

			const isMasterPasswordValid = await bcrypt.compare(
				masterPassword.trim(),
				config.passwordHash,
			);
			if (!isMasterPasswordValid) {
				logger.error(
					"Invalid admin master password attempt by:",
					user.username,
				);
				throw new ApiError(401, "Invalid admin master password");
			}

			logger.info("Admin master password verified for:", user.username);
		}

		// Need a password to proceed
		if (!password) {
			throw new ApiError(400, "Password is required to complete purchase");
		}

		logger.info("Purchase process started:", {
			purchaseId,
			username,
			requestedBy: user.username,
			quantity,
			isAdmin,
			type,
			checkValidation: body.checkValidation === false ? false : true,
		});

		// Regular users can only buy for themselves
		if (!isAdmin && user.username !== username) {
			logger.warn(
				`User ${user.username} attempted to purchase for different user ${username} - blocked`,
			);
			await User.updateOne(
				{ username: user.username },
				{
					$set: {
						isActive: false,
						isBlocked: true,
						blockedAt: new Date(),
						blockedBy: "system",
						blockReason: "Attempted unauthorized purchase!",
					},
				},
			);
			throw new ApiError(
				403,
				"Your account could not be verified, we blocked this purchase attempt and your account for your safety. Please contact support.",
			);
		}

		// Verify target user password
		const targetUser = await getUserByUsername(username);
		if (!targetUser) {
			throw new ApiError(404, `User ${username} does not exist`);
		}

		if (!targetUser.password || targetUser.password.length === 0) {
			throw new ApiError(
				403,
				"Please set up your password before making purchases.",
			);
		}

		// Now check if the password is correct
		let isPasswordValid;
		if (isAdmin) {
			// Admin uses their own password (not the target user's password)
			const adminUser = await getUserByUsername(user.username);
			if (!adminUser || !adminUser.password) {
				throw new ApiError(403, "Admin account requires a password to be set");
			}
			isPasswordValid = await bcrypt.compare(password, adminUser.password);
			logger.info(`Verifying admin password for: ${user.username}`);
		} else {
			// Regular user uses their own password
			isPasswordValid = await bcrypt.compare(password, targetUser.password);
			logger.info(`Verifying user password for: ${username}`);
		}

		if (!isPasswordValid) {
			logger.warn(
				`Invalid password attempt for ${
					isAdmin ? `admin ${user.username}` : `user ${username}`
				}`,
			);
			throw new ApiError(401, "The password you entered is incorrect");
		}

		logger.info(
			`Password verified for ${
				isAdmin ? `admin ${user.username}` : `user ${username}`
			}`,
		);

		const checkValidation =
			isAdmin && body.checkValidation === false ? false : true;

		// Prevent unreasonable quantity purchases
		const MAX_QUANTITY_PER_PURCHASE = 10;
		if (quantity > MAX_QUANTITY_PER_PURCHASE) {
			throw new ApiError(
				400,
				`Maximum ${MAX_QUANTITY_PER_PURCHASE} packages per transaction`,
				{
					requested: quantity,
					maximum: MAX_QUANTITY_PER_PURCHASE,
				},
			);
		}

		if (!isAdmin && body.checkValidation === false) {
			logger.warn(
				`Non-admin user ${username} attempted to bypass validation - blocked`,
			);
			await User.updateOne(
				{ username: user.username },
				{
					$set: {
						isActive: false,
						isBlocked: true,
						blockedAt: new Date(),
						blockedBy: "system",
						blockReason: "Attempted unauthorized purchase!",
					},
				},
			);
			throw new ApiError(403, "You are not authorized for this action");
		}

		const existingLock = await PurchaseLock.findOne({
			$or: [
				{ transactionHash: body.transactionHash, status: "processing" },
				{ transactionHash: body.transactionHash, status: "completed" },
				{ username, packageId: body.packageId, status: "processing" },
				{
					username,
					packageId: body.packageId,
					createdAt: { $gte: new Date(Date.now() - 60000) },
				},
			],
		});

		if (existingLock) {
			logger.warn(`Purchase lock found for user ${username}`);

			if (existingLock.status === "completed") {
				throw new ApiError(
					409,
					"This transaction has already been processed successfully",
					{
						purchaseId: existingLock.purchaseId,
						completedAt: existingLock.completedAt,
					},
				);
			} else if (existingLock.status === "processing") {
				throw new ApiError(
					409,
					"This purchase is already being processed. Please wait.",
				);
			} else {
				throw new ApiError(
					429,
					"A similar purchase was recently attempted. Please wait a moment before trying again.",
				);
			}
		}

		try {
			await PurchaseLock.create({
				purchaseId,
				username,
				packageId: body.packageId,
				transactionHash: body.transactionHash,
				quantity,
				status: "processing",
				expiresAt: new Date(Date.now() + 300000),
				packagesCreated: 0,
				clientIP,
				userAgent: body.userAgent || "unknown",
			});
		} catch (lockError) {
			logger.error(`Failed to acquire purchase lock: ${lockError.message}`);

			const conflictLock = await PurchaseLock.findOne({
				$or: [
					{ transactionHash: body.transactionHash },
					{ username, packageId: body.packageId, status: "processing" },
				],
			});

			if (conflictLock) {
				throw new ApiError(
					409,
					"Another purchase request is being processed. Please wait.",
					{
						purchaseId: conflictLock.purchaseId,
					},
				);
			} else {
				throw new ApiError(
					500,
					"Unable to process purchase request. Please try again.",
				);
			}
		}

		// Run validation checks
		let validationResult;
		if (type === "dbe_package") {
			validationResult = await validateDBEJoinData({
				username,
				programId: body.packageId,
				quantity,
				feeUSD: body.priceUSD,
				feeNTE: body.priceNTE,
				transactionHash: body.transactionHash,
				fromAddress: body.fromAddress,
				checkValidation,
			});
		} else {
			validationResult = await validatePurchaseData({
				username,
				packageId: body.packageId,
				quantity,
				priceUSD: body.priceUSD,
				priceNTE: body.priceNTE,
				transactionHash: body.transactionHash,
				fromAddress: body.fromAddress,
				checkValidation,
			});
		}

		await PurchaseAnalytics.trackPurchaseAttempt({
			purchaseId,
			username,
			packageId: body.packageId,
			quantity,
			priceUSD: body.priceUSD,
			priceNTE: body.priceNTE,
			transactionHash: body.transactionHash,
			fromAddress: body.fromAddress,
			status: validationResult.isValid ? "attempt" : "failed_validation",
			validationErrors: validationResult.errors,
			validationWarnings: validationResult.warnings,
		});

		if (!validationResult.isValid) {
			await PurchaseLock.deleteOne({ purchaseId });

			await PurchaseAnalytics.trackPurchaseFailure({
				purchaseId,
				username,
				packageId: body.packageId,
				quantity,
				priceUSD: body.priceUSD,
				priceNTE: body.priceNTE,
				transactionHash: body.transactionHash,
				fromAddress: body.fromAddress,
				errorType: "validation_failed",
				errorDetails: validationResult.errors,
			});

			logPurchaseAttempt(body, validationResult, "FAILED");

			throw new ApiError(400, "Purchase validation failed", {
				details: validationResult.errors,
				warnings: validationResult.warnings,
				summary: validationResult.summary,
			});
		}

		if (validationResult.warnings && validationResult.warnings.length > 0) {
			logger.warn("Purchase warnings:", validationResult.warnings);
		}

		const packageDetails = validationResult.packageDetails;
		const userDoc = validationResult.userDoc;
		const targetWalletAddress = getValidatedReserveWallet(packageDetails);

		// Figure out the per-unit prices
		const totalPriceUSD = parseFloat(body.priceUSD);
		const totalPriceNTE = parseFloat(body.priceNTE);
		const pricePerUnitUSD = totalPriceUSD / quantity;
		const pricePerUnitNTE = totalPriceNTE / quantity;

		logger.info(
			`Price calculation - Total USD: ${totalPriceUSD}, Total NTE: ${totalPriceNTE} for ${quantity} units (USD per unit: ${pricePerUnitUSD}, NTE per unit: ${pricePerUnitNTE})`,
		);

		const userPackages = [];
		const transactionIds = [];
		const distributionIds = [];
		const createdPackageIds = [];

		const logProgress = (stage, percentage, details) => {
			logger.info(
				`Purchase ${purchaseId} - ${stage} (${percentage}%): ${details}`,
			);
		};

		purchaseTimeout = setTimeout(async () => {
			logger.error(`Purchase timeout for ${purchaseId}`);
			try {
				await PurchaseLock.deleteOne({ purchaseId });
				logger.info(`Purchase lock removed for ${purchaseId} due to timeout`);
			} catch (err) {
				logger.error("Error removing purchase lock:", err);
			}
		}, 300000);

		try {
			logProgress(
				"Package Creation",
				55,
				`Creating ${quantity} packages for ${username}`,
			);
			logger.info(`Creating ${quantity} packages for ${username}`);

			// Process packages in small batches
			const batchSize = Math.min(5, quantity);
			const batches = Math.ceil(quantity / batchSize);

			for (let batchIndex = 0; batchIndex < batches; batchIndex++) {
				const startIndex = batchIndex * batchSize;
				const endIndex = Math.min(startIndex + batchSize, quantity);
				const currentBatchSize = endIndex - startIndex;

				// Calculate progress within package creation phase (55-75%)
				const batchProgress = 55 + (batchIndex / batches) * 20;
				logProgress(
					"Package Creation",
					Math.round(batchProgress),
					`Processing batch ${
						batchIndex + 1
					}/${batches} (${currentBatchSize} packages)`,
				);

				logger.info(
					`Processing batch ${
						batchIndex + 1
					}/${batches} (${currentBatchSize} packages)`,
				);

				for (let i = startIndex; i < endIndex; i++) {
					const packageIndex = i + 1;
					const transactionId = generateTransactionId();
					transactionIds.push(transactionId);

					const packageProgress = 55 + (packageIndex / quantity) * 20;
					if (quantity <= 10 || packageIndex % Math.ceil(quantity / 10) === 0) {
						logProgress(
							"Package Creation",
							Math.round(packageProgress),
							`Creating package ${packageIndex}/${quantity}`,
						);
					}

					logger.info(
						`Creating package ${packageIndex}/${quantity} with transaction ID: ${transactionId}`,
					);

					let userPackage;
					let packageRetries = 0;
					const maxPackageRetries = 3;

					while (packageRetries < maxPackageRetries) {
						try {
							if (type === "dbe_package") {
								userPackage = await joinDBEProgram({
									programId: body.packageId,
									username,
									feeUSD: pricePerUnitUSD,
									feeNTE: pricePerUnitNTE,
									transactionId,
									blockchainTransactionHash: body.transactionHash,
									fromAddress: body.fromAddress,
									toAddress: targetWalletAddress,
								});
							} else {
								userPackage = await createUserPackage({
									packageId: body.packageId,
									userId: userDoc._id,
									price: pricePerUnitUSD,
									priceNTE: pricePerUnitNTE,
									transactionId,
									username,
									blockchainTransactionHash: body.transactionHash,
									fromAddress: body.fromAddress,
									toAddress: targetWalletAddress,
								});
							}
							break;
						} catch (packageError) {
							packageRetries++;
							logger.warn(
								`Package creation failed for ${packageIndex}/${quantity}, retry ${packageRetries}/${maxPackageRetries}:`,
								packageError.message,
							);

							if (packageRetries >= maxPackageRetries) {
								throw new Error(
									`Failed to create package ${packageIndex} after ${maxPackageRetries} attempts: ${packageError.message}`,
								);
							}

							await new Promise((resolve) =>
								setTimeout(resolve, 1000 * Math.pow(2, packageRetries - 1)),
							);
						}
					}

					createdPackageIds.push(userPackage._id);
					userPackages.push(userPackage);

					let transactionResult;
					let transactionRetries = 0;
					const maxTransactionRetries = 3;

					while (transactionRetries < maxTransactionRetries) {
						try {
							transactionResult = await recordPackagePurchase({
								username,
								userId: userDoc._id,
								packageId: body.packageId,
								packageName: packageDetails.name,
								packageType: packageDetails.packageType || "user_package",
								priceNTE: pricePerUnitNTE,
								priceUSD: pricePerUnitUSD,
								transactionId,
								userPackageId: userPackage._id.toString(),
								walletBalanceBefore: userDoc.walletBalance || 0,
								blockchainTransactionHash: body.transactionHash,
								fromAddress: body.fromAddress,
								toAddress: targetWalletAddress,
							});

							if (!transactionResult.success) {
								throw new Error(
									transactionResult.error || "Transaction recording failed",
								);
							}
							break;
						} catch (transactionError) {
							transactionRetries++;
							logger.warn(
								`Transaction recording failed for package ${packageIndex}/${quantity}, retry ${transactionRetries}/${maxTransactionRetries}:`,
								transactionError.message,
							);

							if (transactionRetries >= maxTransactionRetries) {
								throw new Error(
									`Failed to record transaction for package ${packageIndex} after ${maxTransactionRetries} attempts: ${transactionError.message}`,
								);
							}

							// Wait before retrying
							await new Promise((resolve) =>
								setTimeout(resolve, 1000 * Math.pow(2, transactionRetries - 1)),
							);
						}
					}

					distributionIds.push(transactionResult.distributionId);

					// Log progress for larger orders
					if (quantity > 5 && packageIndex % 5 === 0) {
						logger.info(
							`Progress: ${packageIndex}/${quantity} packages created (${(
								(packageIndex / quantity) *
								100
							).toFixed(1)}%)`,
						);
					}
				}

				// Small pause between batches
				if (batchIndex < batches - 1) {
					await new Promise((resolve) => setTimeout(resolve, 50));
				}
			}

			logger.info(`All ${quantity} packages created successfully`);
			logProgress(
				"Package Creation Complete",
				80,
				`${quantity} packages created`,
			);

			if (purchaseTimeout) {
				clearTimeout(purchaseTimeout);
				purchaseTimeout = null;
			}

			logProgress("Bonus Processing", 85, "Processing referral bonuses");

			// Update the lock status
			await PurchaseLock.updateOne(
				{ purchaseId },
				{
					$set: {
						status: "completed",
						completedAt: new Date(),
						packagesCreated: quantity,
						transactionIds,
					},
				},
			);

			// Queue referral bonus processing as background job
			logger.info(
				`Queueing referral bonus processing for ${quantity} packages...`,
			);
			logProgress(
				"Bonus Processing",
				88,
				`Queueing bonuses for ${quantity} packages`,
			);

			try {
				enqueueJob(
					"process-referral-bonuses",
					{
						purchaseId,
						username,
						packageIds: userPackages.map((p) => p._id.toString()),
						distributionIds,
						type,
						quantity,
					},
					{
						attempts: 3,
						backoff: { type: "exponential", delay: 2000 },
					},
				);
				logger.info(`Referral bonus job queued for purchase ${purchaseId}`);
			} catch (queueError) {
				logger.error(
					`Failed to queue bonus processing for ${purchaseId}:`,
					queueError.message,
				);
			}

			await new Promise((resolve) => setTimeout(resolve, 200));

			// Verify everything completed properly
			logger.info(
				`Verifying purchase completion for ${transactionIds.length} transactions...`,
			);
			let completionVerification = { success: true, allRecordsCreated: true };

			try {
				let verificationPromise;
				if (type === "dbe_package") {
					verificationPromise = verifyDBEJoinCompletion(
						transactionIds,
						username,
						body.packageId,
					);
				} else {
					verificationPromise = verifyPurchaseCompletion(
						transactionIds,
						username,
						body.packageId,
					);
				}

				const verificationTimeout = setTimeout(() => {
					logger.warn(
						`Purchase verification timeout for ${purchaseId} - purchase may still be valid`,
					);
				}, 30000);

				completionVerification = await verificationPromise;
				clearTimeout(verificationTimeout);

				if (
					!completionVerification.success ||
					!completionVerification.allRecordsCreated
				) {
					logger.warn(
						`Purchase verification issues for ${purchaseId}:`,
						completionVerification,
					);
				} else {
					logger.info(
						`Purchase completion verified for ${purchaseId}:`,
						completionVerification.completeness,
					);
				}
			} catch (verificationError) {
				logger.error(
					`Error during purchase verification for ${purchaseId}:`,
					verificationError,
				);
			}

			const processingTime = Date.now() - startTime;

			// Log final success with details
			await PurchaseAnalytics.trackPurchaseSuccess({
				purchaseId,
				username,
				packageId: body.packageId,
				quantity,
				priceUSD: totalPriceUSD,
				priceNTE: totalPriceNTE,
				transactionHash: body.transactionHash,
				fromAddress: body.fromAddress,
				duration: processingTime,
				status: "success",
				packagesCreated: userPackages.length,
				bonusSuccessCount: 0,
				bonusFailureCount: 0,
				verification: completionVerification,
			});

			logger.info(
				`Purchase ${purchaseId} completed in ${processingTime}ms - ${quantity} packages created`,
			);

			// Send admin notifications
			notifyAdminsPurchase({
				purchaseId,
				buyerUsername: username,
				quantity,
				totalPrice: totalPriceUSD,
				totalPriceNTE: totalPriceNTE,
				transactionHash: body.transactionHash,
			})
				.then((result) => {
					logger.info(
						`ADMIN NOTIFICATIONS | Alert sent to admins about ${username}'s purchase:`,
						result.results,
					);
				})
				.catch((error) => {
					logger.error(
						`ADMIN NOTIFICATIONS | Failed to send admin alerts about ${username}'s purchase:`,
						error,
					);
				});

			// Queue background jobs for non-DBE packages
			if (type !== "dbe_package") {
				// Queue dynamic pricing update
				try {
					enqueueJob(
						"process-dynamic-pricing",
						{
							packageId: body.packageId,
							eventType: "purchase_completed",
							purchaseId,
						},
						{
							attempts: 2,
							backoff: { type: "exponential", delay: 1000 },
						},
					);
					logger.info(
						`Dynamic pricing job queued for package ${body.packageId}`,
					);
				} catch (queueError) {
					logger.error(
						`Failed to queue dynamic pricing for ${body.packageId}:`,
						queueError.message,
					);
				}

				// Queue package owners cache update
				try {
					enqueueJob(
						"update-package-owners-cache",
						{
							packageId: body.packageId,
							purchaseId,
						},
						{
							attempts: 2,
							backoff: { type: "exponential", delay: 1000 },
						},
					);
					logger.info(
						`Package owners cache job queued for package ${body.packageId}`,
					);
				} catch (queueError) {
					logger.error(
						`Failed to queue package owners cache update for ${body.packageId}:`,
						queueError.message,
					);
				}
			}

			return {
				success: true,
				purchaseId,
				message: `${quantity} package(s) purchased successfully`,
				transactionIds,
				quantity,
				packagesCreated: userPackages.length,
				totalPrice: totalPriceUSD,
				totalPriceNTE: totalPriceNTE,
				processingTime,
				blockchainTransactionHash: body.transactionHash,
				bonusProcessing: {
					status: "queued",
					message: "Referral bonuses are being processed in background",
					total: userPackages.length,
				},
				verification: {
					blockchain: validationResult.transferVerification,
					completion: completionVerification,
				},
			};
		} catch (error) {
			logger.error(
				`Error during package purchase transaction for ${purchaseId}:`,
				{
					error: error.message,
					stack: error.stack,
					transactionIds: transactionIds.length,
					createdPackages: createdPackageIds.length,
					user: username,
					packageId: body.packageId,
					quantity,
					blockchainTx: body.transactionHash,
					processingTime: Date.now() - startTime,
				},
			);

			// Update purchase lock status to failed
			try {
				await PurchaseLock.updateOne(
					{ purchaseId },
					{
						$set: {
							status: "failed",
							failedAt: new Date(),
							error: error.message,
							packagesCreated: createdPackageIds.length,
						},
					},
				);
			} catch (lockError) {
				logger.error(
					`Error updating purchase lock for ${purchaseId}:`,
					lockError,
				);
			}

			// Log failure analytics
			await PurchaseAnalytics.trackPurchaseFailure({
				purchaseId,
				username,
				packageId: body.packageId,
				quantity,
				priceUSD: body.priceUSD,
				priceNTE: body.priceNTE,
				transactionHash: body.transactionHash,
				fromAddress: body.fromAddress,
				errorType: error.name || "server_error",
				errorDetails: error.message,
				processingTime: Date.now() - startTime,
				packagesCreated: createdPackageIds.length,
				transactionIds: transactionIds,
			});

			throw error;
		} finally {
			// Cleanup timeout
			if (purchaseTimeout) {
				clearTimeout(purchaseTimeout);
			}
		}
	} catch (error) {
		logger.error("Purchase package API error:", {
			message: error.message,
			stack: error.stack,
			name: error.name,
			fullError: error,
		});

		if (error instanceof ApiError) {
			throw error;
		}

		let status = 500;
		let message = "Internal server error";

		if (error.message && error.message.includes("not found")) {
			status = 404;
			message = error.message;
		} else if (error.message && error.message.includes("validation")) {
			status = 400;
			message = error.message;
		} else if (error.message && error.message.includes("timeout")) {
			status = 408;
			message = "Purchase timeout - please contact support if payment was made";
		} else if (error.message && error.message.includes("duplicate")) {
			status = 409;
			message = error.message;
		}

		throw new ApiError(status, message, {
			details: error.message,
			errorName: error.name,
			originalStack: error.stack,
		});
	}
}

module.exports = {
	processPurchase,
};
