﻿const { User } = require("../../users/user.model");
const { AdminConfig } = require("../../admin/adminConfig.model");
const { PurchaseLock } = require("../purchaseLock.model");
const { PurchaseAnalytics } = require("../purchaseAnalytics.model");
const { DbeProgramAttendee } = require("../../dbe/dbeProgramAttendee.model");
const { ApiError } = require("../../../utils/ApiError");
const bcrypt = require("bcryptjs");
const mongoose = require("mongoose");
const {
	validatePurchaseData,
	validateDBEJoinData,
	verifyPurchaseCompletion,
	verifyDBEJoinCompletion,
	logPurchaseAttempt,
} = require("./validation.service");
const {
	generateTransactionId,
	createUserPackage,
	getUserByUsername,
	getValidatedReserveWallet,
} = require("./userPackage.service");
const { joinDBEProgram } = require("./dbeAttendee.service");
const { recordPackagePurchase } = require("./transactionTracking.service");
const { logger } = require("../../../core/logger/logger");
const { enqueueJob } = require("../../../queues/job.queue");
const systemNotificationService = require("../../../utils/system-notification.util");
const generateEmailFromTemplate = require("../../../templates/generateEmailFromTemplate");

function initializePurchaseContext({ body, user, clientIP, startTime }) {
	if (!body || typeof body !== "object") {
		throw new ApiError(400, "Invalid request body");
	}

	if (
		!body.username ||
		typeof body.username !== "string" ||
		body.username.trim().length === 0
	) {
		throw new ApiError(400, "Invalid username");
	}

	if (
		!body.transactionHash ||
		typeof body.transactionHash !== "string" ||
		body.transactionHash.length < 10
	) {
		throw new ApiError(400, "Invalid transaction hash");
	}

	if (
		!body.packageId ||
		typeof body.packageId !== "string" ||
		body.packageId.trim().length === 0
	) {
		throw new ApiError(400, "Invalid package ID");
	}

	const parsedQuantity = parseInt(body.quantity);
	if (isNaN(parsedQuantity) || parsedQuantity < 1 || parsedQuantity > 10) {
		throw new ApiError(
			400,
			`Invalid quantity: must be between 1 and 10. Received: ${body.quantity}`,
		);
	}
	const quantity = parsedQuantity;

	const parsedPriceUSD = parseFloat(body.priceUSD);
	const parsedPriceNTE = parseFloat(body.priceNTE);

	if (isNaN(parsedPriceUSD) || parsedPriceUSD <= 0) {
		throw new ApiError(400, "Invalid USD price");
	}

	if (isNaN(parsedPriceNTE) || parsedPriceNTE <= 0) {
		throw new ApiError(400, "Invalid NTE price");
	}

	const username = body.username.trim();
	const purchaseId = `${username}_${body.transactionHash}_${Date.now()}`;
	const type = body.type === "dbe_package" ? "dbe_package" : "user_package";

	return {
		body,
		user,
		clientIP,
		startTime,

		username,
		password: body.password,
		masterPassword: body.masterPassword,
		quantity,
		purchaseId,
		type,
		packageId: body.packageId,
		transactionHash: body.transactionHash,

		totalPriceUSD: parsedPriceUSD,
		totalPriceNTE: parsedPriceNTE,
		pricePerUnitUSD: parsedPriceUSD / quantity,
		pricePerUnitNTE: parsedPriceNTE / quantity,

		dbUser: null,
		isAdmin: false,
		targetUser: null,
		packageDetails: null,
		validationResult: null,
		checkValidation: true,
		targetWalletAddress: null,

		purchaseLock: null,
		purchaseTimeout: null,
		transactionIds: [],
		distributionIds: [],
		createdPackageIds: [],
		userPackages: [],
	};
}

async function executePreflightValidation(context) {
	const { user, username, password, masterPassword, type, quantity, body } =
		context;

	// 1. Verify requesting user exists
	const dbUser = await User.findOne({ username: user.username }).select(
		"+password",
	);
	if (!dbUser) {
		logger.error(`User ${user.username} not found in database`);
		throw new ApiError(403, "Your account could not be verified");
	}

	if (!dbUser.isActive || dbUser.isBlocked) {
		logger.warn(`Blocked/inactive user ${user.username} attempted purchase`);
		throw new ApiError(403, "Your account is not active");
	}

	context.dbUser = dbUser;
	context.isAdmin = dbUser.role === "admin";

	// 2. DBE package specific checks
	if (type === "dbe_package") {
		const existingDBEPackage = await DbeProgramAttendee.findOne({
			username,
			status: { $in: ["active", "pending"] },
		});

		if (existingDBEPackage) {
			throw new ApiError(
				400,
				"You can only have one active DBE package at a time",
				{ error: "DUPLICATE_DBE_PACKAGE" },
			);
		}

		if (quantity > 1) {
			throw new ApiError(400, "Quantity for DBE Package can only be 1", {
				error: "INVALID_QUANTITY",
			});
		}
	}

	// 3. Admin master password verification
	if (context.isAdmin) {
		if (
			!masterPassword ||
			typeof masterPassword !== "string" ||
			masterPassword.trim().length < 8
		) {
			throw new ApiError(400, "Admin master password is required");
		}

		const config = await AdminConfig.findOne({ type: "master_password" });
		if (!config || !config.passwordHash) {
			logger.error("Admin configuration not found in database");
			throw new ApiError(
				500,
				"Admin configuration not found. Please contact support.",
			);
		}

		const isMasterPasswordValid = await bcrypt.compare(
			masterPassword.trim(),
			config.passwordHash,
		);
		if (!isMasterPasswordValid) {
			logger.error("Invalid admin master password attempt by:", user.username);
			throw new ApiError(401, "Invalid admin master password");
		}

		logger.info("Admin master password verified for:", user.username);
	}

	// 4. Password required check
	if (!password) {
		throw new ApiError(400, "Password is required to complete purchase");
	}

	logger.info("Purchase process started:", {
		purchaseId: context.purchaseId,
		username,
		requestedBy: user.username,
		quantity,
		isAdmin: context.isAdmin,
		type,
		checkValidation: body.checkValidation === false ? false : true,
	});

	// 5. Verify regular users can only buy for themselves
	if (!context.isAdmin && user.username !== username) {
		logger.warn(
			`User ${user.username} attempted to purchase for different user ${username} - blocked`,
		);
		await User.updateOne(
			{ username: user.username },
			{
				$set: {
					isActive: false,
					isBlocked: true,
					blockedAt: new Date(),
					blockedBy: "system",
					blockReason: "Attempted unauthorized purchase!",
				},
			},
		);
		throw new ApiError(
			403,
			"Your account could not be verified, we blocked this purchase attempt and your account for your safety. Please contact support.",
		);
	}

	// 6. Verify target user exists and has password
	const targetUser = await getUserByUsername(username);
	if (!targetUser) {
		throw new ApiError(404, `User ${username} does not exist`);
	}

	if (!targetUser.password || targetUser.password.length === 0) {
		throw new ApiError(
			403,
			"Please set up your password before making purchases.",
		);
	}

	context.targetUser = targetUser;

	// 7. Verify password
	let isPasswordValid;
	if (context.isAdmin) {
		const adminUser = await getUserByUsername(user.username);
		if (!adminUser || !adminUser.password) {
			throw new ApiError(403, "Admin account requires a password to be set");
		}
		isPasswordValid = await bcrypt.compare(password, adminUser.password);
		logger.info(`Verifying admin password for: ${user.username}`);
	} else {
		isPasswordValid = await bcrypt.compare(password, targetUser.password);
		logger.info(`Verifying user password for: ${username}`);
	}

	if (!isPasswordValid) {
		logger.warn(
			`Invalid password attempt for ${
				context.isAdmin ? `admin ${user.username}` : `user ${username}`
			}`,
		);
		throw new ApiError(401, "The password you entered is incorrect");
	}

	logger.info(
		`Password verified for ${
			context.isAdmin ? `admin ${user.username}` : `user ${username}`
		}`,
	);

	// 8. Set checkValidation flag
	context.checkValidation =
		context.isAdmin && body.checkValidation === false ? false : true;

	// 9. Prevent unreasonable quantity
	const MAX_QUANTITY_PER_PURCHASE = 10;
	if (quantity > MAX_QUANTITY_PER_PURCHASE) {
		throw new ApiError(
			400,
			`Maximum ${MAX_QUANTITY_PER_PURCHASE} packages per transaction`,
			{
				requested: quantity,
				maximum: MAX_QUANTITY_PER_PURCHASE,
			},
		);
	}

	// 10. Prevent non-admin from bypassing validation
	if (!context.isAdmin && body.checkValidation === false) {
		logger.warn(
			`Non-admin user ${username} attempted to bypass validation - blocked`,
		);
		await User.updateOne(
			{ username: user.username },
			{
				$set: {
					isActive: false,
					isBlocked: true,
					blockedAt: new Date(),
					blockedBy: "system",
					blockReason: "Attempted unauthorized purchase!",
				},
			},
		);
		throw new ApiError(403, "You are not authorized for this action");
	}

	// 11. Check for existing locks
	const existingLock = await PurchaseLock.findOne({
		$or: [
			{ transactionHash: body.transactionHash, status: "processing" },
			{ transactionHash: body.transactionHash, status: "completed" },
			{ username, packageId: body.packageId, status: "processing" },
			{
				username,
				packageId: body.packageId,
				createdAt: { $gte: new Date(Date.now() - 60000) },
			},
		],
	});

	if (existingLock) {
		logger.warn(`Purchase lock found for user ${username}`);

		if (existingLock.status === "completed") {
			throw new ApiError(
				409,
				"This transaction has already been processed successfully",
				{
					purchaseId: existingLock.purchaseId,
					completedAt: existingLock.completedAt,
				},
			);
		} else if (existingLock.status === "processing") {
			throw new ApiError(
				409,
				"This purchase is already being processed. Please wait.",
			);
		} else {
			throw new ApiError(
				429,
				"A similar purchase was recently attempted. Please wait a moment before trying again.",
			);
		}
	}

	// 12. Create purchase lock with atomic operation
	try {
		const lockDoc = await PurchaseLock.create({
			purchaseId: context.purchaseId,
			username,
			packageId: body.packageId,
			transactionHash: body.transactionHash,
			quantity,
			status: "processing",
			expiresAt: new Date(Date.now() + 300000),
			packagesCreated: 0,
			clientIP: context.clientIP,
			userAgent: body.userAgent || "unknown",
		});

		if (!lockDoc || !lockDoc._id) {
			throw new Error("Lock creation returned invalid document");
		}

		context.purchaseLock = lockDoc;
	} catch (lockError) {
		logger.error(`Failed to acquire purchase lock: ${lockError.message}`);

		if (lockError.code === 11000) {
			throw new ApiError(409, "This transaction is already being processed");
		}

		const conflictLock = await PurchaseLock.findOne({
			$or: [
				{ transactionHash: body.transactionHash },
				{ username, packageId: body.packageId, status: "processing" },
			],
		}).lean();

		if (conflictLock) {
			throw new ApiError(409, "Another purchase request is being processed");
		}

		throw new ApiError(
			500,
			"Unable to process purchase request. Please try again.",
		);
	}

	// 13. Run package/program validation
	let validationResult;
	if (type === "dbe_package") {
		validationResult = await validateDBEJoinData({
			username,
			programId: body.packageId,
			quantity,
			feeUSD: body.priceUSD,
			feeNTE: body.priceNTE,
			transactionHash: body.transactionHash,
			fromAddress: body.fromAddress,
			checkValidation: context.checkValidation,
		});
	} else {
		validationResult = await validatePurchaseData({
			username,
			packageId: body.packageId,
			quantity,
			priceUSD: body.priceUSD,
			priceNTE: body.priceNTE,
			transactionHash: body.transactionHash,
			fromAddress: body.fromAddress,
			checkValidation: context.checkValidation,
		});
	}

	// Track purchase attempt
	await PurchaseAnalytics.trackPurchaseAttempt({
		purchaseId: context.purchaseId,
		username,
		packageId: body.packageId,
		quantity,
		priceUSD: body.priceUSD,
		priceNTE: body.priceNTE,
		transactionHash: body.transactionHash,
		fromAddress: body.fromAddress,
		status: validationResult.isValid ? "attempt" : "failed_validation",
		validationErrors: validationResult.errors,
		validationWarnings: validationResult.warnings,
	});

	if (!validationResult.isValid) {
		await PurchaseLock.deleteOne({ purchaseId: context.purchaseId });

		await PurchaseAnalytics.trackPurchaseFailure({
			purchaseId: context.purchaseId,
			username,
			packageId: body.packageId,
			quantity,
			priceUSD: body.priceUSD,
			priceNTE: body.priceNTE,
			transactionHash: body.transactionHash,
			fromAddress: body.fromAddress,
			errorType: "validation_failed",
			errorDetails: validationResult.errors,
		});

		logPurchaseAttempt(body, validationResult, "FAILED");

		throw new ApiError(400, "Purchase validation failed", {
			details: validationResult.errors,
			warnings: validationResult.warnings,
			summary: validationResult.summary,
		});
	}

	if (validationResult.warnings && validationResult.warnings.length > 0) {
		logger.warn("Purchase warnings:", validationResult.warnings);
	}

	context.validationResult = validationResult;
	context.packageDetails = validationResult.packageDetails;
	context.targetWalletAddress = getValidatedReserveWallet(
		validationResult.packageDetails,
	);

	logger.info(
		`Price calculation - Total USD: ${context.totalPriceUSD}, Total NTE: ${context.totalPriceNTE} for ${quantity} units (USD per unit: ${context.pricePerUnitUSD}, NTE per unit: ${context.pricePerUnitNTE})`,
	);

	logger.info(
		`Validation passed for purchase ${context.purchaseId} - User: ${context.username} - Package: ${context.packageId} - Type: ${context.type} - Quantity: ${context.quantity} - Admin: ${context.isAdmin} - CheckValidation: ${context.checkValidation}`,
	);
}

async function executeAtomicTransaction(context, session) {
	if (!session || !session.id) {
		throw new Error("Invalid database session");
	}

	if (
		session.transaction.state !== "STARTING_TRANSACTION" &&
		session.transaction.state !== "TRANSACTION_IN_PROGRESS"
	) {
		throw new Error("Transaction session is not in valid state");
	}

	const {
		purchaseId,
		username,
		packageId,
		transactionHash,
		quantity,
		clientIP,
		body,
		type,
		pricePerUnitUSD,
		pricePerUnitNTE,
		packageDetails,
		targetUser,
		targetWalletAddress,
	} = context;

	const logProgress = (stage, percentage, details) => {
		logger.info(
			`Purchase ${purchaseId} - ${stage} (${percentage}%): ${details}`,
		);
	};

	// Step 1: Create all packages and transactions
	logProgress(
		"Package Creation",
		55,
		`Creating ${quantity} packages for ${username}`,
	);
	logger.info(`Creating ${quantity} packages for ${username}`);

	const batchSize = Math.min(5, quantity);
	const batches = Math.ceil(quantity / batchSize);

	for (let batchIndex = 0; batchIndex < batches; batchIndex++) {
		const startIndex = batchIndex * batchSize;
		const endIndex = Math.min(startIndex + batchSize, quantity);
		const currentBatchSize = endIndex - startIndex;

		const batchProgress = 55 + (batchIndex / batches) * 20;
		logProgress(
			"Package Creation",
			Math.round(batchProgress),
			`Processing batch ${
				batchIndex + 1
			}/${batches} (${currentBatchSize} packages)`,
		);

		for (let i = startIndex; i < endIndex; i++) {
			const packageIndex = i + 1;
			const transactionId = generateTransactionId();
			context.transactionIds.push(transactionId);

			const packageProgress = 55 + (packageIndex / quantity) * 20;
			if (quantity <= 10 || packageIndex % Math.ceil(quantity / 10) === 0) {
				logProgress(
					"Package Creation",
					Math.round(packageProgress),
					`Creating package ${packageIndex}/${quantity}`,
				);
			}

			logger.info(
				`Creating package ${packageIndex}/${quantity} with transaction ID: ${transactionId}`,
			);

			// Create package with retry logic
			let userPackage;
			let packageRetries = 0;
			const maxPackageRetries = 3;

			while (packageRetries < maxPackageRetries) {
				try {
					if (type === "dbe_package") {
						userPackage = await joinDBEProgram({
							programId: packageId,
							username,
							feeUSD: pricePerUnitUSD,
							feeNTE: pricePerUnitNTE,
							transactionId,
							blockchainTransactionHash: transactionHash,
							fromAddress: body.fromAddress,
							toAddress: targetWalletAddress,
						});
					} else {
						userPackage = await createUserPackage({
							packageId,
							userId: targetUser._id,
							price: pricePerUnitUSD,
							priceNTE: pricePerUnitNTE,
							transactionId,
							username,
							blockchainTransactionHash: transactionHash,
							fromAddress: body.fromAddress,
							toAddress: targetWalletAddress,
						});
					}
					break;
				} catch (packageError) {
					packageRetries++;
					logger.warn(
						`Package creation failed for ${packageIndex}/${quantity}, retry ${packageRetries}/${maxPackageRetries}:`,
						packageError.message,
					);

					if (packageRetries >= maxPackageRetries) {
						throw new Error(
							`Failed to create package ${packageIndex} after ${maxPackageRetries} attempts: ${packageError.message}`,
						);
					}

					await new Promise((resolve) =>
						setTimeout(resolve, 1000 * Math.pow(2, packageRetries - 1)),
					);
				}
			}

			context.createdPackageIds.push(userPackage._id);
			context.userPackages.push(userPackage);

			// Record transaction
			let transactionResult;
			let transactionRetries = 0;
			const maxTransactionRetries = 3;

			while (transactionRetries < maxTransactionRetries) {
				try {
					transactionResult = await recordPackagePurchase({
						username,
						userId: targetUser._id,
						packageId,
						packageName: packageDetails.name,
						packageType: packageDetails.packageType || "user_package",
						priceNTE: pricePerUnitNTE,
						priceUSD: pricePerUnitUSD,
						transactionId,
						userPackageId: userPackage._id.toString(),
						walletBalanceBefore: targetUser.walletBalance || 0,
						blockchainTransactionHash: transactionHash,
						fromAddress: body.fromAddress,
						toAddress: targetWalletAddress,
					});

					if (!transactionResult || !transactionResult.success) {
						throw new Error(
							transactionResult?.error || "Transaction recording failed",
						);
					}

					if (!transactionResult.distributionId) {
						throw new Error("Transaction result missing distribution ID");
					}

					break;
				} catch (transactionError) {
					transactionRetries++;
					logger.warn(
						`Transaction recording failed for package ${packageIndex}/${quantity}, retry ${transactionRetries}/${maxTransactionRetries}:`,
						transactionError.message,
					);

					if (transactionRetries >= maxTransactionRetries) {
						throw new Error(
							`Failed to record transaction for package ${packageIndex} after ${maxTransactionRetries} attempts: ${transactionError.message}`,
						);
					}

					// Wait before retrying
					await new Promise((resolve) =>
						setTimeout(resolve, 1000 * Math.pow(2, transactionRetries - 1)),
					);
				}
			}

			context.distributionIds.push(transactionResult.distributionId);

			if (quantity > 5 && packageIndex % 5 === 0) {
				logger.info(
					`Progress: ${packageIndex}/${quantity} packages created (${(
						(packageIndex / quantity) *
						100
					).toFixed(1)}%)`,
				);
			}
		}

		if (batchIndex < batches - 1) {
			await new Promise((resolve) => setTimeout(resolve, 50));
		}
	}

	logger.info(`All ${quantity} packages created successfully`);
	logProgress("Package Creation Complete", 80, `${quantity} packages created`);

	logProgress("Transaction Complete", 85, "All database operations committed");

	return {
		packagesCreated: context.userPackages.length,
		transactionIds: context.transactionIds,
		distributionIds: context.distributionIds,
	};
}

async function executePostCommitOperations(context, transactionResult) {
	const {
		purchaseId,
		username,
		type,
		quantity,
		body,
		startTime,
		validationResult,
		userPackages,
		distributionIds,
	} = context;

	const logProgress = (stage, percentage, details) => {
		logger.info(
			`Purchase ${purchaseId} - ${stage} (${percentage}%): ${details}`,
		);
	};

	// 1. Queue referral bonus processing
	logProgress(
		"Bonus Processing",
		88,
		`Queueing bonuses for ${quantity} packages`,
	);

	try {
		enqueueJob(
			"process-referral-bonuses",
			{
				purchaseId,
				username,
				packageIds: userPackages.map((p) => p._id.toString()),
				distributionIds,
				type,
				quantity,
			},
			{
				attempts: 3,
				backoff: { type: "exponential", delay: 2000 },
			},
		);
		logger.info(`Referral bonus job queued for purchase ${purchaseId}`);
	} catch (queueError) {
		logger.error(
			`Failed to queue bonus processing for ${purchaseId}:`,
			queueError.message,
		);
	}

	await new Promise((resolve) => setTimeout(resolve, 200));

	// 2. Verify completion
	logger.info(
		`Verifying purchase completion for ${context.transactionIds.length} transactions...`,
	);
	let completionVerification = { success: true, allRecordsCreated: true };

	try {
		let verificationPromise;
		if (type === "dbe_package") {
			verificationPromise = verifyDBEJoinCompletion(
				context.transactionIds,
				username,
				body.packageId,
			);
		} else {
			verificationPromise = verifyPurchaseCompletion(
				context.transactionIds,
				username,
				body.packageId,
			);
		}

		const verificationTimeout = setTimeout(() => {
			logger.warn(
				`Purchase verification timeout for ${purchaseId} - purchase may still be valid`,
			);
		}, 30000);

		completionVerification = await verificationPromise;
		clearTimeout(verificationTimeout);

		if (
			!completionVerification.success ||
			!completionVerification.allRecordsCreated
		) {
			logger.warn(
				`Purchase verification issues for ${purchaseId}:`,
				completionVerification,
			);
		} else {
			logger.info(
				`Purchase completion verified for ${purchaseId}:`,
				completionVerification.completeness,
			);
		}
	} catch (verificationError) {
		logger.error(
			`Error during purchase verification for ${purchaseId}:`,
			verificationError,
		);
	}

	// 3. Track success analytics
	const processingTime = Date.now() - startTime;

	try {
		await PurchaseAnalytics.trackPurchaseSuccess({
			purchaseId,
			username,
			packageId: body.packageId,
			quantity,
			priceUSD: context.totalPriceUSD,
			priceNTE: context.totalPriceNTE,
			transactionHash: context.transactionHash,
			fromAddress: body.fromAddress,
			duration: processingTime,
			status: "success",
			packagesCreated: userPackages.length,
			bonusSuccessCount: 0,
			bonusFailureCount: 0,
			verification: completionVerification,
		});
	} catch (analyticsError) {
		logger.error(
			`Failed to track purchase success for ${purchaseId}:`,
			analyticsError.message,
		);
	}

	logger.info(
		`Purchase ${purchaseId} completed in ${processingTime}ms - ${quantity} packages created`,
	);

	// 4. Send notifications to user and admins
	try {
		const packageTypeName = type === "dbe_package" ? "DBE Program" : "Package";

		// Send success notification to user
		const userEmailBody = `
			<div style="color: #e1e4eb; line-height: 1.8;">
				<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">🎉 Purchase Successful!</h2>
				
				<p style="margin-bottom: 15px;">Hello <strong>${username}</strong>,</p>
				
				<p style="margin-bottom: 20px;">Congratulations! Your ${packageTypeName.toLowerCase()} purchase has been successfully completed and is now active in your account.</p>
				
				<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
					<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
					<table style="width: 100%; border-collapse: collapse;">
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">${packageTypeName}:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
								context.packageDetails?.name || "N/A"
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${quantity}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Total Amount:</td>
							<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600; font-size: 18px;">$${context.totalPriceUSD.toFixed(
								2,
							)}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">NTE Value:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">${context.totalPriceNTE.toFixed(
								6,
							)} NTE</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
							<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
								context.transactionHash
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Purchase ID:</td>
							<td style="padding: 8px 0; color: #9aa3b2; text-align: right; font-family: monospace; font-size: 10px;">${
								purchaseId.split("_")[0]
							}...${purchaseId.slice(-8)}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Processing Time:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">${(
								processingTime / 1000
							).toFixed(2)}s</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
							<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Completed</td>
						</tr>
					</table>
				</div>
				
				<div style="background: rgba(16, 185, 129, 0.1); border-left: 4px solid #10b981; padding: 15px; margin: 20px 0; border-radius: 8px;">
					<p style="margin: 0; color: #10b981;"><strong>✓ What's Next?</strong></p>
					<p style="margin: 10px 0 0 0; color: #e1e4eb;">${
						type === "dbe_package"
							? "Your DBE program is now active. Check your dashboard for program details and schedules."
							: "Your packages are now active in your account. Referral bonuses are being processed and will be distributed shortly."
					}</p>
				</div>
				
				<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
					<p style="margin: 0; color: #3b82f6;"><strong>🔗 Blockchain Verification</strong></p>
					<p style="margin: 10px 0 0 0; color: #e1e4eb;">Your purchase has been verified on the blockchain. You can view the transaction details on BSCScan using the link below.</p>
				</div>
				
				<div style="text-align: center; margin-top: 30px;">
					<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View ${
						type === "dbe_package" ? "Program" : "Packages"
					}</a>
					<a href="https://bscscan.com/tx/${
						context.transactionHash
					}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
				</div>
			</div>
		`;

		const userEmailHtml = generateEmailFromTemplate(userEmailBody);

		systemNotificationService.sendToUser({
			username,
			email: context.targetUser?.email,
			subject: `${packageTypeName} Purchase Successful - NodeMeta`,
			title: "Purchase Successful",
			body: `Your purchase of ${quantity} ${packageTypeName.toLowerCase()}${
				quantity > 1 ? "s" : ""
			} ($${context.totalPriceUSD.toFixed(
				2,
			)}) has been completed successfully.`,
			html: userEmailHtml,
			pushPayload: {
				icon: "/icons/icon-192x192.png",
				badge: "/icons/badge-72x72.png",
				data: {
					url: `/dashboard${type === "dbe_package" ? "/dbe" : "/packages"}`,
					purchaseId,
					transactionHash: context.transactionHash,
				},
			},
			sendEmail: true,
			sendPush: true,
		});
	} catch (notificationError) {
		logger.error("Failed to send purchase success notification to user:", {
			errorMessage: notificationError.message,
			errorStack: notificationError.stack,
			username,
			purchaseId,
		});
	}

	// Send notification to admins
	try {
		const packageTypeName = type === "dbe_package" ? "DBE Program" : "Package";

		const adminEmailBody = `
			<div style="color: #e1e4eb; line-height: 1.8;">
				<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">💰 New ${packageTypeName} Purchase</h2>
				
				<p style="margin-bottom: 20px;">A new ${packageTypeName.toLowerCase()} purchase has been completed successfully.</p>
				
				<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
					<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Purchase Details</h3>
					<table style="width: 100%; border-collapse: collapse;">
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Purchase ID:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px;">${purchaseId}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${username}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">${packageTypeName}:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
								context.packageDetails?.name || "N/A"
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Package Type:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">${
								context.packageDetails?.packageType || type
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Quantity:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${quantity}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Total USD:</td>
							<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600; font-size: 18px;">$${context.totalPriceUSD.toFixed(
								2,
							)}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Total NTE:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">${context.totalPriceNTE.toFixed(
								6,
							)} NTE</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Unit Price USD:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">$${context.pricePerUnitUSD.toFixed(
								2,
							)}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Unit Price NTE:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">${context.pricePerUnitNTE.toFixed(
								6,
							)} NTE</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Transaction Hash:</td>
							<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
								context.transactionHash
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">From Address:</td>
							<td style="padding: 8px 0; color: #9aa3b2; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
								context.body.fromAddress || "N/A"
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">To Address:</td>
							<td style="padding: 8px 0; color: #9aa3b2; text-align: right; font-family: monospace; font-size: 10px; word-break: break-all;">${
								context.targetWalletAddress || "N/A"
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Packages Created:</td>
							<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">${
								userPackages.length
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Processing Time:</td>
							<td style="padding: 8px 0; color: #fff; text-align: right;">${(
								processingTime / 1000
							).toFixed(2)}s</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">IP Address:</td>
							<td style="padding: 8px 0; color: #9aa3b2; text-align: right; font-family: monospace;">${
								context.clientIP || "N/A"
							}</td>
						</tr>
						<tr>
							<td style="padding: 8px 0; color: #9aa3b2;">Admin Bypass:</td>
							<td style="padding: 8px 0; color: ${
								context.isAdmin && !context.checkValidation
									? "#eab308"
									: "#9aa3b2"
							}; text-align: right; font-weight: ${
			context.isAdmin && !context.checkValidation ? "600" : "normal"
		};">${
			context.isAdmin && !context.checkValidation
				? "⚠️ Yes (Validation Skipped)"
				: "No"
		}</td>
						</tr>
					</table>
				</div>
				
				<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
					<p style="margin: 0; color: #3b82f6;"><strong>📊 Transaction IDs</strong></p>
					<p style="margin: 10px 0 0 0; color: #e1e4eb; font-family: monospace; font-size: 11px; word-break: break-all;">${context.transactionIds.join(
						", ",
					)}</p>
				</div>
				
				${
					type !== "dbe_package"
						? `
				<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
					<p style="margin: 0; color: #eab308;"><strong>💎 Bonus Processing</strong></p>
					<p style="margin: 10px 0 0 0; color: #e1e4eb;">Referral bonuses for ${
						userPackages.length
					} packages have been queued for background processing. Distribution IDs: ${distributionIds
								.slice(0, 3)
								.join(", ")}${
								distributionIds.length > 3
									? ` and ${distributionIds.length - 3} more...`
									: ""
						  }</p>
				</div>
				`
						: ""
				}
				
				<div style="text-align: center; margin-top: 30px;">
					<a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View All Purchases</a>
					<a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">View User</a>
					<a href="https://bscscan.com/tx/${
						context.transactionHash
					}" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">View on BSCScan</a>
				</div>
			</div>
		`;

		const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);

		systemNotificationService.sendToAdmin({
			subject: `New ${packageTypeName} Purchase: $${context.totalPriceUSD.toFixed(
				2,
			)} - ${username}`,
			title: "New Purchase",
			body: `${username} purchased ${quantity} ${packageTypeName.toLowerCase()}${
				quantity > 1 ? "s" : ""
			} for $${context.totalPriceUSD.toFixed(
				2,
			)} (${context.totalPriceNTE.toFixed(2)} NTE)`,
			html: adminEmailHtml,
			pushPayload: {
				icon: "/icons/icon-192x192.png",
				badge: "/icons/badge-72x72.png",
				data: {
					url: "/admin/purchases",
					purchaseId,
					username,
					transactionHash: context.transactionHash,
				},
			},
			sendEmail: true,
			sendPush: true,
		});
	} catch (notificationError) {
		logger.error("Failed to send purchase success notification to admins:", {
			errorMessage: notificationError.message,
			errorStack: notificationError.stack,
			purchaseId,
		});
	}

	// 5. Queue background jobs for non-DBE packages
	if (type !== "dbe_package") {
		try {
			enqueueJob(
				"process-dynamic-pricing",
				{
					packageId: body.packageId,
					eventType: "purchase_completed",
					purchaseId,
				},
				{
					attempts: 2,
					backoff: { type: "exponential", delay: 1000 },
				},
			);
			logger.info(`Dynamic pricing job queued for package ${body.packageId}`);
		} catch (queueError) {
			logger.error(
				`Failed to queue dynamic pricing for ${body.packageId}:`,
				queueError.message,
			);
		}

		try {
			enqueueJob(
				"update-package-owners-cache",
				{
					packageId: body.packageId,
					purchaseId,
				},
				{
					attempts: 2,
					backoff: { type: "exponential", delay: 1000 },
				},
			);
			logger.info(
				`Package owners cache job queued for package ${body.packageId}`,
			);
		} catch (queueError) {
			logger.error(
				`Failed to queue package owners cache update for ${body.packageId}:`,
				queueError.message,
			);
		}
	}

	return {
		processingTime,
		completionVerification,
	};
}

async function handlePurchaseFailure(context, error) {
	const {
		purchaseId,
		username,
		body,
		quantity,
		startTime,
		createdPackageIds,
		transactionIds,
	} = context;

	if (context.purchaseLock || purchaseId) {
		try {
			await PurchaseLock.updateOne(
				{ purchaseId },
				{
					$set: {
						status: "failed",
						failedAt: new Date(),
						error: error.message,
						packagesCreated: createdPackageIds.length,
					},
				},
			);
		} catch (lockError) {
			logger.error(
				`Error updating purchase lock for ${purchaseId}:`,
				lockError,
			);
		}
	}

	try {
		await PurchaseAnalytics.trackPurchaseFailure({
			purchaseId,
			username,
			packageId: body.packageId,
			quantity,
			priceUSD: body.priceUSD,
			priceNTE: body.priceNTE,
			transactionHash: body.transactionHash,
			fromAddress: body.fromAddress,
			errorType: error.name || "server_error",
			errorDetails: error.message,
			processingTime: Date.now() - startTime,
			packagesCreated: createdPackageIds.length,
			transactionIds: transactionIds,
		});
	} catch (analyticsError) {
		logger.error(
			`Failed to track purchase failure for ${purchaseId}:`,
			analyticsError.message,
		);
	}
}

async function cleanupPurchase(context) {
	if (context.purchaseTimeout) {
		clearTimeout(context.purchaseTimeout);
		context.purchaseTimeout = null;
	}
}

async function processPurchase({ body, user, clientIP, startTime }) {
	console.log("🐞 ~ purchaseProcessing.service.js:1253 ~ processPurchase ~ body:", body)
	const context = initializePurchaseContext({
		body,
		user,
		clientIP,
		startTime,
	});
	const session = await mongoose.startSession();

	context.purchaseTimeout = setTimeout(async () => {
		logger.error(`Purchase timeout for ${context.purchaseId}`);
		try {
			await PurchaseLock.deleteOne({ purchaseId: context.purchaseId });
			logger.info(
				`Purchase lock removed for ${context.purchaseId} due to timeout`,
			);
		} catch (err) {
			logger.error("Error removing purchase lock:", err);
		}
	}, 300000);

	try {
		await executePreflightValidation(context);

		let transactionResult;
		try {
			session.startTransaction();
			transactionResult = await executeAtomicTransaction(context, session);

			if (session.inTransaction()) {
				await session.commitTransaction();
				logger.info(
					`Transaction committed successfully for purchase ${context.purchaseId}`,
				);
			} else {
				logger.warn(
					`Transaction already ended before commit for ${context.purchaseId}`,
				);
			}
		} catch (txError) {
			if (session.inTransaction()) {
				await session.abortTransaction();
				logger.error(
					`Transaction aborted for purchase ${context.purchaseId}:`,
					txError.message,
				);
			} else {
				logger.warn(
					`Transaction already ended, cannot abort for ${context.purchaseId}`,
				);
			}
			throw txError;
		}

		if (context.purchaseTimeout) {
			clearTimeout(context.purchaseTimeout);
			context.purchaseTimeout = null;
		}

		const lockUpdateResult = await PurchaseLock.updateOne(
			{
				purchaseId: context.purchaseId,
				status: "processing",
				createdAt: { $exists: true },
			},
			{
				$set: {
					status: "completed",
					completedAt: new Date(),
					packagesCreated: context.quantity,
					transactionIds: context.transactionIds,
				},
			},
		);

		if (lockUpdateResult.matchedCount === 0) {
			logger.warn(
				`Purchase lock race detected for ${context.purchaseId}: ` +
					`Lock may have been taken over by timeout or already completed`,
			);
		}

		const postCommitResult = await executePostCommitOperations(
			context,
			transactionResult,
		);

		return {
			success: true,
			purchaseId: context.purchaseId,
			message: `${context.quantity} package(s) purchased successfully`,
			transactionIds: context.transactionIds,
			quantity: context.quantity,
			packagesCreated: context.userPackages.length,
			totalPrice: context.totalPriceUSD,
			totalPriceNTE: context.totalPriceNTE,
			processingTime: postCommitResult.processingTime,
			blockchainTransactionHash: context.transactionHash,
			bonusProcessing: {
				status: "queued",
				message: "Referral bonuses are being processed in background",
				total: context.userPackages.length,
			},
			verification: {
				blockchain: context.validationResult.transferVerification,
				completion: postCommitResult.completionVerification,
			},
		};
	} catch (error) {
		logger.error(`Error during purchase ${context.purchaseId}:`, {
			error: error.message,
			stack: error.stack,
			transactionIds: context.transactionIds.length,
			createdPackages: context.createdPackageIds.length,
			user: context.username,
			packageId: context.packageId,
			quantity: context.quantity,
			blockchainTx: context.transactionHash,
			processingTime: Date.now() - startTime,
		});

		if (session.inTransaction()) {
			await session.abortTransaction();
			logger.info(
				`Transaction auto-rolled back for purchase ${context.purchaseId}`,
			);
		}

		await handlePurchaseFailure(context, error);

		if (error instanceof ApiError) {
			throw error;
		}

		let status = 500;
		let message = "Internal server error";

		if (error.message && error.message.includes("not found")) {
			status = 404;
			message = error.message;
		} else if (error.message && error.message.includes("validation")) {
			status = 400;
			message = error.message;
		} else if (error.message && error.message.includes("timeout")) {
			status = 408;
			message = "Purchase timeout - please contact support if payment was made";
		} else if (error.message && error.message.includes("duplicate")) {
			status = 409;
			message = error.message;
		}

		throw new ApiError(status, message);
	} finally {
		if (session) {
			try {
				if (session.inTransaction()) {
					await session.abortTransaction();
				}
			} catch (abortError) {
				logger.error(
					"Error aborting transaction in finally:",
					abortError.message,
				);
			}

			try {
				await session.endSession();
			} catch (endError) {
				logger.error("Error ending session in finally:", endError.message);
			}
		}

		await cleanupPurchase(context);
	}
}

module.exports = {
	processPurchase,
};
