﻿const mongoose = require("mongoose");
const { logger } = require("../../../core/logger/logger");
const { generateTraceId } = require("./userPackage.service");
const { BonusTransaction } = require("../../bonuses/bonusTransaction.model");
const { WalletTransaction } = require("../../wallets/walletTransaction.model");
const { User } = require("../../users/user.model");

async function recordPackagePurchase(options) {
	const {
		username,
		userId,
		packageId,
		packageName,
		packageType,
		priceNTE,
		priceUSD,
		transactionId,
		userPackageId,
		walletBalanceBefore,
		blockchainTransactionHash,
		fromAddress,
		toAddress,
	} = options;

	const parsedBalance = parseFloat(walletBalanceBefore);
	if (!Number.isFinite(parsedBalance) || parsedBalance < 0) {
		logger.error(
			`Invalid wallet balance for ${username}: ${walletBalanceBefore}, defaulting to 0`,
		);
		walletBalanceBefore = 0;
	}

	const traceId = generateTraceId("package_purchase", username);

	return recordComprehensiveTransaction({
		username,
		userId,
		transactionType: "package_purchase",
		bonusType: "package_purchase",
		amountNTE: priceNTE,
		amountUSD: priceUSD,
		description: `Purchase of ${packageName} package via blockchain transfer`,
		metadata: {
			packageId,
			packageName,
			packageType,
			transactionId,
			userPackageId,
			triggerType: "user_action",
			blockchainTransaction: {
				hash: blockchainTransactionHash,
				fromAddress: fromAddress,
				toAddress: toAddress,
				tokenContract: process.env.TOKEN_CONTRACT_ADDRESS,
				network: "BSC",
				paymentMethod: "external_wallet",
			},
		},
		walletBalanceBefore: parsedBalance,
		isDeduction: true,
		skipWalletUpdate: true,
		traceId,
	});
}

async function recordReferralBonus(options) {
	const {
		username,
		userId,
		purchaserUsername,
		generation,
		percentage,
		baseValue,
		amountNTE,
		amountUSD,
		packageDetails,
		walletBalanceBefore,
		purchaseDistributionId,
		bonusType = "immediate_referral", // Allow custom bonus type, default to immediate_referral
		session,
	} = options;

	try {
		const traceId = generateTraceId(bonusType, username);

		const description =
			bonusType === "daily_referral"
				? `Daily referral bonus (${percentage}%) - Generation ${generation} - ${purchaserUsername}'s ${packageDetails.name} purchase`
				: `Immediate referral bonus (${percentage}%) - Generation ${generation} - ${purchaserUsername}'s ${packageDetails.name} purchase`;

		return recordComprehensiveTransaction({
			username,
			userId,
			transactionType: "bonus_payment",
			bonusType: bonusType,
			amountNTE,
			amountUSD,
			description,
			metadata: {
				traceId,
				packageId: packageDetails.packageId,
				packageName: packageDetails.name,
				packageType: packageDetails.packageType,
				purchaserUsername,
				generation,
				percentage,
				baseValue,
				triggerType: "user_action",
				purchaseDistributionId: purchaseDistributionId || null,
			},
			walletBalanceBefore,
			isDeduction: false,
			skipWalletUpdate: false,
			traceId,
			session,
		});
	} catch (error) {
		logger.error("Error recording referral bonus:", error);
		throw error;
	}
}

// Record transaction with all required records
async function recordComprehensiveTransaction(options) {
	try {
		const {
			username,
			userId,
			transactionType,
			bonusType,
			amountNTE,
			amountUSD,
			description,
			metadata = {},
			walletBalanceBefore,
			isDeduction = false,
			skipWalletUpdate = false, // Flag to skip wallet updates for external payments
			traceId,
		} = options;

		// Apply sign based on transaction type
		const signMultiplier = isDeduction ? -1 : 1;
		const finalAmountNTE = amountNTE * signMultiplier;
		const finalAmountUSD = amountUSD * signMultiplier;

		// Generate trace ID if not provided
		const finalTraceId =
			traceId || generateTraceId(bonusType || transactionType, username);

		// 1. Get or validate distribution ID - BonusDistribution is created ONLY by the main distribution function
		let distributionId;
		try {
			distributionId = metadata.distributionId
				? typeof metadata.distributionId === "string"
					? new mongoose.Types.ObjectId(metadata.distributionId)
					: metadata.distributionId
				: new mongoose.Types.ObjectId();
		} catch (error) {
			logger.error(
				`Invalid distributionId format: ${metadata.distributionId}`,
				error,
			);
			throw new Error(`Invalid distribution ID: ${error.message}`);
		}
		const distributionDate = new Date().toISOString().split("T")[0];

		// 2. Create bonus transaction record - tracks individual bonus payment
		const bonusTransactionId = new mongoose.Types.ObjectId();

		// TODO: Re-enable duplicate check if needed
		// Check for duplicate transaction using COMPOSITE KEY
		/* 	const duplicateCheck = {
			username,
			bonusType: bonusType || transactionType,
			createdAt: {
				$gte: new Date(distributionDate + "T00:00:00.000Z"),
				$lt: new Date(distributionDate + "T23:59:59.999Z"),
			},
			status: "completed",
		};

		// Add package-specific check if available
		if (metadata.userPackageId) {
			duplicateCheck["metadata.userPackageId"] = metadata.userPackageId;
		}

		const existingTransaction = await BonusTransaction.findOne(
			duplicateCheck,
		).lean();

		if (existingTransaction) {
			logger.warn(
				`Duplicate transaction detected for ${username} - ${
					bonusType || transactionType
				} on ${distributionDate}`,
				{
					existingId: existingTransaction._id,
					traceId: finalTraceId,
					packageId: metadata.userPackageId,
				},
			);
			return {
				success: true,
				distributionId: distributionId.toString(),
				bonusTransactionId: existingTransaction._id.toString(),
				walletTransactionId:
					existingTransaction.metadata?.walletTransactionId || "",
				walletBalanceAfter: walletBalanceBefore,
				isExternalPayment: skipWalletUpdate,
				traceId: finalTraceId,
				skipped: true,
				duplicateDetected: true,
			};
		} */

		// Skip creating bonus transaction record for manual balance adjustments and certain types
		if (
			bonusType !== "manual_balance_adjustment" &&
			bonusType !== "package_purchase" &&
			bonusType !== "fund_transfer" &&
			bonusType !== "refund" &&
			bonusType !== "withdrawal"
		) {
			await BonusTransaction.create([
				{
					_id: bonusTransactionId,
					transactionType,
					bonusType: bonusType || transactionType,
					amountNTE: finalAmountNTE,
					amountUSD: finalAmountUSD,
					description,
					username,
					createdAt: new Date(),
					status: "completed",
					generatedBy: "user_action",
					distributionId: distributionId,
					metadata: {
						distributionId: distributionId,
						paymentMethod: skipWalletUpdate ? "external" : "wallet",
						traceId: finalTraceId,
						...metadata,
					},
				},
			]);
		}

		// 3. Create wallet transaction record
		const walletTransactionId = new mongoose.Types.ObjectId();
		// For external payments, wallet balance remains the same
		const walletBalanceAfter = skipWalletUpdate
			? walletBalanceBefore
			: walletBalanceBefore + finalAmountNTE;

		await WalletTransaction.create([
			{
				_id: walletTransactionId,
				userId,
				username,
				type: transactionType,
				subType: bonusType || transactionType,
				amountNTE: finalAmountNTE,
				amountUSD: finalAmountUSD,
				description: skipWalletUpdate
					? `${description} (External Payment)`
					: description,
				metadata: {
					...metadata,
					distributionId: distributionId,
					bonusTransactionId: bonusTransactionId,
					paymentMethod: skipWalletUpdate ? "external" : "wallet",
					isExternalPayment: skipWalletUpdate,
					traceId: finalTraceId,
				},
				balanceBefore: walletBalanceBefore,
				balanceAfter: walletBalanceAfter,
				status: "completed",
				createdAt: new Date(),
				processedAt: new Date(),
				isExternalPayment: skipWalletUpdate,
			},
		]);

		// Update BonusTransaction to include walletTransactionId for tracking
		await BonusTransaction.updateOne(
			{ _id: bonusTransactionId },
			{
				$set: {
					"metadata.walletTransactionId": walletTransactionId.toString(),
				},
			},
		);

		// 4. Update user's wallet balance ONLY if not an external payment
		if (!skipWalletUpdate) {
			await User.updateOne(
				{ _id: userId },
				{ $inc: { walletBalance: finalAmountNTE } },
			);
		}

		logger.info(
			`Transaction recorded: ${finalTraceId} - Distribution: ${distributionId}`,
		);

		return {
			success: true,
			distributionId: distributionId.toString(),
			bonusTransactionId: bonusTransactionId.toString(),
			walletTransactionId: walletTransactionId.toString(),
			walletBalanceAfter,
			isExternalPayment: skipWalletUpdate,
			traceId: finalTraceId,
		};
	} catch (error) {
		logger.error("Error recording transaction:", error);
		throw error;
	}
}

async function recordBonusClaim(options) {
	const {
		username,
		userId,
		packageId,
		packageName,
		packageType,
		bonusType,
		claimType,
		amountNTE,
		amountUSD,
		description,
		walletBalanceBefore,
		originalInvestment,
		session,
		metadata = {},
	} = options;

	// Generate trace ID for final bonus claim
	const traceId = generateTraceId("final_bonus_claim", username);

	return recordComprehensiveTransaction({
		username,
		userId,
		transactionType: "bonus_payment",
		bonusType: bonusType || "final_bonus_claim",
		amountNTE,
		amountUSD,
		description,
		metadata: {
			packageId,
			packageName,
			packageType,
			claimType,
			originalInvestment,
			triggerType: "user_action",
			...metadata,
		},
		walletBalanceBefore,
		isDeduction: false,
		skipWalletUpdate: false,
		traceId,
		session,
	});
}

module.exports = {
	recordPackagePurchase,
	generateTraceId,
	recordReferralBonus,
	recordComprehensiveTransaction,
	recordBonusClaim,
};
