const { Product } = require("../../products/product.model");
const { logger } = require("../../../core/logger/logger");
const mongoose = require("mongoose");
const crypto = require("crypto");
const { User } = require("../../users/user.model");
const {
	getValidatedReserveWallet: getValidatedReserveWalletUtil,
} = require("../../../utils/contractUtils");
const UserPackage = require("../userPackage.model");

function generateTransactionId() {
	const timestamp = Date.now().toString(36);
	const random = crypto.randomBytes(6).toString("base64").replace(/[+/=]/g, "");
	const uniqueChars = crypto
		.randomBytes(3)
		.toString("base64")
		.replace(/[+/=]/g, "");
	return `NTE_${timestamp}_${random}_${uniqueChars}`.toUpperCase();
}

function generateTraceId(type, username, date = null) {
	const targetDate = date || new Date();
	const dateString = targetDate.toISOString().split("T")[0].replace(/-/g, "");
	return `${type}_${username}_${dateString}`;
}

async function createUserPackage({
	packageId,
	userId,
	price,
	priceNTE,
	transactionId,
	username,
	blockchainTransactionHash,
	fromAddress,
	toAddress,
}) {
	try {
		if (!packageId || typeof packageId !== "string") {
			throw new Error("Invalid packageId");
		}

		if (!userId) {
			throw new Error("Invalid userId");
		}

		const parsedPrice = parseFloat(price);
		const parsedPriceNTE = parseFloat(priceNTE);

		if (isNaN(parsedPrice) || parsedPrice < 0) {
			throw new Error("Invalid price value");
		}

		if (isNaN(parsedPriceNTE) || parsedPriceNTE < 0) {
			throw new Error("Invalid priceNTE value");
		}

		if (!username || typeof username !== "string") {
			throw new Error("Invalid username");
		}

		let validPackageId;
		try {
			validPackageId = new mongoose.Types.ObjectId(packageId);
		} catch (err) {
			throw new Error("Invalid packageId format");
		}

		let validUserId;
		try {
			validUserId =
				typeof userId === "string"
					? new mongoose.Types.ObjectId(userId)
					: userId;
		} catch (err) {
			throw new Error("Invalid userId format");
		}

		const product = await Product.findById(validPackageId).lean();
		if (!product) {
			throw new Error("Product not found");
		}

		const purchaseDate = new Date();
		let startDate, endDate;

		if (product.packageType === "unlocked") {
			startDate = product.startDate
				? new Date(product.startDate)
				: purchaseDate;
			const lockPeriodValue =
				product.userBonusCalculation?.lockPeriod?.value || 2;
			const lockPeriodUnit =
				product.userBonusCalculation?.lockPeriod?.unit || "years";

			endDate = new Date(startDate);
			if (lockPeriodUnit === "years") {
				endDate.setFullYear(endDate.getFullYear() + lockPeriodValue);
			} else if (lockPeriodUnit === "days") {
				endDate.setDate(endDate.getDate() + lockPeriodValue);
			} else if (lockPeriodUnit === "months") {
				endDate.setMonth(endDate.getMonth() + lockPeriodValue);
			}
		} else {
			startDate = purchaseDate;
			const lockPeriodValue = product.lockPeriod?.value || 15;
			const lockPeriodUnit = product.lockPeriod?.unit || "days";

			endDate = new Date(startDate);
			if (lockPeriodUnit === "years") {
				endDate.setFullYear(endDate.getFullYear() + lockPeriodValue);
			} else if (lockPeriodUnit === "days") {
				endDate.setDate(endDate.getDate() + lockPeriodValue);
			} else if (lockPeriodUnit === "months") {
				endDate.setMonth(endDate.getMonth() + lockPeriodValue);
			}
		}

		const traceId = generateTraceId("package_creation", username);

		const newPackage = {
			packageId: validPackageId,
			userId: validUserId,
			price: parsedPrice,
			priceNTE: parsedPriceNTE,
			transactionId,
			username,
			status: "active",
			startDate,
			endDate,
			purchasedAt: purchaseDate,
			bonusClaimStatus: "pending",
			finalBonusClaimed: false,
			blockchainTransactionHash,
			blockchainTransaction: {
				hash: blockchainTransactionHash,
				fromAddress,
				toAddress: toAddress || process.env.RESERVE_WALLET_ADDRESS,
				network: "BSC",
				tokenContract: process.env.TOKEN_CONTRACT_ADDRESS,
				verifiedAt: purchaseDate,
				traceId,
			},
			metadata: {
				traceId,
				creationMethod: "blockchain_purchase",
			},
		};

		const result = await UserPackage.create(newPackage);
		logger.info(`User package created: ${result._id}`);
		return result;
	} catch (error) {
		logger.error("Error creating user package:", error);
		throw error;
	}
}

async function getUserByUsername(username) {
	return await User.findOne({ username }).select("+password");
}
module.exports = {
	generateTransactionId,
	generateTraceId,
	createUserPackage,
	getUserByUsername,
	getValidatedReserveWallet: getValidatedReserveWalletUtil,
};
