const { User } = require("../../users/user.model");
const { Product } = require("../../products/product.model");
const { DbeProgram } = require("../../dbe/dbeProgram.model");
const UserPackage = require("../userPackage.model");
const { DbeProgramAttendee } = require("../../dbe/dbeProgramAttendee.model");
const { BonusTransaction } = require("../../bonuses/bonusTransaction.model");
const { BonusDistribution } = require("../../bonuses/bonusDistribution.model");
const { WalletTransaction } = require("../../wallets/walletTransaction.model");
const { logger } = require("../../../core/logger/logger");
const { getTokenValue } = require("../../config/globalConfig.service");
const { getValidatedReserveWallet } = require("./userPackage.service");
const { verifyNTETransfer } = require("./blockchainVerification.service");
const { generateTraceId } = require("./transactionTracking.service");
const { PurchaseAnalyticsModel } = require("../purchaseAnalytics.model");
const { createWalletHash } = require("../../users/user.service");

async function validatePurchaseData(purchaseData) {
	const validationErrors = [];
	const warnings = [];

	try {
		const {
			username,
			packageId,
			quantity,
			priceUSD,
			priceNTE,
			transactionHash,
			fromAddress,
			checkValidation = true,
		} = purchaseData;

		if (!username || typeof username !== "string") {
			validationErrors.push("Valid username is required");
		}

		if (!packageId || typeof packageId !== "string") {
			validationErrors.push("Valid package ID is required");
		}

		const validQuantity = Math.max(1, Math.min(1000, parseInt(quantity) || 1));
		if (validQuantity !== quantity) {
			warnings.push(
				`Quantity adjusted from ${quantity} to ${validQuantity} for safety`,
			);
		}

		if (typeof priceUSD !== "number" || isNaN(priceUSD) || priceUSD <= 0) {
			validationErrors.push("Valid USD price is required");
		}

		if (typeof priceNTE !== "number" || isNaN(priceNTE) || priceNTE <= 0) {
			validationErrors.push("Valid NTE price is required");
		}

		if (!transactionHash || typeof transactionHash !== "string") {
			validationErrors.push("Valid transaction hash is required");
		}

		if (!fromAddress || typeof fromAddress !== "string") {
			validationErrors.push("Valid sender address is required");
		}

		if (transactionHash) {
			const [existingPackage, existingAnalytics] = await Promise.all([
				UserPackage.findOne({
					$or: [
						{ blockchainTransactionHash: transactionHash },
						{ "blockchainTransaction.hash": transactionHash },
					],
				}),
				PurchaseAnalyticsModel.findOne({ transactionHash }),
			]);

			if (existingPackage || existingAnalytics) {
				validationErrors.push(
					"Transaction hash already used: This blockchain transaction has already been used for a purchase.",
				);
				logger.error(
					`Replay attack blocked - tx ${transactionHash.slice(
						0,
						10,
					)}... already used by ${
						existingPackage?.username || existingAnalytics?.username
					}`,
				);
			} else {
				logger.info(
					`Tx ${transactionHash.slice(
						0,
						10,
					)}... is unique, replay check passed`,
				);
			}
		}

		let isFlexibleStaking = false;
		const currentNtePrice = await getTokenValue();

		// Package validation
		let packageDetails = null;
		try {
			packageDetails = await Product.findById(packageId);
			if (!packageDetails) {
				validationErrors.push("Package not found");
			} else {
				// Validate package status
				if (!packageDetails.isActive) {
					validationErrors.push("Package is not active");
				}

				// Validate price based on package type
				if (packageDetails.packageType === "unlocked") {
					const expectedPrice =
						packageDetails.price?.fixedUSD || packageDetails.price?.startUSD;
					const totalExpectedPrice = expectedPrice * validQuantity;

					if (
						expectedPrice &&
						Math.abs(Number(priceUSD) - totalExpectedPrice) > 0.01
					) {
						validationErrors.push(
							`Invalid price for package quantity. Expected: $${totalExpectedPrice}, Received: $${priceUSD}`,
						);
					}
				} else if (packageDetails.packageType === "locked") {
					// Locked packages: Check if it's flexible staking or fixed staking
					const packageNTE =
						packageDetails.userBonusCalculation?.packageUSD || 0;

					// Check if it's flexible staking (packageUSD is 0)
					if (packageNTE === 0) {
						isFlexibleStaking = true;
					}
					const packageUSD = packageNTE * currentNtePrice;
					if (!isFlexibleStaking) {
						if (packageUSD > 0) {
							// Fixed staking packages
							const totalExpectedPrice = packageUSD * validQuantity;

							if (Math.abs(Number(priceUSD) - totalExpectedPrice) > 0.01) {
								validationErrors.push(
									`Invalid price for staking package. Expected: $${totalExpectedPrice}, Received: $${priceUSD}`,
								);
							}
						} else {
							// Flexible staking: User can choose any amount, just validate it's positive
							if (Number(priceUSD) <= 0) {
								validationErrors.push(
									"Invalid price for flexible staking. Amount must be greater than $0",
								);
							}
						}
					}
				}

				// Check stock availability
				if (packageDetails.maxSellsCount && packageDetails.maxSellsCount > 0) {
					const soldCount = await UserPackage.countDocuments({ packageId });

					if (soldCount + validQuantity > packageDetails.maxSellsCount) {
						const available = packageDetails.maxSellsCount - soldCount;
						validationErrors.push(
							`Not enough packages available. Only ${available} remaining`,
						);
					}
				}
			}
		} catch (error) {
			validationErrors.push(`Error validating package: ${error.message}`);
		}

		// Price validation to prevent price manipulation
		if (packageDetails && priceUSD && checkValidation) {
			let packagePriceUSD = 0;

			// Determine package price based on package type
			if (packageDetails.packageType === "unlocked") {
				const fixedPrice = parseFloat(packageDetails.price?.fixedUSD);
				const startPrice = parseFloat(packageDetails.price?.startUSD);
				packagePriceUSD =
					!isNaN(fixedPrice) && fixedPrice > 0
						? fixedPrice
						: !isNaN(startPrice) && startPrice > 0
						? startPrice
						: 0;
			} else if (packageDetails.packageType === "locked") {
				const packageNTE =
					parseFloat(packageDetails.userBonusCalculation?.packageUSD) || 0;
				if (isNaN(packageNTE) || packageNTE < 0) {
					validationErrors.push("Invalid package NTE configuration");
					packagePriceUSD = 0;
				} else {
					packagePriceUSD = packageNTE * currentNtePrice;
					if (packageNTE === 0) {
						isFlexibleStaking = true;
					}
				}
			}

			// Skip strict price validation for flexible staking
			if (isFlexibleStaking) {
				logger.info("Flexible staking validation:", {
					packageName: packageDetails.name,
					packageType: packageDetails.packageType,
					userChosenAmount: priceUSD,
					note: "User can choose any amount for flexible staking",
				});

				if (priceUSD <= 0) {
					validationErrors.push(
						"Invalid amount for flexible staking: Must be greater than $0",
					);
				} else {
					logger.info("Flexible staking amount validation passed");
				}
			} else if (packagePriceUSD <= 0) {
				validationErrors.push("Package has no valid price configured");
			} else {
				// Allow only 2% tolerance for dynamic pricing fluctuations
				const priceTolerance = 0.02;
				const minPackagePrice = packagePriceUSD * (1 - priceTolerance);
				const maxPackagePrice = packagePriceUSD * (1 + priceTolerance);
				const pricePerUnit = priceUSD / quantity;

				logger.info("Package price validation:", {
					packageName: packageDetails.name,
					packageType: packageDetails.packageType,
					configuredPrice: packagePriceUSD,
					clientPrice: pricePerUnit,
					quantity,
					totalPrice: priceUSD,
					acceptableRange: `${minPackagePrice.toFixed(
						2,
					)} - ${maxPackagePrice.toFixed(2)}`,
				});

				if (pricePerUnit < minPackagePrice) {
					validationErrors.push(
						`Price too low: Sent $${pricePerUnit.toFixed(
							2,
						)} but package costs $${packagePriceUSD} (min: $${minPackagePrice.toFixed(
							2,
						)})`,
					);
				} else if (pricePerUnit > maxPackagePrice) {
					validationErrors.push(
						`Price too high: Sent $${pricePerUnit.toFixed(
							2,
						)} but package costs $${packagePriceUSD} (max: $${maxPackagePrice.toFixed(
							2,
						)})`,
					);
				} else {
					logger.info("Package price validation passed");
				}
			}

			// NTE amount validation
			try {
				const serverTokenPrice = await getTokenValue();

				if (serverTokenPrice && serverTokenPrice > 0) {
					const expectedNTE = priceUSD / serverTokenPrice;
					const tolerance = 0.02;
					const minAcceptableNTE = expectedNTE * (1 - tolerance);
					const maxAcceptableNTE = expectedNTE * (1 + tolerance);

					logger.info("Server NTE price validation:", {
						packageName: packageDetails.name,
						packageType: packageDetails.packageType,
						isFlexibleStaking,
						priceUSD,
						clientNTE: priceNTE,
						serverTokenPrice,
						expectedNTE: expectedNTE.toFixed(4),
						tolerance: `${tolerance * 100}%`,
						range: `${minAcceptableNTE.toFixed(4)} - ${maxAcceptableNTE.toFixed(
							4,
						)}`,
					});

					if (priceNTE < minAcceptableNTE || priceNTE > maxAcceptableNTE) {
						const priceDifferencePercent = (
							(Math.abs(priceNTE - expectedNTE) / expectedNTE) *
							100
						).toFixed(2);

						if (isFlexibleStaking) {
							validationErrors.push(
								`Price manipulation detected for flexible staking: Sent ${priceNTE.toFixed(
									4,
								)} NTE for $${priceUSD} but system expects ${expectedNTE.toFixed(
									4,
								)} NTE (±2% tolerance). Difference: ${priceDifferencePercent}%`,
							);
							logger.error(
								`Flexible staking NTE validation failed - Client: ${priceNTE}, Expected: ${expectedNTE}, Diff: ${priceDifferencePercent}%`,
							);
						} else {
							validationErrors.push(
								`Price manipulation detected: Sent ${priceNTE.toFixed(
									4,
								)} NTE for $${priceUSD} but system expects ${expectedNTE.toFixed(
									4,
								)} NTE (±2% tolerance). Difference: ${priceDifferencePercent}%`,
							);
							logger.error(
								`Price manipulation blocked - Client: ${priceNTE}, Expected: ${expectedNTE}, Diff: ${priceDifferencePercent}%`,
							);
						}
					} else {
						if (isFlexibleStaking) {
							logger.info(
								`Flexible staking NTE validation passed - User amount: $${priceUSD}, NTE: ${priceNTE}`,
							);
						} else {
							logger.info(
								"NTE price validation passed - within acceptable range",
							);
						}
					}
				} else {
					warnings.push("Could not retrieve system token price for validation");
				}
			} catch (error) {
				logger.error("Error in system price validation:", error);
				warnings.push(`Price validation warning: ${error.message}`);
			}
		}

		// Wallet validation
		let targetWalletAddress = null;
		if (packageDetails) {
			try {
				targetWalletAddress = getValidatedReserveWallet(packageDetails);
			} catch (error) {
				validationErrors.push(`Wallet validation error: ${error.message}`);
			}

			if (!targetWalletAddress) {
				validationErrors.push("No valid target wallet address found");
			}
		}

		// User validation
		let userDoc = null;
		try {
			userDoc = await User.findOne({ username });
			if (!userDoc) {
				validationErrors.push("User not found in database");
			}
		} catch (error) {
			validationErrors.push(`Error validating user: ${error.message}`);
		}

		if (!transactionHash || !fromAddress) {
			validationErrors.push("Transaction hash and sender address are required");
			return {
				isValid: validationErrors.length === 0,
				errors: validationErrors,
				warnings: warnings,
			};
		}

		const txHashRegex = /^0x[a-f0-9]{64}$/;

		if (!txHashRegex.test(transactionHash.toLowerCase())) {
			validationErrors.push("Invalid transaction hash format");
		}

		// Blockchain verification
		let transferVerification = null;
		if (transactionHash && fromAddress && targetWalletAddress && priceNTE) {
			try {
				const expectedAmount = parseFloat(priceNTE);
				if (isNaN(expectedAmount) || expectedAmount <= 0) {
					validationErrors.push(
						"Invalid NTE amount for blockchain verification",
					);
					return {
						isValid: false,
						errors: validationErrors,
						warnings: warnings,
					};
				}

				logger.info("Verifying blockchain transfer:", {
					transactionHash: transactionHash,
					fromAddress: fromAddress,
					toAddress: targetWalletAddress,
					expectedAmount: expectedAmount,
					priceNTE: priceNTE,
				});

				transferVerification = await verifyNTETransfer(
					transactionHash,
					fromAddress,
					targetWalletAddress,
					priceNTE,
					checkValidation,
				);

				// match wallet hash against stored hash for fromAddress
				const walletHash = createWalletHash(fromAddress);
				if (
					userDoc &&
					userDoc.walletHash &&
					userDoc.walletHash !== walletHash
				) {
					logger.warn(
						`Wallet hash mismatch detected for user ${username} during purchase validation`,
					);
					await User.updateOne(
						{ username },
						{
							$set: {
								isActive: false,
								isBlocked: true,
								blockedAt: new Date(),
								blockedBy: "system",
								blockReason: "Attempted unauthorized purchase!",
							},
						},
					);
					validationErrors.push(
						"Sender address does not match user's registered wallet addresses",
					);
				}

				logger.info("Blockchain verification result:", {
					isValid: transferVerification.isValid,
					error: transferVerification.error || "None",
					amount: transferVerification.amount,
					confirmations: transferVerification.confirmations,
				});

				if (!transferVerification.isValid) {
					// Block ALL blockchain verification failures including fake/non-existent transactions
					validationErrors.push(
						`Blockchain verification failed: ${
							transferVerification.error || "Transaction verification failed"
						}`,
					);
					logger.error(
						"Blockchain verification failed - purchase blocked:",
						transferVerification.error,
					);
				}
			} catch (error) {
				// Blockchain verification errors must block purchases
				validationErrors.push(
					`Blockchain verification error: ${
						error.message || "Failed to verify transaction"
					}`,
				);
				logger.error(
					"Blockchain verification exception - purchase blocked:",
					error,
				);
			}
		}

		// Price ratio validation
		if (priceUSD && priceNTE) {
			const impliedTokenPrice = priceUSD / priceNTE;
			if (impliedTokenPrice < 0.001 || impliedTokenPrice > 1) {
				warnings.push(
					`Unusual token price ratio detected: $${impliedTokenPrice.toFixed(
						6,
					)} per NTE`,
				);
			}
		}

		return {
			isValid: validationErrors.length === 0,
			errors: validationErrors,
			warnings,
			packageDetails,
			userDoc,
			transferVerification,
			validatedQuantity: validQuantity,
			summary: {
				username,
				packageId,
				packageName: packageDetails?.name || "Unknown",
				quantity: validQuantity,
				priceUSD,
				priceNTE,
				transactionHash: transactionHash?.slice(0, 10) + "...",
				fromAddress: fromAddress?.slice(0, 6) + "..." + fromAddress?.slice(-4),
				toAddress:
					targetWalletAddress?.slice(0, 6) +
					"..." +
					targetWalletAddress?.slice(-4),
			},
		};
	} catch (error) {
		return {
			isValid: false,
			errors: [`Validation process failed: ${error.message}`],
			warnings: [],
			packageDetails: null,
			userDoc: null,
			transferVerification: null,
			validatedQuantity: 1,
			summary: null,
		};
	}
}

// DBE join validation before processing
async function validateDBEJoinData(joinData) {
	const validationErrors = [];
	const warnings = [];

	try {
		let {
			username,
			programId,
			feeUSD,
			feeNTE,
			transactionHash,
			fromAddress,
			checkValidation = true,
		} = joinData;

		// Parse fees to numbers
		feeUSD = parseFloat(feeUSD) || 0;
		feeNTE = parseFloat(feeNTE) || 0;

		// Basic data validation
		if (!username || typeof username !== "string") {
			validationErrors.push("Valid username is required");
		}

		if (!programId || typeof programId !== "string") {
			validationErrors.push("Valid program ID is required");
		}

		if (!feeUSD || feeUSD <= 0) {
			validationErrors.push("Valid USD fee is required");
		}

		if (!feeNTE || feeNTE <= 0) {
			validationErrors.push("Valid NTE fee is required");
		}

		if (!transactionHash || typeof transactionHash !== "string") {
			validationErrors.push("Valid transaction hash is required");
		}

		if (!fromAddress || typeof fromAddress !== "string") {
			validationErrors.push("Valid sender address is required");
		}

		// Check if transaction hash has already been used for DBE joins (Replay Attack Prevention)
		if (transactionHash) {
			const [existingAttendance, existingAnalytics] = await Promise.all([
				DbeProgramAttendee.findOne({
					$or: [
						{ blockchainTransactionHash: transactionHash },
						{ "blockchainTransaction.hash": transactionHash },
					],
				}),
				PurchaseAnalyticsModel.findOne({ transactionHash }),
			]);

			if (existingAttendance || existingAnalytics) {
				validationErrors.push(
					"Transaction hash already used: This blockchain transaction has already been used for a DBE join.",
				);
				logger.error(
					`DBE REPLAY ATTACK BLOCKED: Transaction ${transactionHash.slice(
						0,
						10,
					)}... already used by user ${
						existingAttendance?.username || existingAnalytics?.username
					} for program ${existingAttendance?.programId}`,
				);
			} else {
				logger.info(
					`DBE Transaction hash ${transactionHash.slice(
						0,
						10,
					)}... is unique - proceeding with validation`,
				);
			}
		}

		// Program validation
		let programDetails = null;
		try {
			programDetails = await DbeProgram.findById(programId);
			if (!programDetails) {
				validationErrors.push("DBE program not found");
			} else {
				// Validate program status
				if (!programDetails.isActive) {
					validationErrors.push("DBE program is not active");
				}

				// Validate fee based on program
				const fixedFee = parseFloat(programDetails.price?.fixedUSD);
				const startFee = parseFloat(programDetails.price?.startUSD);
				const expectedFee =
					!isNaN(fixedFee) && fixedFee > 0
						? fixedFee
						: !isNaN(startFee) && startFee > 0
						? startFee
						: 0;

				if (expectedFee > 0 && Math.abs(feeUSD - expectedFee) > 0.01) {
					validationErrors.push(
						`Invalid fee for program. Expected: $${expectedFee}, Received: $${feeUSD}`,
					);
				}

				// Check if user is already joined this specific program
				const existingAttendance = await DbeProgramAttendee.findOne({
					username,
					programId,
				});

				if (existingAttendance) {
					validationErrors.push(
						"User is already participating in this DBE program",
					);
				}

				// Check if user is already in another active program
				const existingActiveAttendance = await DbeProgramAttendee.findOne({
					username,
					status: "active",
				});

				if (existingActiveAttendance) {
					validationErrors.push(
						"User is already participating in another active DBE program. You can only join one program at a time.",
					);
				}

				// Check program capacity
				if (programDetails.maxUsers && programDetails.maxUsers > 0) {
					const currentUsers = await DbeProgramAttendee.countDocuments({
						programId,
					});

					if (currentUsers >= programDetails.maxUsers) {
						validationErrors.push("DBE program has reached maximum capacity");
					}
				}
			}
		} catch (error) {
			validationErrors.push(`Error validating DBE program: ${error.message}`);
		}

		// Fee validation
		if (programDetails && feeUSD && checkValidation) {
			const expectedFee =
				programDetails.price?.fixedUSD || programDetails.price?.startUSD || 0;

			if (expectedFee <= 0) {
				validationErrors.push("DBE program has no valid fee configured");
			} else {
				// Allow 2% tolerance
				const feeTolerance = 0.02;
				const minFee = expectedFee * (1 - feeTolerance);
				const maxFee = expectedFee * (1 + feeTolerance);

				logger.info("DBE Program fee validation:", {
					programName: programDetails.name,
					configuredFee: expectedFee,
					clientFee: feeUSD,
					acceptableRange: `${minFee.toFixed(2)} - ${maxFee.toFixed(2)}`,
				});

				if (feeUSD < minFee) {
					validationErrors.push(
						`Fee too low: Sent $${feeUSD} but program costs $${expectedFee} (min: $${minFee.toFixed(
							2,
						)})`,
					);
				} else if (feeUSD > maxFee) {
					validationErrors.push(
						`Fee too high: Sent $${feeUSD} but program costs $${expectedFee} (max: $${maxFee.toFixed(
							2,
						)})`,
					);
				} else {
					logger.info("DBE Program fee validation passed");
				}
			}

			try {
				const serverTokenPrice = await getTokenValue();

				if (serverTokenPrice && serverTokenPrice > 0) {
					const expectedNTE = feeUSD / serverTokenPrice;
					const tolerance = 0.02;
					const minAcceptableNTE = expectedNTE * (1 - tolerance);
					const maxAcceptableNTE = expectedNTE * (1 + tolerance);

					logger.info("Server DBE NTE fee validation:", {
						programName: programDetails.name,
						feeUSD,
						clientNTE: feeNTE,
						serverTokenPrice,
						expectedNTE: expectedNTE.toFixed(4),
						range: `${minAcceptableNTE.toFixed(4)} - ${maxAcceptableNTE.toFixed(
							4,
						)}`,
					});

					if (feeNTE < minAcceptableNTE || feeNTE > maxAcceptableNTE) {
						const priceDifferencePercent = (
							(Math.abs(feeNTE - expectedNTE) / expectedNTE) *
							100
						).toFixed(2);
						validationErrors.push(
							`Price manipulation detected: Sent ${feeNTE.toFixed(
								4,
							)} NTE for $${feeUSD} but system expects ${expectedNTE.toFixed(
								4,
							)} NTE (±2% tolerance). Difference: ${priceDifferencePercent}%`,
						);
						logger.error(
							`DBE NTE validation failed - Client: ${feeNTE}, Expected: ${expectedNTE}, Diff: ${priceDifferencePercent}%`,
						);
					} else {
						logger.info("DBE NTE validation passed - within acceptable range");
					}
				} else {
					warnings.push("Could not retrieve system token price for validation");
				}
			} catch (error) {
				logger.error("Error in DBE fee validation:", error);
				warnings.push(`Fee validation warning: ${error.message}`);
			}
		}

		// Wallet validation
		let targetWalletAddress = null;
		if (programDetails) {
			try {
				targetWalletAddress = getValidatedReserveWallet(programDetails);
			} catch (error) {
				validationErrors.push(`Wallet validation error: ${error.message}`);
			}

			if (!targetWalletAddress) {
				validationErrors.push(
					"No valid target wallet address found for DBE program",
				);
			}
		}

		// User validation
		let userDoc = null;
		try {
			userDoc = await User.findOne({ username });
			if (!userDoc) {
				validationErrors.push("User not found in database");
			}
		} catch (error) {
			validationErrors.push(`Error validating user: ${error.message}`);
		}

		// Blockchain verification for DBE join
		let transferVerification = null;
		if (transactionHash && fromAddress && targetWalletAddress && feeNTE) {
			try {
				const expectedAmount = parseFloat(feeNTE);
				if (isNaN(expectedAmount) || expectedAmount <= 0) {
					validationErrors.push(
						"Invalid DBE NTE fee amount for blockchain verification",
					);
					return {
						isValid: false,
						errors: validationErrors,
						warnings: warnings,
					};
				}

				logger.info("Verifying DBE blockchain transfer:", {
					transactionHash: transactionHash,
					fromAddress: fromAddress,
					toAddress: targetWalletAddress,
					expectedAmount: expectedAmount,
					feeNTE: feeNTE,
				});

				transferVerification = await verifyNTETransfer(
					transactionHash,
					fromAddress,
					targetWalletAddress,
					feeNTE,
					checkValidation,
				);

				// match wallet hash against stored hash for fromAddress
				const walletHash = createWalletHash(fromAddress);
				if (
					userDoc &&
					userDoc.walletHash &&
					userDoc.walletHash !== walletHash
				) {
					logger.warn(
						`Wallet hash mismatch detected for user ${username} during purchase validation`,
					);
					await User.updateOne(
						{ username },
						{
							$set: {
								isActive: false,
								isBlocked: true,
								blockedAt: new Date(),
								blockedBy: "system",
								blockReason: "Attempted unauthorized purchase!",
							},
						},
					);
					validationErrors.push(
						"Sender address does not match user's registered wallet addresses",
					);
				}

				logger.info("DBE Blockchain verification result:", {
					isValid: transferVerification.isValid,
					error: transferVerification.error || "None",
					amount: transferVerification.amount,
					confirmations: transferVerification.confirmations,
				});

				if (!transferVerification.isValid) {
					validationErrors.push(
						`DBE Blockchain verification failed: ${
							transferVerification.error || "Transaction verification failed"
						}`,
					);
					logger.error(
						"DBE Blockchain verification failed - join blocked:",
						transferVerification.error,
					);
				}
			} catch (error) {
				validationErrors.push(
					`DBE Blockchain verification error: ${
						error.message || "Failed to verify transaction"
					}`,
				);
				logger.error(
					"DBE Blockchain verification exception - join blocked:",
					error,
				);
			}
		}

		if (feeUSD && feeNTE) {
			const impliedTokenPrice = feeUSD / feeNTE;
			if (impliedTokenPrice < 0.001 || impliedTokenPrice > 1) {
				warnings.push(
					`Unusual DBE token price ratio detected: $${impliedTokenPrice.toFixed(
						6,
					)} per NTE`,
				);
			}
		}

		return {
			isValid: validationErrors.length === 0,
			errors: validationErrors,
			warnings,
			packageDetails: programDetails,
			userDoc,
			transferVerification,
			summary: {
				username,
				programId,
				programName: programDetails?.name || "Unknown",
				feeUSD,
				feeNTE,
				transactionHash: transactionHash,
				fromAddress: fromAddress,
				toAddress: targetWalletAddress,
			},
		};
	} catch (error) {
		return {
			isValid: false,
			errors: [`Validation process failed: ${error.message}`],
			warnings: [],
			packageDetails: null,
			userDoc: null,
			transferVerification: null,
			summary: null,
		};
	}
}

// Verify package purchase completion
async function verifyPurchaseCompletion(transactionIds, username, packageId) {
	try {
		// Check user packages
		const userPackages = await UserPackage.find({
			username,
			packageId,
			transactionId: { $in: transactionIds },
		});

		// Check transaction records
		const bonusTransactions = await BonusTransaction.find({
			username,
			"metadata.packageId": packageId,
			"metadata.transactionId": { $in: transactionIds },
		}).lean();

		const walletTransactions = await WalletTransaction.find({
			username,
			"metadata.packageId": packageId,
			"metadata.transactionId": { $in: transactionIds },
		}).lean();

		const bonusDistributions = await BonusDistribution.find({
			username,
			"metadata.packageId": packageId,
			"metadata.transactionId": { $in: transactionIds },
		}).lean();

		return {
			success: true,
			completeness: {
				userPackages: userPackages.length,
				bonusTransactions: bonusTransactions.length,
				walletTransactions: walletTransactions.length,
				bonusDistributions: bonusDistributions.length,
				expectedCount: transactionIds.length,
			},
			allRecordsCreated:
				userPackages.length === transactionIds.length &&
				bonusTransactions.length === transactionIds.length &&
				walletTransactions.length === transactionIds.length &&
				bonusDistributions.length === transactionIds.length,
			details: {
				userPackageIds: userPackages.map((p) => p._id.toString()),
				transactionIds,
			},
		};
	} catch (error) {
		return {
			success: false,
			error: error.message,
			completeness: null,
			allRecordsCreated: false,
			details: null,
		};
	}
}

// Verify DBE join completion
async function verifyDBEJoinCompletion(attendanceId, username, programId) {
	try {
		// Check DBE attendance
		const attendance = await DbeProgramAttendee.findOne({
			_id: attendanceId,
			username,
			programId,
		});

		return {
			success: true,
			completeness: {
				attendance: attendance ? 1 : 0,
				expectedCount: 1,
			},
			allRecordsCreated: !!attendance,
			details: {
				attendanceId: attendance?._id.toString(),
			},
		};
	} catch (error) {
		return {
			success: false,
			error: error.message,
			completeness: null,
			allRecordsCreated: false,
			details: null,
		};
	}
}

function logPurchaseAttempt(purchaseData, validationResult, status) {
	// Get target wallet address from validation result
	const targetWalletAddress = validationResult.packageDetails
		? (() => {
				try {
					return getValidatedReserveWallet(validationResult.packageDetails);
				} catch (error) {
					return null;
				}
		  })()
		: null;

	// Generate trace ID for purchase attempt
	let traceId = null;
	try {
		traceId = generateTraceId(
			"purchase_attempt",
			purchaseData.username || "ANONYMOUS",
		);
	} catch (error) {
		logger.warn("Failed to generate trace ID for purchase attempt:", error);
	}

	const logData = {
		timestamp: new Date().toISOString(),
		status,
		username: purchaseData.username,
		packageId: purchaseData.packageId,
		quantity: validationResult.validatedQuantity || purchaseData.quantity,
		priceUSD: purchaseData.priceUSD,
		priceNTE: purchaseData.priceNTE,
		transactionHash: purchaseData.transactionHash,
		fromAddress: purchaseData.fromAddress,
		toAddress: targetWalletAddress,
		traceId: traceId,
		validation: {
			isValid: validationResult.isValid,
			errorCount: validationResult.errors.length,
			warningCount: validationResult.warnings.length,
			errors: validationResult.errors,
			warnings: validationResult.warnings,
		},
		blockchainVerification: validationResult.transferVerification
			? {
					isValid: validationResult.transferVerification.isValid,
					blockNumber: validationResult.transferVerification.blockNumber,
					confirmations: validationResult.transferVerification.confirmations,
			  }
			: null,
	};

	if (status === "SUCCESS") {
		logger.info("PURCHASE SUCCESS:", logData);
	} else if (status === "FAILED") {
		logger.error("PURCHASE FAILED:", logData);
	} else {
		logger.info("PURCHASE ATTEMPT:", logData);
	}

	return logData;
}

module.exports = {
	validatePurchaseData,
	validateDBEJoinData,
	verifyPurchaseCompletion,
	verifyDBEJoinCompletion,
	logPurchaseAttempt,
};
