const { rankService } = require("./rank.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const rankController = {
	// Get all ranks with search, filters and pagination
	async listRanks(req, res, next) {
		try {
			const {
				page = 1,
				limit = 50,
				search = "",
				sortBy = "level",
				sortOrder = "asc",
				activeOnly = "false",
			} = req.query;

			const options = {
				page: parseInt(page),
				limit: parseInt(limit),
				search,
				sortBy,
				sortOrder,
				activeOnly: activeOnly === "true",
			};

			const result = await rankService.listRanks(options);
			res.json(new ApiResponse(200, result, "Ranks retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Get rank by ID
	async getRankById(req, res, next) {
		try {
			const { id } = req.params;
			const rank = await rankService.getRankById(id);

			res.json(new ApiResponse(200, { rank }, "Rank retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Get rank by level
	async getRankByLevel(req, res, next) {
		try {
			const { level } = req.params;
			const rank = await rankService.getRankByLevel(parseInt(level));

			res.json(new ApiResponse(200, { rank }, "Rank retrieved successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Create a new rank - admin only
	async createRank(req, res, next) {
		try {
			const allowedFields = [
				"name",
				"level",
				"requirement",
				"monthlySalary",
				"description",
				"packageName",
				"packageType",
				"isActive",
				"requirements",
				"pairs",
				"peoplePerPair",
			];

			const rankData = {};
			allowedFields.forEach((field) => {
				if (req.body[field] !== undefined) {
					rankData[field] = req.body[field];
				}
			});

			const rank = await rankService.createRank(rankData);

			res
				.status(201)
				.json(new ApiResponse(201, { rank }, "Rank created successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Update rank - admin only
	async updateRank(req, res, next) {
		try {
			const { id } = req.params;

			const allowedFields = [
				"name",
				"level",
				"requirement",
				"monthlySalary",
				"description",
				"packageName",
				"packageType",
				"isActive",
				"requirements",
				"pairs",
				"peoplePerPair",
			];

			const updateData = {};
			allowedFields.forEach((field) => {
				if (req.body[field] !== undefined) {
					updateData[field] = req.body[field];
				}
			});

			const rank = await rankService.updateRank(id, updateData);

			res.json(new ApiResponse(200, { rank }, "Rank updated successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Delete a rank - admin only
	async deleteRank(req, res, next) {
		try {
			const { id } = req.params;
			const result = await rankService.deleteRank(id);

			res.json(new ApiResponse(200, result, "Rank deleted successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Toggle rank active status - admin only
	async toggleRankStatus(req, res, next) {
		try {
			const { id } = req.params;
			const { isActive } = req.body;

			if (typeof isActive !== "boolean") {
				throw new ApiError(400, "isActive field must be a boolean");
			}

			const rank = await rankService.toggleRankStatus(id, isActive);

			res.json(
				new ApiResponse(
					200,
					{ rank },
					`Rank ${isActive ? "activated" : "deactivated"} successfully`,
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get all active ranks
	async getActiveRanks(req, res, next) {
		try {
			const ranks = await rankService.getActiveRanks();
			res.json(
				new ApiResponse(200, { ranks }, "Active ranks retrieved successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Check user rank eligibility and progress
	async checkUserRankEligibility(req, res, next) {
		try {
			const { walletAddress } = req.query;

			if (!walletAddress || walletAddress.trim() === "") {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Wallet address is required"));
			}

			const rankData = await rankService.checkUserRankEligibility(walletAddress);

			if (!rankData) {
				return res
					.status(404)
					.json(new ApiResponse(404, null, "User not found"));
			}

			res.json(
				new ApiResponse(
					200,
					{
						currentRankLevel: rankData.currentRankLevel,
						currentRank: rankData.currentRank,
						ranks: rankData.ranks.map((rank) => ({
							_id: rank._id,
							name: rank.name,
							level: rank.level,
							monthlySalary: rank.monthlySalary,
							requirement: rank.requirement,
							achieved: rank.achieved,
							qualifies: rank.qualifies,
							isNext: rank.isNext,
							progress: rank.progress,
						})),
					},
					"User rank eligibility retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { rankController };
