const { Rank } = require("./rank.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { createWalletHash } = require("../users/user.service");
const rankManagementService = require("./rankManagement.service");
const { User } = require("../users/user.model");

const rankService = {
	// Get all ranks with optional filtering
	async listRanks(options = {}) {
		const {
			page = 1,
			limit = 50,
			search = "",
			sortBy = "level",
			sortOrder = "asc",
			activeOnly = false,
		} = options;

		const skip = (page - 1) * limit;
		const filter = {};

		if (activeOnly) {
			filter.isActive = true;
		}

		if (search) {
			filter.$or = [
				{ name: { $regex: search, $options: "i" } },
				{ description: { $regex: search, $options: "i" } },
				{ requirement: { $regex: search, $options: "i" } },
				{ packageName: { $regex: search, $options: "i" } },
			];
		}

		const sortOptions = {};
		sortOptions[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [ranks, totalRanks] = await Promise.all([
			Rank.find(filter).sort(sortOptions).skip(skip).limit(limit).lean(),
			Rank.countDocuments(filter),
		]);

		return {
			ranks,
			pagination: {
				page,
				limit,
				total: totalRanks,
				totalPages: Math.ceil(totalRanks / limit),
			},
		};
	},

	// Get rank by ID
	async getRankById(id) {
		const rank = await Rank.findById(id).lean();
		if (!rank) {
			throw new ApiError(404, "Rank not found");
		}
		return rank;
	},

	// Get rank by level
	async getRankByLevel(level) {
		const rank = await Rank.findOne({ level }).lean();
		if (!rank) {
			throw new ApiError(404, "Rank not found");
		}
		return rank;
	},

	// Create a new rank
	async createRank(rankData) {
		const {
			name,
			level,
			requirement,
			monthlySalary,
			description,
			packageName,
			packageType,
			isActive = true,
			requirements = {},
			pairs,
			peoplePerPair,
		} = rankData;

		// Validate required fields
		if (!name || !name.trim()) {
			throw new ApiError(400, "Rank name is required");
		}

		if (level === undefined || level === null) {
			throw new ApiError(400, "Rank level is required");
		}

		if (!requirement || !requirement.trim()) {
			throw new ApiError(400, "Rank requirement is required");
		}

		if (
			monthlySalary === undefined ||
			monthlySalary === null ||
			monthlySalary < 0
		) {
			throw new ApiError(400, "Monthly salary must be a positive number");
		}

		if (!description || !description.trim()) {
			throw new ApiError(400, "Rank description is required");
		}

		if (!packageName || !packageName.trim()) {
			throw new ApiError(400, "Package name is required");
		}

		if (!packageType || !packageType.trim()) {
			throw new ApiError(400, "Package type is required");
		}

		// Check if rank with same name or level already exists
		const existingRank = await Rank.findOne({
			$or: [{ name }, { level }],
		});

		if (existingRank) {
			if (existingRank.name === name) {
				throw new ApiError(409, `Rank with name "${name}" already exists`);
			}
			if (existingRank.level === level) {
				throw new ApiError(409, `Rank with level ${level} already exists`);
			}
		}

		const rank = new Rank({
			name: name.trim(),
			level,
			requirement: requirement.trim(),
			monthlySalary,
			description: description.trim(),
			packageName: packageName.trim(),
			packageType: packageType.trim(),
			isActive,
			requirements,
			pairs,
			peoplePerPair,
		});

		await rank.save();

		logger.info("Rank created", {
			rankId: rank._id,
			name: rank.name,
			level: rank.level,
		});

		return rank.toObject();
	},

	// Update rank
	async updateRank(id, updateData) {
		const rank = await Rank.findById(id);
		if (!rank) {
			throw new ApiError(404, "Rank not found");
		}

		// If updating name or level, check for duplicates
		if (updateData.name && updateData.name !== rank.name) {
			const existingName = await Rank.findOne({
				name: updateData.name,
				_id: { $ne: id },
			});
			if (existingName) {
				throw new ApiError(
					409,
					`Rank with name "${updateData.name}" already exists`,
				);
			}
		}

		if (updateData.level !== undefined && updateData.level !== rank.level) {
			const existingLevel = await Rank.findOne({
				level: updateData.level,
				_id: { $ne: id },
			});
			if (existingLevel) {
				throw new ApiError(
					409,
					`Rank with level ${updateData.level} already exists`,
				);
			}
		}

		// Validate monthlySalary if provided
		if (
			updateData.monthlySalary !== undefined &&
			updateData.monthlySalary < 0
		) {
			throw new ApiError(400, "Monthly salary must be a positive number");
		}

		// Update the rank
		Object.keys(updateData).forEach((key) => {
			if (updateData[key] !== undefined) {
				rank[key] = updateData[key];
			}
		});

		await rank.save();

		logger.info("Rank updated", {
			rankId: rank._id,
			name: rank.name,
			level: rank.level,
		});

		return rank.toObject();
	},

	// Delete rank
	async deleteRank(id) {
		const rank = await Rank.findById(id);
		if (!rank) {
			throw new ApiError(404, "Rank not found");
		}

		await Rank.findByIdAndDelete(id);

		logger.info("Rank deleted", {
			rankId: id,
			name: rank.name,
			level: rank.level,
		});

		return { deletedId: id, name: rank.name };
	},

	// Toggle rank active status
	async toggleRankStatus(id, isActive) {
		const rank = await Rank.findById(id);
		if (!rank) {
			throw new ApiError(404, "Rank not found");
		}

		rank.isActive = isActive;
		await rank.save();

		logger.info("Rank status toggled", {
			rankId: rank._id,
			name: rank.name,
			isActive,
		});

		return rank.toObject();
	},

	// Get all active ranks ordered by level
	async getActiveRanks() {
		const ranks = await Rank.find({ isActive: true }).sort({ level: 1 }).lean();
		return ranks;
	},

	// Get next rank by level
	async getNextRank(currentLevel) {
		const nextRank = await Rank.findOne({
			level: { $gt: currentLevel },
			isActive: true,
		})
			.sort({ level: 1 })
			.lean();
		return nextRank;
	},

	// Check user rank eligibility and progress
	async checkUserRankEligibility(walletAddress) {
		try {
			// Get user data
			const user = await User.findOne({
				walletHash: createWalletHash(walletAddress),
			});

			if (!user) {
				return null;
			}

			const currentLevel = user.currentRankLevel || 0;

			// Get all active ranks
			const ranks = await this.getActiveRanks();

			// For level 1 rank, we need binary tree analysis
			// For other ranks, we need referral data
			let referralsData = null;
			let binaryTreeData = null;

			if (ranks.some((r) => r.level === currentLevel + 1 && r.level === 1)) {
				// Get binary tree data for level 1 rank
				const level1Rank = ranks.find((r) => r.level === 1);
				binaryTreeData = await rankManagementService.getUserBinaryTreeStats(
					user.username,
					level1Rank,
				);
			}

			// Only fetch full referral data if needed for higher ranks
			if (ranks.some((r) => r.level === currentLevel + 1 && r.level > 1)) {
				referralsData = await rankManagementService.getUserReferrals(
					user.username,
				);
			}

			const eligibility = [];

			for (const rank of ranks) {
				let achieved = false;
				let qualifies = false;
				let progress = { current: 0, required: 0, description: "" };

				if (rank.level <= currentLevel) {
					achieved = true;
				} else if (rank.level === currentLevel + 1) {
					// Check next rank requirements
					if (rank.level === 1) {
						// Binary tree requirement: 50-50% split
						const totalPairs = rank.requirements?.totalPairs || rank.pairs || 0;
						const peoplePerPair = rank.peoplePerPair || 2;
						const requiredPerSide = totalPairs * (peoplePerPair / 2);

						if (binaryTreeData) {
							const leftCount = binaryTreeData.leftSide.totalNodes;
							const rightCount = binaryTreeData.rightSide.totalNodes;
							const pairs = Math.min(leftCount, rightCount);

							progress = {
								current: pairs,
								required: requiredPerSide,
								description: `${pairs}/${requiredPerSide} pairs (Left: ${leftCount}, Right: ${rightCount})`,
								leftSide: leftCount,
								rightSide: rightCount,
							};
							qualifies = pairs >= requiredPerSide;
						} else {
							progress = {
								current: 0,
								required: requiredPerSide,
								description: `0/${requiredPerSide} pairs`,
							};
						}
					} else {
						// Higher levels: direct previous rank holders
						if (!referralsData) {
							referralsData = await rankManagementService.getUserReferrals(
								user.username,
							);
						}

						const previousRankLevel =
							rank.requirements?.previousRankLevel || rank.level - 1;
						const minDirect = rank.requirements?.directPreviousRankMin || 0;
						const maxDirect = rank.requirements?.directPreviousRankMax;

						// Check if previous rank exists
						const previousRank = ranks.find(
							(r) => r.level === previousRankLevel,
						);
						if (!previousRank) {
							progress = {
								current: 0,
								required: minDirect,
								description: `0/${minDirect} direct ${
									"Rank Level " + previousRankLevel
								} holders (Previous rank not found)`,
							};
							qualifies = false;
						} else {
							// Get direct referral usernames (level 1 only, active users)
							const directReferralUsernames = referralsData.referredUsers
								.filter((u) => u.level === 1 && u.isActive)
								.map((u) => u.username);

							if (directReferralUsernames.length === 0) {
								progress = {
									current: 0,
									required: minDirect,
									description: `0/${minDirect} direct ${previousRank.name} holders`,
								};
								qualifies = minDirect === 0;
							} else {
								const directReferralUsers = await User.find({
									username: { $in: directReferralUsernames },
								}).lean();

								// Count how many have achieved the previous rank level
								let directPreviousRankCount = 0;
								for (const referralUser of directReferralUsers) {
									const userCurrentLevel = referralUser.currentRankLevel || 0;
									if (userCurrentLevel >= previousRankLevel) {
										directPreviousRankCount++;
									}
								}

								progress = {
									current: directPreviousRankCount,
									required: minDirect,
									description: `${directPreviousRankCount}/${minDirect} direct ${previousRank.name} holders`,
								};

								// Check min and max requirements
								if (directPreviousRankCount < minDirect) {
									qualifies = false;
								} else if (
									maxDirect !== null &&
									maxDirect !== undefined &&
									directPreviousRankCount > maxDirect
								) {
									qualifies = false;
								} else {
									qualifies = true;
								}
							}
						}
					}
				}

				eligibility.push({
					...rank,
					achieved,
					qualifies,
					progress,
					isNext: rank.level === currentLevel + 1,
				});
			}

			return {
				currentRankLevel: currentLevel || 0,
				currentRank: ranks.find((r) => r.level === currentLevel) || 0,
				ranks: eligibility,
			};
		} catch (error) {
			logger.error("Error checking user rank eligibility:", error);
			throw error;
		}
	},
};

module.exports = { rankService };
