const { Rank } = require("../ranks/rank.model");
const { User } = require("../users/user.model");
const { RankCronJobResult } = require("./rankCronJobResult.model");
const UserPackage = require("../purchase/userPackage.model");
const { logger } = require("../../core/logger/logger");
const { cacheService } = require("../../core/cache/cache.service");
const { CacheEntry } = require("../../core/models/cacheEntry.model");

class RankManagementService {
	constructor() {
		this.CACHE_TTL_SECONDS = 5 * 60; // 5 minutes
	}

	async getUserReferrals(username, maxLevels = Infinity) {
		try {
			const cacheKey = `referrals:${username.toLowerCase()}:${maxLevels}`;
			const cached = await cacheService.get(cacheKey);
			if (cached) return cached;

			const queue = [{ username, level: 0 }];
			const visitedUsernames = new Set();
			const allReferredUsersDetails = [];
			const activeReferredUsernames = new Set();

			visitedUsernames.add(username.toLowerCase());

			while (queue.length > 0) {
				const { username: currentUsername, level: currentLevel } =
					queue.shift();

				if (currentLevel >= maxLevels && maxLevels !== Infinity) continue;

				const directReferrals = await User.find({
					referralCode: new RegExp(`^${currentUsername}$`, "i"),
				})
					.select("username fullName createdAt isActive currentRankLevel")
					.lean();

				for (const user of directReferrals) {
					if (!visitedUsernames.has(user.username.toLowerCase())) {
						visitedUsernames.add(user.username.toLowerCase());

						allReferredUsersDetails.push({
							username: user.username,
							fullName: user.fullName,
							createdAt: user.createdAt,
							isActive: user.isActive,
							currentRankLevel: user.currentRankLevel || 0,
							level: currentLevel + 1,
							referrerUsername: currentUsername,
						});

						if (user.isActive) {
							activeReferredUsernames.add(user.username);
						}

						queue.push({ username: user.username, level: currentLevel + 1 });
					}
				}
			}

			const result = {
				totalReferrals: allReferredUsersDetails.length,
				activeReferrals: activeReferredUsernames.size,
				referredUsers: allReferredUsersDetails,
			};

			await cacheService.set(cacheKey, result, this.CACHE_TTL_SECONDS);
			return result;
		} catch (error) {
			logger.error("Error getting user referrals:", error);
			throw error;
		}
	}

	async getUserBinaryTreeStats(username, rank = null) {
		try {
			const cacheKey = `binary_tree:${username.toLowerCase()}:${
				rank?.packageType || "all"
			}:${rank?.packageName || "all"}`;
			const cached = await cacheService.get(cacheKey);
			if (cached) return cached;

			// Get direct referrals first
			const directReferrals = await User.find({
				referralCode: new RegExp(`^${username}$`, "i"),
			})
				.select("username")
				.lean();

			if (directReferrals.length === 0) {
				const result = {
					leftSide: { totalNodes: 0, referrals: [], nodeDetails: [] },
					rightSide: { totalNodes: 0, referrals: [], nodeDetails: [] },
				};
				await cacheService.set(cacheKey, result, this.CACHE_TTL_SECONDS);
				return result;
			}

			// Split direct referrals into two sides
			const leftSideRoots = [];
			const rightSideRoots = [];

			directReferrals.forEach((referral, index) => {
				if (index % 2 === 0) {
					leftSideRoots.push(referral);
				} else {
					rightSideRoots.push(referral);
				}
			});

			// Function to get all nodes under a set of root users
			const getNodesUnderRoots = async (roots) => {
				if (roots.length === 0)
					return { totalNodes: 0, referrals: [], nodeDetails: [] };

				const rootUsernames = roots.map((r) => r.username);
				const visitedUsernames = new Set();
				const queue = [...rootUsernames];
				const allReferrals = [];

				while (queue.length > 0) {
					const currentUsername = queue.shift();
					if (visitedUsernames.has(currentUsername.toLowerCase())) continue;

					visitedUsernames.add(currentUsername.toLowerCase());
					allReferrals.push(currentUsername);

					// Get children of current user
					const children = await User.find({
						referralCode: new RegExp(`^${currentUsername}$`, "i"),
					})
						.select("username")
						.lean();

					for (const child of children) {
						if (!visitedUsernames.has(child.username.toLowerCase())) {
							queue.push(child.username);
						}
					}
				}

				// Get all unlocked packages for these users with product details
				const matchConditions = {
					username: { $in: allReferrals },
				};

				// Build product match conditions based on rank requirements
				const productMatchConditions = {};

				if (rank?.packageType) {
					productMatchConditions["productInfo.packageType"] =
						rank.packageType.toLowerCase();
				}

				if (rank?.packageName) {
					productMatchConditions["productInfo.name"] = {
						$regex: new RegExp(rank.packageName, "i"),
					};
				}

				const nodeDetails = await UserPackage.aggregate([
					{ $match: matchConditions },
					{
						$lookup: {
							from: "products",
							localField: "packageId",
							foreignField: "_id",
							as: "productInfo",
						},
					},
					{
						$unwind: {
							path: "$productInfo",
							preserveNullAndEmptyArrays: true,
						},
					},
					...(Object.keys(productMatchConditions).length > 0
						? [{ $match: productMatchConditions }]
						: []),
					{
						$project: {
							username: 1,
							productName: "$productInfo.name",
							packageType: "$productInfo.packageType",
							price: 1,
							priceNTE: 1,
							purchaseDate: 1,
							transactionId: 1,
						},
					},
					{ $sort: { username: 1, purchaseDate: -1 } },
				]).exec();

				return {
					totalNodes: nodeDetails.length,
					referrals: allReferrals,
					nodeDetails: nodeDetails,
				};
			};

			const [leftSide, rightSide] = await Promise.all([
				getNodesUnderRoots(leftSideRoots),
				getNodesUnderRoots(rightSideRoots),
			]);

			const result = {
				leftSide,
				rightSide,
			};

			await cacheService.set(cacheKey, result, this.CACHE_TTL_SECONDS);
			return result;
		} catch (error) {
			logger.error("Error getting binary tree stats:", error);
			throw error;
		}
	}

	async checkRankQualification(
		username,
		rank,
		referralData = null,
		allRanks = null,
	) {
		try {
			if (!referralData) {
				referralData = await this.getUserReferrals(username, Infinity);
			}

			if (!allRanks) {
				allRanks = await Rank.find({ isActive: true })
					.sort({ level: 1 })
					.lean();
			}

			// Level 1: Binary tree requirement (50-50% split)
			if (rank.level === 1) {
				const binaryTreeData = await this.getUserBinaryTreeStats(
					username,
					rank,
				);

				const totalPairs = rank.requirements?.totalPairs || rank.pairs || 0;
				const peoplePerPair = rank.peoplePerPair || 2;
				const requiredPerSide = totalPairs * (peoplePerPair / 2);
				if (binaryTreeData) {
					const leftCount = binaryTreeData.leftSide.totalNodes;
					const rightCount = binaryTreeData.rightSide.totalNodes;
					const pairs = Math.min(leftCount, rightCount);

					return pairs >= requiredPerSide;
				} else {
					return false;
				}
			}

			// Level 2+: Check for direct referrals who achieved the previous rank
			if (rank.level > 1 && rank.requirements) {
				const previousRankLevel =
					rank.requirements.previousRankLevel || rank.level - 1;
				const minRequired = rank.requirements.directPreviousRankMin || 0;
				const maxAllowed = rank.requirements.directPreviousRankMax;

				const previousRank = allRanks.find(
					(r) => r.level === previousRankLevel,
				);
				if (!previousRank) {
					return false;
				}

				// Get direct referral usernames (level 1 only, active users)
				const directReferralUsernames = referralData.referredUsers
					.filter((u) => u.level === 1 && u.isActive)
					.map((u) => u.username);

				if (directReferralUsernames.length === 0) {
					return minRequired === 0;
				}

				// Query database to get actual currentRankLevel for each direct referral
				const directReferralUsers = await User.find({
					username: { $in: directReferralUsernames },
				})
					.select("username currentRankLevel")
					.lean();

				// Count how many have achieved the previous rank level
				let directPreviousRankCount = 0;
				for (const referralUser of directReferralUsers) {
					const userCurrentLevel = referralUser.currentRankLevel || 0;
					if (userCurrentLevel >= previousRankLevel) {
						directPreviousRankCount++;
					}
				}

				// Check min and max requirements
				if (directPreviousRankCount < minRequired) {
					return false;
				}

				if (
					maxAllowed !== null &&
					maxAllowed !== undefined &&
					directPreviousRankCount > maxAllowed
				) {
					return false;
				}

				return true;
			}

			return false;
		} catch (error) {
			logger.error("Error checking rank qualification:", error);
			throw error;
		}
	}

	async calculateUserRankQuick(username, visitedUsers = new Set()) {
		try {
			const cacheKey = `rank:${username.toLowerCase()}`;
			const cached = await cacheService.get(cacheKey);
			if (cached) return cached;

			if (visitedUsers.has(username.toLowerCase())) {
				logger.warn(`Circular reference detected for user: ${username}`);
				return null;
			}
			visitedUsers.add(username.toLowerCase());

			const allRanks = await Rank.find({ isActive: true })
				.sort({ level: 1 })
				.lean();

			if (allRanks.length === 0) {
				return null;
			}

			const user = await User.findOne({ username: username.toLowerCase() });
			if (!user) {
				throw new Error(`User ${username} not found`);
			}

			const referralData = await this.getUserReferrals(username, Infinity);

			let qualifiedRank = null;

			for (let i = allRanks.length - 1; i >= 0; i--) {
				const rank = allRanks[i];

				if (
					await this.checkRankQualification(
						username,
						rank,
						referralData,
						allRanks,
					)
				) {
					qualifiedRank = rank;
					break;
				}
			}

			await cacheService.set(cacheKey, qualifiedRank, this.CACHE_TTL_SECONDS);
			return qualifiedRank;
		} catch (error) {
			logger.error("Error calculating user rank:", error);
			throw error;
		}
	}

	async clearRankCache() {
		try {
			// Delete referral cache entries
			await CacheEntry.deleteMany({ key: { $regex: /^referrals:/ } });
			// Delete rank cache entries
			await CacheEntry.deleteMany({ key: { $regex: /^rank:/ } });
			logger.info("Rank cache cleared from database");
		} catch (error) {
			logger.error("Failed to clear rank cache from database:", error.message);
		}
	}

	async saveCronJobResult(jobResult) {
		const result = new RankCronJobResult(jobResult);
		await result.save();
	}

	async updateAllUsersRanks(isManual = false) {
		const startTime = Date.now();
		const jobId = `rank-update-${Date.now()}`;
		logger.info(`RANK UPDATE JOB STARTED | Job ID: ${jobId}`);

		let jobResult = {
			jobType: "rank-update",
			jobId,
			status: "running",
			startTime: new Date(),
			totalUsers: 0,
			usersUpdated: 0,
			errorCount: 0,
			errorDetails: [],
			userUpdates: [],
			durationMs: 0,
			isManual,
		};

		try {
			const allUsers = await User.find({
				isActive: true,
			})
				.select("username currentRankLevel fullName")
				.lean();

			jobResult.totalUsers = allUsers.length;
			logger.info(
				`RANK UPDATE | Processing ${allUsers.length} active users | Job: ${jobId}`,
			);

			let usersUpdated = 0;
			let errorCount = 0;
			const errorDetails = [];
			const userUpdates = [];
			const batchSize = 10;

			for (let i = 0; i < allUsers.length; i += batchSize) {
				const batch = allUsers.slice(i, i + batchSize);
				logger.info(
					`RANK UPDATE | Processing batch ${
						Math.floor(i / batchSize) + 1
					}/${Math.ceil(allUsers.length / batchSize)} | Users: ${
						batch.length
					} | Job: ${jobId}`,
				);

				const batchPromises = batch.map(async (user) => {
					try {
						logger.info(`RANK UPDATE | Calculating rank for user ${user.username} | Job: ${jobId}`);
						const qualifiedRank = await this.calculateUserRankQuick(
							user.username,
						);
						const currentRankLevel = user.currentRankLevel || 0;
						const newRankLevel = qualifiedRank?.level || 0;

						const userUpdate = {
							username: user.username,
							fullName: user.fullName || "",
							previousRankLevel: currentRankLevel,
							newRankLevel: newRankLevel,
							rankChanged: currentRankLevel !== newRankLevel,
							status: "processing",
							processedAt: new Date(),
							qualifiedRank: qualifiedRank
								? {
										name: qualifiedRank.name,
										level: qualifiedRank.level,
										monthlySalary: qualifiedRank.monthlySalary,
								  }
								: null,
						};

						if (currentRankLevel !== newRankLevel) {
							const updateData = {
								currentRankLevel: newRankLevel,
								rankAchievedAt: newRankLevel > 0 ? new Date() : null,
								updatedAt: new Date(),
							};

							if (
								user.currentRankLevel !== undefined &&
								user.currentRankLevel !== null
							) {
								const userDoc = await User.findOne({ username: user.username });
								const rankHistory = userDoc.rankHistory || [];

								rankHistory.push({
									level: user.currentRankLevel,
									achievedAt:
										userDoc.rankAchievedAt || userDoc.createdAt || new Date(),
									endedAt: new Date(),
								});

								if (newRankLevel > 0) {
									rankHistory.push({
										level: newRankLevel,
										achievedAt: new Date(),
										endedAt: null,
									});
								}

								updateData.rankHistory = rankHistory;
							} else if (newRankLevel > 0) {
								updateData.rankHistory = [
									{
										level: newRankLevel,
										achievedAt: new Date(),
										endedAt: null,
									},
								];
							}

							await User.updateOne(
								{ username: user.username },
								{ $set: updateData },
							);

							logger.info(
								`RANK UPDATE | DB WRITE: User ${
									user.username
								} rank updated from Level ${currentRankLevel} to Level ${newRankLevel} | Package: ${
									userUpdate.qualifiedRank?.name || "None"
								} | Job: ${jobId}`,
							);
							usersUpdated++;
							userUpdate.status = "updated";
						} else {
							logger.info(
								`RANK UPDATE | No change for user ${user.username} (remains at Level ${currentRankLevel}) | Job: ${jobId}`,
							);
							userUpdate.status = "no_change";
						}

						userUpdates.push(userUpdate);
						return { success: true, username: user.username };
					} catch (error) {
						logger.error(
							`RANK UPDATE | ERROR: User ${user.username} failed to update | Error: ${error.message} | Job: ${jobId}`,
						);
						errorCount++;

						const errorDetail = {
							username: user.username,
							error: error.message,
							timestamp: new Date(),
						};
						errorDetails.push(errorDetail);

						userUpdates.push({
							username: user.username,
							fullName: user.fullName || "",
							previousRankLevel: user.currentRankLevel || 0,
							newRankLevel: user.currentRankLevel || 0,
							rankChanged: false,
							status: "error",
							errorMessage: error.message,
							processedAt: new Date(),
						});

						return {
							success: false,
							username: user.username,
							error: error.message,
						};
					}
				});

				await Promise.all(batchPromises);

				if (i + batchSize < allUsers.length) {
					await new Promise((resolve) => setTimeout(resolve, 1000));
				}
			}

			await this.clearRankCache();
			logger.info(`RANK UPDATE | Cache cleared after updates | Job: ${jobId}`);

			const duration = Date.now() - startTime;

			jobResult = {
				...jobResult,
				status: "success",
				usersUpdated,
				errorCount,
				errorDetails: errorDetails.slice(0, 10),
				userUpdates: userUpdates.slice(0, 100),
				durationMs: duration,
				endTime: new Date(),
				summary: `Updated ${usersUpdated} out of ${allUsers.length} users in ${(
					duration / 1000
				).toFixed(2)}s`,
			};

			await this.saveCronJobResult(jobResult);

			const result = {
				totalUsers: allUsers.length,
				usersUpdated,
				errorCount,
				duration: `${(duration / 1000).toFixed(2)}s`,
				completedAt: new Date(),
			};

			logger.info(
				`RANK UPDATE | Job completed: ${usersUpdated}/${
					allUsers.length
				} users updated | ${errorCount} errors | Duration: ${(
					duration / 1000
				).toFixed(2)}s | Job: ${jobId}`,
			);
			return result;
		} catch (error) {
			const duration = Date.now() - startTime;

			jobResult = {
				...jobResult,
				status: "error",
				durationMs: duration,
				endTime: new Date(),
				errorMessage: error.message,
				summary: `Job failed after ${(duration / 1000).toFixed(2)}s: ${
					error.message
				}`,
			};

			await this.saveCronJobResult(jobResult);

			logger.error(
				`RANK UPDATE | Job failed: ${error.message} | Duration: ${(
					duration / 1000
				).toFixed(2)}s`,
			);
			throw error;
		}
	}

	async triggerManualRankUpdate() {
		logger.info("RANK UPDATE | Manual trigger initiated");
		const result = await this.updateAllUsersRanks(true);

		return result;
	}
}

module.exports = new RankManagementService();
