const rankSalaryService = require("./rankSalary.service");
const rankManagementService = require("./rankManagement.service");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

const rankSalaryController = {
	// Check rank salary eligibility for a user
	async checkEligibility(req, res) {
		try {
			const { username } = req.params;
			const { rankLevel } = req.query;

			const result = await rankSalaryService.checkRankSalaryEligibility(
				username,
				rankLevel ? parseInt(rankLevel) : null,
			);

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in checkEligibility:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Calculate rank salary for a user
	async calculateSalary(req, res) {
		try {
			const { username } = req.params;
			const { rankLevel } = req.query;

			const result = await rankSalaryService.calculateRankSalary(
				username,
				rankLevel ? parseInt(rankLevel) : null,
			);

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in calculateSalary:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Get all eligible users for rank salary
	async getEligibleUsers(req, res) {
		try {
			const result = await rankSalaryService.getAllEligibleUsersForRankSalary();

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in getEligibleUsers:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Check if rank salary has been distributed for a user
	async checkDistributionStatus(req, res) {
		try {
			const { username } = req.params;

			const distributed = await rankSalaryService.hasRankSalaryBeenDistributed(
				username,
			);

			res.status(200).json({
				success: true,
				data: {
					username,
					distributed,
				},
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in checkDistributionStatus:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Distribute rank salary to all eligible users
	async distributeToAllUsers(req, res) {
		try {
			const { force } = req.query;

			const result = await rankSalaryService.distributeRankSalariesToAllUsers(
				force === "true",
			);

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in distributeToAllUsers:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Distribute rank salary to a single user
	async distributeToUser(req, res) {
		try {
			const { username } = req.params;
			const { force } = req.query;

			const result = await rankSalaryService.distributeRankSalaryToUser(
				username,
				force === "true",
			);

			if (result.success) {
				res.status(200).json({
					success: true,
					data: result,
				});
			} else {
				res.status(400).json({
					success: false,
					message: result.error,
				});
			}
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in distributeToUser:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Get detailed rank salary transactions
	async getTransactions(req, res) {
		try {
			const filters = req.query;

			const result = await rankSalaryService.getDetailedRankSalaryTransactions(
				filters,
			);

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in getTransactions:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Get rank salary statistics
	async getStatistics(req, res) {
		try {
			const { monthKey } = req.query;

			const result = await rankSalaryService.getRankSalaryStatistics(monthKey);

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in getStatistics:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Update all users ranks
	async updateAllRanks(req, res) {
		try {
			const result = await rankManagementService.updateAllUsersRanks();

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in updateAllRanks:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},

	// Trigger manual rank update
	async triggerManualUpdate(req, res) {
		try {
			const result = await rankManagementService.triggerManualRankUpdate();

			res.status(200).json({
				success: true,
				data: result,
			});
		} catch (error) {
			if (error instanceof ApiError) {
				res.status(error.statusCode).json({
					success: false,
					message: error.message,
				});
			} else {
				logger.error("Error in triggerManualUpdate:", error);
				res.status(500).json({
					success: false,
					message: "Internal server error",
				});
			}
		}
	},
};

module.exports = { rankSalaryController };
