const { Rank } = require("../ranks/rank.model");
const { User } = require("../users/user.model");
const { BonusDistribution } = require("../bonuses/bonusDistribution.model");
const { BonusTransaction } = require("../bonuses/bonusTransaction.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { convertUSDToNTE } = require("../../utils/productCalculations");
const {
	recordComprehensiveTransaction,
} = require("../purchase/services/transactionTracking.service");

class RankSalaryService {
	getDaysInMonth(year, month) {
		return new Date(year, month + 1, 0).getDate();
	}

	getCurrentMonthInfo() {
		const now = new Date();
		const year = now.getFullYear();
		const month = now.getMonth();
		const daysInMonth = this.getDaysInMonth(year, month);

		const monthStart = new Date(year, month, 1);
		const monthEnd = new Date(year, month, daysInMonth, 23, 59, 59, 999);

		return {
			year,
			month,
			daysInMonth,
			monthStart,
			monthEnd,
			monthName: monthStart.toLocaleString("default", {
				month: "long",
				year: "numeric",
			}),
		};
	}

	async checkRankSalaryEligibility(username, rankLevel = null) {
		try {
			const user = await User.findOne({ username: username.toLowerCase() });
			if (!user) {
				return {
					eligible: false,
					reason: "User not found",
					details: null,
				};
			}

			const currentRankLevel = user.currentRankLevel || 0;
			const checkRankLevel = rankLevel !== null ? rankLevel : currentRankLevel;

			if (checkRankLevel === 0) {
				return {
					eligible: false,
					reason: "User has no rank",
					details: {
						currentRankLevel: 0,
						requiredDays: 0,
						daysHeld: 0,
					},
				};
			}

			const rank = await Rank.findOne({
				level: checkRankLevel,
				isActive: true,
			});
			if (!rank) {
				return {
					eligible: false,
					reason: "Rank not found or inactive",
					details: null,
				};
			}

			const monthInfo = this.getCurrentMonthInfo();
			const now = new Date();

			const rankHistory = user.rankHistory || [];

			let currentRankPeriod = null;

			for (let i = rankHistory.length - 1; i >= 0; i--) {
				const period = rankHistory[i];
				if (period.level === checkRankLevel && !period.endedAt) {
					currentRankPeriod = period;
					break;
				}
			}

			if (!currentRankPeriod && user.currentRankLevel === checkRankLevel) {
				const achievedAt = user.rankAchievedAt || user.createdAt;
				currentRankPeriod = {
					level: checkRankLevel,
					achievedAt: achievedAt,
					endedAt: null,
				};
			}

			if (!currentRankPeriod) {
				return {
					eligible: false,
					reason: "No active rank period found",
					details: {
						currentRankLevel,
						requiredDays: monthInfo.daysInMonth,
						daysHeld: 0,
					},
				};
			}

			const rankStartDate = new Date(currentRankPeriod.achievedAt);

			const eligibilityDate = monthInfo.monthStart;

			if (rankStartDate > eligibilityDate) {
				const daysHeld = Math.floor(
					(now - rankStartDate) / (1000 * 60 * 60 * 24),
				);

				return {
					eligible: false,
					reason:
						"Rank achieved during current month - must hold rank for full month",
					details: {
						currentRankLevel: checkRankLevel,
						rankName: rank.name,
						monthlySalary: rank.monthlySalary || 0,
						requiredDays: monthInfo.daysInMonth,
						daysHeld,
						rankAchievedAt: rankStartDate,
						eligibilityDate: new Date(monthInfo.year, monthInfo.month + 1, 1),
						monthInfo,
					},
				};
			}

			const daysHeld = Math.floor(
				(now - rankStartDate) / (1000 * 60 * 60 * 24),
			);

			return {
				eligible: true,
				reason: "Rank held for full month period",
				details: {
					currentRankLevel: checkRankLevel,
					rankName: rank.name,
					monthlySalary: rank.monthlySalary || 0,
					requiredDays: monthInfo.daysInMonth,
					daysHeld: Math.min(daysHeld, monthInfo.daysInMonth),
					daysHeldTotal: daysHeld,
					rankAchievedAt: rankStartDate,
					eligibilityDate: eligibilityDate,
					monthInfo,
				},
			};
		} catch (error) {
			logger.error("Error checking rank salary eligibility:", error);
			return {
				eligible: false,
				reason: "Error checking eligibility",
				error: error.message,
				details: null,
			};
		}
	}

	async calculateRankSalary(username, rankLevel = null) {
		try {
			const eligibility = await this.checkRankSalaryEligibility(
				username,
				rankLevel,
			);

			if (!eligibility.eligible) {
				return {
					success: false,
					eligible: false,
					reason: eligibility.reason,
					salaryUSD: 0,
					salaryNTE: 0,
					details: eligibility.details,
				};
			}

			const salaryUSD = eligibility.details.monthlySalary || 0;
			const salaryNTE = convertUSDToNTE(salaryUSD);

			return {
				success: true,
				eligible: true,
				salaryUSD,
				salaryNTE,
				details: eligibility.details,
			};
		} catch (error) {
			logger.error("Error calculating rank salary:", error);
			return {
				success: false,
				eligible: false,
				reason: "Error calculating salary",
				error: error.message,
				salaryUSD: 0,
				salaryNTE: 0,
			};
		}
	}

	async getAllEligibleUsersForRankSalary() {
		try {
			const usersWithRanks = await User.find({
				currentRankLevel: { $gt: 0 },
				isActive: true,
			}).lean();

			const eligibleUsers = [];

			for (const user of usersWithRanks) {
				const salaryResult = await this.calculateRankSalary(user.username);

				if (salaryResult.eligible && salaryResult.salaryUSD > 0) {
					eligibleUsers.push({
						username: user.username,
						fullName: user.fullName || "",
						currentRankLevel: user.currentRankLevel,
						...salaryResult,
					});
				}
			}

			return eligibleUsers;
		} catch (error) {
			logger.error("Error getting eligible users for rank salary:", error);
			throw error;
		}
	}

	async hasRankSalaryBeenDistributed(username) {
		try {
			const monthInfo = this.getCurrentMonthInfo();
			const monthKey = `${monthInfo.year}-${String(
				monthInfo.month + 1,
			).padStart(2, "0")}`;

			const existingDistribution = await BonusDistribution.findOne({
				username,
				bonusType: "rank_salary",
				"metadata.monthKey": monthKey,
				status: "completed",
			});

			return !!existingDistribution;
		} catch (error) {
			logger.error("Error checking rank salary distribution status:", error);
			return false;
		}
	}

	async distributeRankSalariesToAllUsers(force = false) {
		try {
			logger.info(
				"RANK SALARY DISTRIBUTION | Job started | Processing all eligible users",
			);

			const eligibleUsers = await this.getAllEligibleUsersForRankSalary();

			if (eligibleUsers.length === 0) {
				logger.warn("RANK SALARY DISTRIBUTION | No eligible users found");
				return {
					success: true,
					totalEligible: 0,
					successful: 0,
					failed: 0,
					alreadyDistributed: 0,
					totalSalariesDistributed: 0,
					errors: [],
					message: "No users eligible for rank salary",
				};
			}

			logger.info(
				`RANK SALARY DISTRIBUTION | Processing ${eligibleUsers.length} eligible users`,
			);

			const results = {
				totalEligible: eligibleUsers.length,
				successful: 0,
				failed: 0,
				alreadyDistributed: 0,
				totalSalariesDistributed: 0,
				errors: [],
				distributions: [],
			};

			const monthInfo = this.getCurrentMonthInfo();
			const monthKey = `${monthInfo.year}-${String(
				monthInfo.month + 1,
			).padStart(2, "0")}`;

			for (let i = 0; i < eligibleUsers.length; i++) {
				const eligibleUser = eligibleUsers[i];
				try {
					if (
						!force &&
						(await this.hasRankSalaryBeenDistributed(eligibleUser.username))
					) {
						results.alreadyDistributed++;
						logger.info(
							`RANK SALARY DISTRIBUTION | SKIP: User ${
								eligibleUser.username
							} rank salary already distributed for ${monthKey} | User ${
								i + 1
							}/${eligibleUsers.length}`,
						);
						continue;
					}

					const user = await User.findOne({ username: eligibleUser.username });
					if (!user) {
						results.failed++;
						logger.error(
							`RANK SALARY DISTRIBUTION | ERROR: User ${
								eligibleUser.username
							} not found in database | User ${i + 1}/${eligibleUsers.length}`,
						);
						continue;
					}

					const transactionResult = await recordComprehensiveTransaction({
						username: eligibleUser.username,
						userId: user._id,
						transactionType: "bonus_payment",
						bonusType: "rank_salary",
						amountNTE: eligibleUser.salaryNTE,
						amountUSD: eligibleUser.salaryUSD,
						description: `Monthly rank salary for ${eligibleUser.rankName} - ${monthInfo.monthName}`,
						metadata: {
							rankLevel: eligibleUser.rankLevel,
							rankName: eligibleUser.rankName,
							monthKey,
							monthName: monthInfo.monthName,
							year: monthInfo.year,
							month: monthInfo.month + 1,
							daysInMonth: monthInfo.daysInMonth,
							daysHeld: eligibleUser.daysHeld,
							rankAchievedAt: eligibleUser.rankAchievedAt,
							triggerType: "automatic_distribution",
							calculationMethod: "NTE-First",
						},
						walletBalanceBefore: user.walletBalance || 0,
						isDeduction: false,
					});

					if (transactionResult.success) {
						results.successful++;
						results.totalSalariesDistributed += eligibleUser.salaryNTE;
						results.distributions.push({
							username: eligibleUser.username,
							rankLevel: eligibleUser.rankLevel,
							rankName: eligibleUser.rankName,
							salaryNTE: eligibleUser.salaryNTE,
							salaryUSD: eligibleUser.salaryUSD,
							distributionId: transactionResult.distributionId,
						});
						logger.info(
							`RANK SALARY DISTRIBUTION | DB WRITE: User ${
								eligibleUser.username
							} rank salary distributed | Rank: ${eligibleUser.rankName} (L${
								eligibleUser.rankLevel
							}) | Salary: ${eligibleUser.salaryNTE.toFixed(
								4,
							)} NTE | Month: ${monthKey} | User ${i + 1}/${
								eligibleUsers.length
							}`,
						);
					} else {
						results.failed++;
						results.errors.push({
							username: eligibleUser.username,
							error: transactionResult.error || "Transaction failed",
						});
						logger.error(
							`RANK SALARY DISTRIBUTION | ERROR: Failed to record transaction for ${
								eligibleUser.username
							} | Error: ${transactionResult.error} | User ${i + 1}/${
								eligibleUsers.length
							}`,
						);
					}
				} catch (error) {
					results.failed++;
					results.errors.push({
						username: eligibleUser.username,
						error: error.message,
					});
					logger.error(
						`RANK SALARY DISTRIBUTION | CRITICAL ERROR: User ${
							eligibleUser.username
						} distribution crashed | Error: ${error.message} | User ${i + 1}/${
							eligibleUsers.length
						}`,
					);
				}
			}

			logger.info(
				`RANK SALARY DISTRIBUTION | Job completed: ${results.successful}/${
					results.totalEligible
				} users processed | ${results.totalSalariesDistributed.toFixed(
					4,
				)} NTE distributed | ${results.failed} errors | Month: ${monthKey}`,
			);

			return {
				success: true,
				...results,
				message: `Distributed rank salaries to ${results.successful} out of ${results.totalEligible} eligible users`,
			};
		} catch (error) {
			logger.error("RANK SALARY DISTRIBUTION | Job failed:", error.message);
			return {
				success: false,
				error: error.message,
				totalEligible: 0,
				successful: 0,
				failed: 0,
			};
		}
	}

	async distributeRankSalaryToUser(username, force = false) {
		try {
			if (!force && (await this.hasRankSalaryBeenDistributed(username))) {
				return {
					success: false,
					reason: "Rank salary already distributed for current month",
					alreadyDistributed: true,
				};
			}

			const salaryResult = await this.calculateRankSalary(username);

			if (!salaryResult.eligible) {
				return {
					success: false,
					reason: salaryResult.reason,
					eligible: false,
					details: salaryResult.details,
				};
			}

			if (salaryResult.salaryNTE <= 0) {
				return {
					success: false,
					reason: "No salary amount to distribute",
					eligible: true,
					salaryAmount: 0,
				};
			}

			const user = await User.findOne({ username: username.toLowerCase() });
			if (!user) {
				return {
					success: false,
					reason: "User not found",
				};
			}

			const monthInfo = this.getCurrentMonthInfo();
			const monthKey = `${monthInfo.year}-${String(
				monthInfo.month + 1,
			).padStart(2, "0")}`;

			const transactionResult = await recordComprehensiveTransaction({
				username,
				userId: user._id,
				transactionType: "bonus_payment",
				bonusType: "rank_salary",
				amountNTE: salaryResult.salaryNTE,
				amountUSD: salaryResult.salaryUSD,
				description: `Monthly rank salary for ${salaryResult.details.rankName} - ${monthInfo.monthName}`,
				metadata: {
					rankLevel: salaryResult.details.currentRankLevel,
					rankName: salaryResult.details.rankName,
					monthKey,
					monthName: monthInfo.monthName,
					year: monthInfo.year,
					month: monthInfo.month + 1,
					daysInMonth: monthInfo.daysInMonth,
					daysHeld: salaryResult.details.daysHeld,
					rankAchievedAt: salaryResult.details.rankAchievedAt,
					triggerType: "manual_distribution",
					calculationMethod: "NTE-First",
				},
				walletBalanceBefore: user.walletBalance || 0,
				isDeduction: false,
			});

			if (!transactionResult.success) {
				return {
					success: false,
					reason: "Failed to record transaction",
					error: transactionResult.error,
				};
			}

			return {
				success: true,
				salaryUSD: salaryResult.salaryUSD,
				salaryNTE: salaryResult.salaryNTE,
				rankLevel: salaryResult.details.currentRankLevel,
				rankName: salaryResult.details.rankName,
				monthKey,
				monthName: monthInfo.monthName,
				distributionId: transactionResult.distributionId,
				walletBalanceAfter: transactionResult.walletBalanceAfter,
			};
		} catch (error) {
			logger.error("Error distributing rank salary to user:", error);
			return {
				success: false,
				reason: "Error distributing rank salary",
				error: error.message,
			};
		}
	}

	async getDetailedRankSalaryTransactions(filters = {}) {
		try {
			const {
				username,
				rankLevel,
				monthKey,
				startDate,
				endDate,
				minAmount,
				maxAmount,
				page = 1,
				limit = 20,
				sortBy = "createdAt",
				sortOrder = "desc",
			} = filters;

			const searchQuery = {
				bonusType: "rank_salary",
			};

			if (username) {
				searchQuery.username = { $regex: new RegExp(`^${username}$`, "i") };
			}

			if (rankLevel) {
				searchQuery["metadata.rankLevel"] = parseInt(rankLevel);
			}

			if (monthKey) {
				searchQuery["metadata.monthKey"] = monthKey;
			}

			if (startDate || endDate) {
				searchQuery.createdAt = {};
				if (startDate) {
					searchQuery.createdAt.$gte = new Date(startDate);
				}
				if (endDate) {
					const endDateTime = new Date(endDate);
					endDateTime.setHours(23, 59, 59, 999);
					searchQuery.createdAt.$lte = endDateTime;
				}
			}

			if (minAmount || maxAmount) {
				searchQuery.amountNTE = {};
				if (minAmount) {
					searchQuery.amountNTE.$gte = parseFloat(minAmount);
				}
				if (maxAmount) {
					searchQuery.amountNTE.$lte = parseFloat(maxAmount);
				}
			}

			const sortObj = {};
			sortObj[sortBy] = sortOrder === "asc" ? 1 : -1;

			const skip = (page - 1) * limit;

			const [transactions, totalCount] = await Promise.all([
				BonusTransaction.find(searchQuery)
					.sort(sortObj)
					.skip(skip)
					.limit(limit)
					.lean(),
				BonusTransaction.countDocuments(searchQuery),
			]);

			const summaryPipeline = [
				{ $match: searchQuery },
				{
					$group: {
						_id: null,
						totalAmountNTE: { $sum: "$amountNTE" },
						totalAmountUSD: { $sum: "$amountUSD" },
						averageAmountNTE: { $avg: "$amountNTE" },
						totalTransactions: { $sum: 1 },
					},
				},
			];

			const [summaryResult] = await BonusTransaction.aggregate(
				summaryPipeline,
			).exec();

			const summary = summaryResult || {
				totalAmountNTE: 0,
				totalAmountUSD: 0,
				averageAmountNTE: 0,
				totalTransactions: 0,
			};

			const totalPages = Math.ceil(totalCount / limit);
			const hasNext = page < totalPages;
			const hasPrev = page > 1;

			return {
				success: true,
				data: {
					transactions: transactions.map((tx) => ({
						...tx,
						_id: tx._id.toString(),
						distributionId: tx.distributionId?.toString(),
						balanceBefore: tx.metadata?.walletBalanceBefore || 0,
						balanceAfter: tx.metadata?.walletBalanceAfter || 0,
					})),
					pagination: {
						total: totalCount,
						page,
						limit,
						totalPages,
						hasNext,
						hasPrev,
					},
					summary,
				},
			};
		} catch (error) {
			logger.error("Error getting detailed rank salary transactions:", error);
			return {
				success: false,
				message: "Error retrieving rank salary transactions",
				error: error.message,
				data: {
					transactions: [],
					pagination: {
						total: 0,
						page: 1,
						limit: 20,
						totalPages: 0,
						hasNext: false,
						hasPrev: false,
					},
					summary: {
						totalAmountNTE: 0,
						totalAmountUSD: 0,
						averageAmountNTE: 0,
						totalTransactions: 0,
					},
				},
			};
		}
	}

	async getRankSalaryStatistics(monthKey = null) {
		try {
			const monthInfo = this.getCurrentMonthInfo();
			const targetMonthKey =
				monthKey ||
				`${monthInfo.year}-${String(monthInfo.month + 1).padStart(2, "0")}`;

			const distributions = await BonusDistribution.find({
				bonusType: "rank_salary",
				"metadata.monthKey": targetMonthKey,
				status: "completed",
			}).lean();

			const activeRanks = await Rank.find({ isActive: true })
				.sort({ level: 1 })
				.lean();

			const rankStats = {};
			let totalDistributed = 0;
			let totalUsers = 0;

			for (const rank of activeRanks) {
				rankStats[rank.level] = {
					rankName: rank.name,
					monthlySalary: rank.monthlySalary || 0,
					usersReceived: 0,
					totalDistributed: 0,
				};
			}

			for (const dist of distributions) {
				const rankLevel = dist.metadata?.rankLevel || 0;
				if (rankStats[rankLevel]) {
					rankStats[rankLevel].usersReceived++;
					rankStats[rankLevel].totalDistributed += dist.amountNTE || 0;
				}
				totalDistributed += dist.amountNTE || 0;
				totalUsers++;
			}

			let eligibleUsers = [];
			if (
				!monthKey ||
				targetMonthKey ===
					`${monthInfo.year}-${String(monthInfo.month + 1).padStart(2, "0")}`
			) {
				eligibleUsers = await this.getAllEligibleUsersForRankSalary();
			}

			return {
				monthInfo: monthKey ? null : monthInfo,
				monthKey: targetMonthKey,
				totalEligibleUsers: eligibleUsers.length,
				totalUsersReceived: totalUsers,
				totalSalaryDistributedNTE: totalDistributed,
				totalSalaryDistributedUSD: totalDistributed,
				rankStatistics: Object.values(rankStats).filter(
					(stat) => stat.monthlySalary > 0,
				),
				distributionStatus: {
					completed: totalUsers,
					pending: Math.max(0, eligibleUsers.length - totalUsers),
				},
			};
		} catch (error) {
			logger.error("Error getting rank salary statistics:", error);
			throw error;
		}
	}
}

module.exports = new RankSalaryService();
