const { ApiResponse } = require("../../utils/ApiResponse");
const {
	getCronJobStatus,
	getHealthStatus,
	getCronJobResults,
	getCronJobAnalytics,
	forceRestartAllCronJobs,
	getSystemCronJobInfo,
	getRankCronJobResults,
} = require("../../core/scheduler/cronJob.service");

class CronJobController {
	/**
	 * Get cron job status
	 */
	async getStatus(req, res, next) {
		try {
			const status = getCronJobStatus();
			return res
				.status(200)
				.json(new ApiResponse(200, status, "Cron job status retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get health status
	 */
	async getHealth(req, res, next) {
		try {
			const health = getHealthStatus();
			return res
				.status(200)
				.json(new ApiResponse(200, health, "Health status retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get cron job execution history
	 */
	async getHistory(req, res, next) {
		try {
			const {
				limit = 20,
				page = 1,
				search = "",
				sortBy = "createdAt",
				sortOrder = "desc",
			} = req.query;

			const history = await getCronJobResults(
				parseInt(limit),
				parseInt(page),
				search,
				sortBy,
				sortOrder,
			);

			return res
				.status(200)
				.json(new ApiResponse(200, history, "Cron job history retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get cron job logs
	 */
	async getLogs(req, res, next) {
		try {
			const {
				limit = 50,
				page = 1,
				search = "",
				sortBy = "createdAt",
				sortOrder = "desc",
				jobType = "",
				status = "",
			} = req.query;

			const { CronJobLog } = require("./cronJobLog.model");

			const filter = {};

			if (jobType) {
				filter.jobType = jobType;
			}

			if (status) {
				filter.status = status;
			}

			if (search) {
				filter.$or = [
					{ jobType: { $regex: search, $options: "i" } },
					{ jobId: { $regex: search, $options: "i" } },
				];
			}

			const sortOptions = {};
			sortOptions[sortBy] = sortOrder === "desc" ? -1 : 1;

			const skip = (parseInt(page) - 1) * parseInt(limit);

			const [logs, total] = await Promise.all([
				CronJobLog.find(filter)
					.sort(sortOptions)
					.skip(skip)
					.limit(parseInt(limit))
					.lean(),
				CronJobLog.countDocuments(filter),
			]);

			const result = {
				logs,
				pagination: {
					total,
					page: parseInt(page),
					limit: parseInt(limit),
					totalPages: Math.ceil(total / parseInt(limit)),
				},
			};

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Cron job logs retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get cron job statistics
	 */
	async getStats(req, res, next) {
		try {
			const { CronJobLog } = require("./cronJobLog.model");

			const [
				totalJobs,
				successfulJobs,
				failedJobs,
				recentJobs,
				jobTypeStats,
			] = await Promise.all([
				CronJobLog.countDocuments(),
				CronJobLog.countDocuments({ isSuccess: true }),
				CronJobLog.countDocuments({ isSuccess: false }),
				CronJobLog.find()
					.sort({ startTime: -1 })
					.limit(10)
					.select("jobType status isSuccess durationMs startTime")
					.lean(),
				CronJobLog.aggregate([
					{
						$group: {
							_id: "$jobType",
							count: { $sum: 1 },
							successCount: {
								$sum: { $cond: ["$isSuccess", 1, 0] },
							},
							failureCount: {
								$sum: { $cond: ["$isSuccess", 0, 1] },
							},
							avgDuration: { $avg: "$durationMs" },
						},
					},
					{
						$sort: { count: -1 },
					},
				]),
			]);

			const stats = {
				totalJobs,
				successfulJobs,
				failedJobs,
				successRate:
					totalJobs > 0
						? ((successfulJobs / totalJobs) * 100).toFixed(2)
						: 0,
				recentJobs,
				jobTypeStats,
			};

			return res
				.status(200)
				.json(new ApiResponse(200, stats, "Cron job statistics retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get cron job analytics
	 */
	async getAnalytics(req, res, next) {
		try {
			const { jobType = null, limit = 10 } = req.query;

			const analytics = await getCronJobAnalytics(jobType, parseInt(limit));

			return res
				.status(200)
				.json(new ApiResponse(200, analytics, "Cron job analytics retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Force restart all cron jobs
	 */
	async forceRestart(req, res, next) {
		try {
			const result = forceRestartAllCronJobs();

			return res
				.status(200)
				.json(new ApiResponse(200, result, "Cron jobs restarted successfully"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get system cron job info
	 */
	async getSystemInfo(req, res, next) {
		try {
			const info = getSystemCronJobInfo();

			return res
				.status(200)
				.json(new ApiResponse(200, info, "System cron job info retrieved"));
		} catch (error) {
			next(error);
		}
	}

	/**
	 * Get rank cron job results
	 */
	async getRankResults(req, res, next) {
		try {
			const {
				limit = 20,
				page = 1,
				jobType = "",
				status = "",
				sortBy = "startTime",
				sortOrder = "desc",
			} = req.query;


			const results = await getRankCronJobResults(
				parseInt(limit),
				parseInt(page),
				jobType,
				status,
				sortBy,
				sortOrder,
			);

			return res
				.status(200)
				.json(new ApiResponse(200, results, "Rank cron job results retrieved"));
		} catch (error) {
			next(error);
		}
	}
}

module.exports = new CronJobController();
