const mongoose = require("mongoose");
const { DatabaseBackup } = require("./databaseBackup.model");
const path = require("path");
const fs = require("fs");
const { promisify } = require("util");
const { exec } = require("child_process");
const { logger } = require("../../core/logger/logger");
const execAsync = promisify(exec);

const loggerDB = {
	info: (message, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "INFO",
			message,
			...data,
		};
		logger.info(message, logEntry);
		return logEntry;
	},

	error: (message, error = null, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "ERROR",
			message,
			error: error
				? {
						message: error.message,
						stack: error.stack,
						name: error.name,
				  }
				: null,
			...data,
		};
		logger.error(message, logEntry);
		return logEntry;
	},

	success: (message, data = {}) => {
		const logEntry = {
			timestamp: new Date().toISOString(),
			level: "SUCCESS",
			message,
			...data,
		};
		logger.info(message, logEntry);
		return logEntry;
	},
};

// Makes a complete backup of our MongoDB database
async function createDatabaseBackup(maxRetries = 3) {
	let lastError;

	for (let attempt = 1; attempt <= maxRetries; attempt++) {
		try {
			loggerDB.info(
				`Starting database backup process... (attempt ${attempt}/${maxRetries})`,
			);

			const result = await performBackup();
			loggerDB.success("Database backup completed successfully", result);
			return result;
		} catch (error) {
			lastError = error;
			loggerDB.error(
				`Database backup failed (attempt ${attempt}/${maxRetries})`,
				error,
			);

			// If this isn't the last attempt, wait before retrying
			if (attempt < maxRetries) {
				const waitTime = Math.min(1000 * Math.pow(2, attempt - 1), 10000); // Exponential backoff, max 10s
				loggerDB.info(`Waiting ${waitTime}ms before retry...`);
				await new Promise((resolve) => setTimeout(resolve, waitTime));
			}
		}
	}

	// All attempts failed
	loggerDB.error("Database backup failed after all retry attempts", lastError);
	throw lastError;
}

async function performBackup() {
	const startTime = Date.now();

	loggerDB.info("Starting database backup process...");

	// Make sure we have a place to store backups
	const backupDir = path.join(process.cwd(), "nodemeta-db-backups");
	if (!fs.existsSync(backupDir)) {
		fs.mkdirSync(backupDir, { recursive: true });
	}

	// Give the backup a unique name with the current date/time
	const timestamp = new Date().toISOString().replace(/[:.]/g, "-");
	const backupPath = path.join(backupDir, `backup-${timestamp}`);
	fs.mkdirSync(backupPath, { recursive: true });

	const dbName =
		process.env.MONGODB_DB_NAME || mongoose.connection.db.databaseName;

	if (!dbName) {
		throw new Error("Database name not found");
	}

	loggerDB.info("Starting database export using MongoDB driver...", {
		database: dbName,
		outputPath: backupPath,
	});

	// Grab the actual database connection from Mongoose
	const db = mongoose.connection.db;

	// Find all the data tables in the database
	const collections = await db.listCollections().toArray();
	loggerDB.info(`Found ${collections.length} collections to backup`);

	let totalDocuments = 0;
	let totalSize = 0;

	// Save each table's data as JSON files
	for (const collectionInfo of collections) {
		const collectionName = collectionInfo.name;
		const collection = db.collection(collectionName);

		const documents = await collection.find({}).toArray();
		totalDocuments += documents.length;

		const filePath = path.join(backupPath, `${collectionName}.json`);
		fs.writeFileSync(filePath, JSON.stringify(documents, null, 2), "utf-8");

		const stats = fs.statSync(filePath);
		totalSize += stats.size;

		loggerDB.info(`Exported ${collectionName}: ${documents.length} documents`);
	}

	// Keep track of backup details
	const metadata = {
		database: dbName,
		timestamp: new Date().toISOString(),
		collections: collections.length,
		totalDocuments,
		mongoVersion: (await db.admin().serverInfo()).version,
	};

	fs.writeFileSync(
		path.join(backupPath, "_metadata.json"),
		JSON.stringify(metadata, null, 2),
		"utf-8",
	);

	const backupStats = { totalSize, fileCount: collections.length + 1 };

	// Zip everything up to save space
	const compressedBackupPath = `${backupPath}.tar.gz`;
	loggerDB.info("Compressing backup...");

	// Use platform-specific compression
	let compressCommand;
	if (process.platform === "win32") {
		// On Windows, try different compression methods
		try {
			// First try 7zip if available
			const sevenZipPath = compressedBackupPath.replace('.tar.gz', '.7z');
			compressCommand = `7z a "${sevenZipPath}" "${backupPath}\\*"`;
			await execAsync(compressCommand);

			// If 7zip worked, rename to .tar.gz for consistency
			if (fs.existsSync(sevenZipPath)) {
				fs.renameSync(sevenZipPath, compressedBackupPath);
			}
		} catch (sevenZipError) {
			try {
				// Try PowerShell compression to create a zip file
				const zipPath = compressedBackupPath.replace('.tar.gz', '.zip');
				const sourcePath = backupPath;
				const destPath = zipPath;

				compressCommand = `powershell -Command "Compress-Archive -Path '${sourcePath}' -DestinationPath '${destPath}' -Force"`;
				await execAsync(compressCommand);

				// Rename zip to tar.gz for consistency
				if (fs.existsSync(zipPath)) {
					fs.renameSync(zipPath, compressedBackupPath);
				}
			} catch (powershellError) {
				// Last resort: try tar with different path handling
				loggerDB.info("PowerShell compression failed, trying tar with Unix paths...");

				// Convert Windows paths to Unix-style paths for tar
				const unixBackupPath = backupPath.replace(/\\/g, '/').replace(/^([A-Z]):/, '/$1');
				const unixCompressedPath = compressedBackupPath.replace(/\\/g, '/').replace(/^([A-Z]):/, '/$1');

				compressCommand = `tar -czf "${unixCompressedPath}" -C "${unixBackupPath}" .`;
				await execAsync(compressCommand);
			}
		}
	} else {
		// Unix-like systems
		compressCommand = `tar -czf "${compressedBackupPath}" -C "${backupPath}" .`;
		await execAsync(compressCommand);
	}

	// Clean up the temporary files
	fs.rmSync(backupPath, { recursive: true, force: true });

	// Check how big the compressed file is
	const compressedStats = fs.statSync(compressedBackupPath);

	const endTime = Date.now();
	const duration = endTime - startTime;

	// Log this backup in our database
	await saveBackupRecord({
		timestamp: new Date(),
		backupPath: compressedBackupPath,
		database: dbName,
		originalSize: backupStats.totalSize,
		compressedSize: compressedStats.size,
		fileCount: backupStats.fileCount,
		duration,
		status: "success",
	});

	// Delete old backups to save space
	await cleanupOldBackups(backupDir);

	const result = {
		success: true,
		backupPath: compressedBackupPath,
		database: dbName,
		originalSize: backupStats.totalSize,
		compressedSize: compressedStats.size,
		compressionRatio: (
			((backupStats.totalSize - compressedStats.size) / backupStats.totalSize) *
			100
		).toFixed(1),
		fileCount: backupStats.fileCount,
		duration,
		timestamp: new Date().toISOString(),
	};

	return result;
}

// Removes old backup files, keeping only the most recent ones
async function cleanupOldBackups(backupDir) {
	try {
		const retentionDays =
			parseInt(process.env.DATABASE_BACKUP_RETENTION_DAYS) || 30;

		const files = fs.readdirSync(backupDir);
		const backupFiles = files.filter(
			(file) => file.startsWith("backup-") && (file.endsWith(".tar.gz") || file.endsWith(".zip") || file.endsWith(".7z")),
		);

		if (backupFiles.length <= retentionDays) {
			return;
		}

		// Put the newest backups first
		const fileStats = backupFiles.map((file) => ({
			name: file,
			path: path.join(backupDir, file),
			mtime: fs.statSync(path.join(backupDir, file)).mtime,
		}));

		fileStats.sort((a, b) => b.mtime - a.mtime);

		// Delete the old ones we don't need anymore
		const filesToDelete = fileStats.slice(retentionDays);

		for (const file of filesToDelete) {
			fs.unlinkSync(file.path);
		}

		loggerDB.info("Cleanup completed", {
			total: backupFiles.length,
			kept: Math.min(retentionDays, backupFiles.length),
			deleted: filesToDelete.length,
			retentionDays,
		});
	} catch (error) {
		loggerDB.error("Failed to cleanup old backups", error);
	}
}

// Stores backup info in the database for tracking
async function saveBackupRecord(backupData) {
	try {
		const backupRecord = new DatabaseBackup(backupData);
		await backupRecord.save();
	} catch (error) {
		loggerDB.error("Failed to save backup record", error);
	}
}

// Fetches the list of recent backups
async function getBackupHistory(limit = 10) {
	try {
		const backups = await DatabaseBackup.find({})
			.sort({ timestamp: -1 })
			.limit(limit)
			.lean();
		return backups;
	} catch (error) {
		loggerDB.error("Failed to get backup history", error);
		return [];
	}
}

// Puts the database back to how it was from a backup
async function restoreFromBackup(backupPath, maxRetries = 3) {
	let lastError;

	for (let attempt = 1; attempt <= maxRetries; attempt++) {
		try {
			loggerDB.info(
				`Starting database restore process... (attempt ${attempt}/${maxRetries})`,
				{ backupPath },
			);

			const result = await performRestore(backupPath);
			loggerDB.success("Database restore completed successfully");
			return result;
		} catch (error) {
			lastError = error;
			loggerDB.error(
				`Database restore failed (attempt ${attempt}/${maxRetries})`,
				error,
			);

			// If this isn't the last attempt, wait before retrying
			if (attempt < maxRetries) {
				const waitTime = Math.min(1000 * Math.pow(2, attempt - 1), 10000); // Exponential backoff, max 10s
				loggerDB.info(`Waiting ${waitTime}ms before retry...`);
				await new Promise((resolve) => setTimeout(resolve, waitTime));
			}
		}
	}

	// All attempts failed
	loggerDB.error("Database restore failed after all retry attempts", lastError);
	throw lastError;
}

async function performRestore(backupPath) {
	if (!fs.existsSync(backupPath)) {
		throw new Error(`Backup file not found: ${backupPath}`);
	}

	// Make a temporary folder to unpack the backup
	const tempDir = path.join(process.cwd(), "temp-restore");
	if (fs.existsSync(tempDir)) {
		fs.rmSync(tempDir, { recursive: true, force: true });
	}
	fs.mkdirSync(tempDir, { recursive: true });

	// Unzip the backup files
	loggerDB.info("Extracting backup...");
	let extractCommand;

	if (process.platform === "win32") {
		// On Windows, try different extraction methods
		try {
			// First try 7zip if available
			if (backupPath.toLowerCase().endsWith('.7z')) {
				extractCommand = `7z x "${backupPath}" -o"${tempDir}"`;
				await execAsync(extractCommand);
			} else if (backupPath.toLowerCase().endsWith('.zip')) {
				// Use PowerShell for zip files
				const tempDirWin = tempDir;
				const backupPathWin = backupPath;
				extractCommand = `powershell -Command "Expand-Archive -Path '${backupPathWin}' -DestinationPath '${tempDirWin}' -Force"`;
				await execAsync(extractCommand);
			} else {
				// Try tar with Unix path conversion
				const unixBackupPath = backupPath.replace(/\\/g, '/').replace(/^([A-Z]):/, '/$1');
				const unixTempDir = tempDir.replace(/\\/g, '/').replace(/^([A-Z]):/, '/$1');
				extractCommand = `tar -xzf "${unixBackupPath}" -C "${unixTempDir}"`;
				await execAsync(extractCommand);
			}
		} catch (primaryError) {
			try {
				// Fallback: try PowerShell Expand-Archive for any format
				loggerDB.info("Primary extraction failed, trying PowerShell...");
				const tempDirWin = tempDir;
				const backupPathWin = backupPath;
				extractCommand = `powershell -Command "Expand-Archive -Path '${backupPathWin}' -DestinationPath '${tempDirWin}' -Force"`;
				await execAsync(extractCommand);
			} catch (powershellError) {
				// Last resort: try tar with original paths
				loggerDB.info("PowerShell extraction failed, trying tar...");
				extractCommand = `tar -xzf "${backupPath}" -C "${tempDir}"`;
				await execAsync(extractCommand);
			}
		}
	} else {
		// Unix-like systems
		extractCommand = `tar -xzf "${backupPath}" -C "${tempDir}"`;
		await execAsync(extractCommand);
	}

	// Figure out which database we're working with
	const db = mongoose.connection.db;
	const dbName = db.databaseName;

	// Check the backup details
	const metadataPath = path.join(tempDir, "_metadata.json");
	if (!fs.existsSync(metadataPath)) {
		throw new Error("Backup metadata not found");
	}

	const metadata = JSON.parse(fs.readFileSync(metadataPath, "utf-8"));
	loggerDB.info("Restoring database", {
		from: metadata.database,
		to: dbName,
		collections: metadata.collections,
	});

	// Put each table's data back
	const files = fs.readdirSync(tempDir);
	const jsonFiles = files.filter(
		(file) => file.endsWith(".json") && file !== "_metadata.json",
	);

	for (const file of jsonFiles) {
		const collectionName = path.basename(file, ".json");
		const filePath = path.join(tempDir, file);

		const documents = JSON.parse(fs.readFileSync(filePath, "utf-8"));

		if (documents.length > 0) {
			const collection = db.collection(collectionName);
			await collection.deleteMany({}); // Remove what's currently there
			await collection.insertMany(documents);
		}

		loggerDB.info(`Restored ${collectionName}: ${documents.length} documents`);
	}

	// Remove the temporary files
	fs.rmSync(tempDir, { recursive: true, force: true });

	return { success: true, restoredCollections: jsonFiles.length };
}

module.exports = {
	createDatabaseBackup,
	getBackupHistory,
	restoreFromBackup,
};
