const { Task } = require('./task.model');
const { enqueueJob, registerProcessor } = require('../../queues/job.queue');
const { cacheService } = require('../../core/cache/cache.service');
const { validators } = require('../../utils/validators');

const TASK_QUEUE_LOCK_TTL = 10 * 60;

const taskService = {
  async create(type, payload, { priority = 0, requestId } = {}) {
    const task = await Task.create({ 
      type: validators.sanitizeString(type), 
      payload, 
      priority, 
      status: 'QUEUED' 
    });
    await enqueueJob(type, { taskId: task.id, ...payload, requestId });
    await cacheService.set(`task-queue-lock:${task.id}`, true, TASK_QUEUE_LOCK_TTL);
    return task;
  },

  registerProcessor(type, processor) {
    return registerProcessor(type, processor);
  },

  async markRunning(taskId) {
    return Task.findByIdAndUpdate(taskId, { status: 'RUNNING', $inc: { attempts: 1 } }, { new: true });
  },

  async markCompleted(taskId) {
    await cacheService.delete(`task-queue-lock:${taskId}`);
    return Task.findByIdAndUpdate(taskId, { status: 'COMPLETED' }, { new: true });
  },

  async markFailed(taskId, error) {
    await cacheService.delete(`task-queue-lock:${taskId}`);
    return Task.findByIdAndUpdate(taskId, { status: 'FAILED', lastError: error }, { new: true });
  },

  async findQueued(limit = 100) {
    return Task.find({ status: 'QUEUED' }).sort({ priority: -1, createdAt: -1 }).limit(limit);
  },

  async list({ status, type, limit = 50, skip = 0, userId } = {}) {
    const query = {};
    
    // Validate and sanitize filters
    if (status && validators.isValidTaskStatus(status)) {
      query.status = status;
    }
    if (type) {
      query.type = validators.sanitizeString(type);
    }
    if (userId) {
      query['payload.userId'] = userId;
    }

    const pagination = validators.parsePagination(limit, skip);

    const [items, total] = await Promise.all([
      Task.find(query)
        .sort({ priority: -1, createdAt: -1 })
        .skip(pagination.skip)
        .limit(pagination.limit),
      Task.countDocuments(query)
    ]);

    return { items, total, ...pagination };
  },

  async getById(taskId) {
    return Task.findById(taskId);
  },

  async countOldCompleted(beforeDate) {
    return Task.countDocuments({
      status: 'COMPLETED',
      updatedAt: { $lt: beforeDate }
    });
  },

  async deleteOldCompleted(beforeDate) {
    const result = await Task.deleteMany({
      status: 'COMPLETED',
      updatedAt: { $lt: beforeDate }
    });
    return result.deletedCount || 0;
  }
};

module.exports = { taskService };
