const UserPackage = require("../purchase/userPackage.model");
const { ApiError } = require("../../utils/ApiError");

const userPackageService = {
	async getUserPackages(queryParams) {
		const {
			page = 1,
			limit = 10,
			sortBy = "purchasedAt",
			sortOrder = "desc",
			search,
			username,
			packageName,
			packageType,
			status,
			startDate,
			endDate,
			minPrice,
			maxPrice,
			includeStats,
		} = queryParams;

		const parsedLimit = Math.min(Math.max(Number(limit) || 10, 1), 100);
		const parsedPage = Math.max(1, Number(page) || 1);

		// Build search filter with $or
		let searchFilter = {};
		if (search && search.trim()) {
			searchFilter = {
				$or: [
					{ transactionId: { $regex: search, $options: "i" } },
					{ status: { $regex: search, $options: "i" } },
				],
			};
		}

		// Build advanced filters
		const advancedFilters = [];

		if (status && status.trim()) {
			advancedFilters.push({ status: status });
		}

		if (startDate && startDate.trim()) {
			const dateFilter = { $gte: new Date(startDate) };
			if (endDate && endDate.trim()) {
				const end = new Date(endDate);
				end.setHours(23, 59, 59, 999);
				dateFilter.$lte = end;
			}
			advancedFilters.push({ purchasedAt: dateFilter });
		} else if (endDate && endDate.trim()) {
			const end = new Date(endDate);
			end.setHours(23, 59, 59, 999);
			advancedFilters.push({ purchasedAt: { $lte: end } });
		}

		if (minPrice && String(minPrice).trim()) {
			const priceFilter = { $gte: parseFloat(minPrice) };
			if (maxPrice && String(maxPrice).trim()) {
				priceFilter.$lte = parseFloat(maxPrice);
			}
			advancedFilters.push({ price: priceFilter });
		} else if (maxPrice && String(maxPrice).trim()) {
			advancedFilters.push({ price: { $lte: parseFloat(maxPrice) } });
		}

		// Combine filters
		const combinedFilter = {
			...searchFilter,
			...(advancedFilters.length > 0 ? { $and: advancedFilters } : {}),
		};

		const sortObject = {};
		sortObject[sortBy] = sortOrder === "desc" ? -1 : 1;

		const pipeline = [];

		if (Object.keys(combinedFilter).length > 0) {
			pipeline.push({ $match: combinedFilter });
		}

		pipeline.push(
			{
				$lookup: {
					from: "users",
					localField: "userId",
					foreignField: "_id",
					as: "userDetails",
				},
			},
			{
				$lookup: {
					from: "products",
					localField: "packageId",
					foreignField: "_id",
					as: "packageDetails",
				},
			},
			{
				$addFields: {
					userName: { $arrayElemAt: ["$userDetails.username", 0] },
					userFullName: { $arrayElemAt: ["$userDetails.fullName", 0] },
					userEmail: { $arrayElemAt: ["$userDetails.email", 0] },
					packageName: { $arrayElemAt: ["$packageDetails.name", 0] },
					packageType: { $arrayElemAt: ["$packageDetails.packageType", 0] },
				},
			},
		);

		// Filter by username after lookup
		if (username && username.trim()) {
			pipeline.push({
				$match: { userName: { $regex: username, $options: "i" } },
			});
		}

		// Filter by package name after lookup
		if (packageName && packageName.trim()) {
			pipeline.push({
				$match: { packageName: { $regex: packageName, $options: "i" } },
			});
		}

		// Filter by package type after lookup
		if (packageType && packageType.trim()) {
			pipeline.push({
				$match: { packageType: packageType },
			});
		}

		// Get total count
		const countPipeline = [...pipeline, { $count: "total" }];
		const countResult = await UserPackage.aggregate(countPipeline);
		const total = countResult.length > 0 ? countResult[0].total : 0;

		const totalPages = Math.ceil(total / parsedLimit);
		const skip = (parsedPage - 1) * parsedLimit;

		// Calculate filtered stats if requested
		let filteredStats = null;
		if (includeStats === "true" && total > 0) {
			const statsPipeline = [...pipeline];
			statsPipeline.push({
				$facet: {
					statusCounts: [{ $group: { _id: "$status", count: { $sum: 1 } } }],
					revenueTotals: [
						{
							$group: {
								_id: null,
								totalRevenue: { $sum: "$price" },
								totalRevenueNTE: { $sum: "$priceNTE" },
							},
						},
					],
					packageTypeCounts: [
						{ $group: { _id: "$packageType", count: { $sum: 1 } } },
					],
				},
			});

			const statsResult = await UserPackage.aggregate(statsPipeline);
			if (statsResult.length > 0) {
				const stats = statsResult[0];

				const statusCounts = {};
				stats.statusCounts.forEach((item) => {
					statusCounts[item._id] = item.count;
				});

				const packageTypeCounts = {};
				stats.packageTypeCounts.forEach((item) => {
					packageTypeCounts[item._id] = item.count;
				});

				filteredStats = {
					totalPackages: total,
					activePackages: statusCounts.active || 0,
					deactivatedPackages:
						statusCounts.deactivated_for_technical_problem || 0,
					completedPackages: statusCounts.completed || 0,
					expiredPackages: statusCounts.expired || 0,
					inactivePackages: statusCounts.inactive || 0,
					totalRevenue: stats.revenueTotals[0]?.totalRevenue || 0,
					totalRevenueNTE: stats.revenueTotals[0]?.totalRevenueNTE || 0,
					lockedPackages: packageTypeCounts.locked || 0,
					unlockedPackages: packageTypeCounts.unlocked || 0,
					averagePrice:
						total > 0 ? (stats.revenueTotals[0]?.totalRevenue || 0) / total : 0,
					averagePriceNTE:
						total > 0
							? (stats.revenueTotals[0]?.totalRevenueNTE || 0) / total
							: 0,
				};
			}
		}

		// Project, sort, paginate
		pipeline.push(
			{ $project: { userDetails: 0, packageDetails: 0 } },
			{ $sort: sortObject },
			{ $skip: skip },
			{ $limit: parsedLimit },
		);

		const packages = await UserPackage.aggregate(pipeline);

		return {
			success: true,
			packages: packages.map((pkg) => ({
				...pkg,
				_id: pkg._id.toString(),
				packageId: pkg.packageId.toString(),
				userId: pkg.userId.toString(),
			})),
			pagination: {
				page: parsedPage,
				limit: parsedLimit,
				totalPages,
				totalEntries: total,
				hasNext: parsedPage < totalPages,
				hasPrev: parsedPage > 1,
			},
			...(filteredStats ? { filteredStats } : {}),
		};
	},

	// Update package status
	async updatePackageStatus(packageId, newStatus) {
		const userPackage = await UserPackage.findById(packageId);

		if (!userPackage) {
			throw new ApiError(404, "Package not found");
		}

		const oldStatus = userPackage.status;
		userPackage.status = newStatus;

		if (
			newStatus === "inactive" ||
			newStatus === "deactivated_for_technical_problem"
		) {
			userPackage.deactivatedAt = new Date();
			userPackage.deactivationReason =
				newStatus === "deactivated_for_technical_problem"
					? "Technical problem"
					: "Admin deactivation";
		}

		if (newStatus === "expired") {
			userPackage.expiredAt = new Date();
		}

		if (newStatus === "active" && oldStatus !== "active") {
			userPackage.reactivatedAt = new Date();
		}

		await userPackage.save();

		return {
			success: true,
			message: "Package status updated successfully",
			package: {
				_id: userPackage._id,
				status: userPackage.status,
				updatedAt: userPackage.updatedAt,
			},
		};
	},

	async getPackageStats(queryParams) {
		const {
			search,
			username,
			packageName,
			packageType,
			status,
			startDate,
			endDate,
			minPrice,
			maxPrice,
		} = queryParams;

		let searchFilter = {};
		if (search && search.trim()) {
			searchFilter = {
				$or: [
					{ transactionId: { $regex: search, $options: "i" } },
					{ status: { $regex: search, $options: "i" } },
				],
			};
		}

		const advancedFilters = [];

		if (status && status.trim()) {
			advancedFilters.push({ status: status });
		}

		if (startDate && startDate.trim()) {
			const dateFilter = { $gte: new Date(startDate) };
			if (endDate && endDate.trim()) {
				const end = new Date(endDate);
				end.setHours(23, 59, 59, 999);
				dateFilter.$lte = end;
			}
			advancedFilters.push({ purchasedAt: dateFilter });
		} else if (endDate && endDate.trim()) {
			const end = new Date(endDate);
			end.setHours(23, 59, 59, 999);
			advancedFilters.push({ purchasedAt: { $lte: end } });
		}

		if (minPrice && String(minPrice).trim()) {
			const priceFilter = { $gte: parseFloat(minPrice) };
			if (maxPrice && String(maxPrice).trim()) {
				priceFilter.$lte = parseFloat(maxPrice);
			}
			advancedFilters.push({ price: priceFilter });
		} else if (maxPrice && String(maxPrice).trim()) {
			advancedFilters.push({ price: { $lte: parseFloat(maxPrice) } });
		}

		const combinedFilter = {
			...searchFilter,
			...(advancedFilters.length > 0 ? { $and: advancedFilters } : {}),
		};

		const pipeline = [];

		if (Object.keys(combinedFilter).length > 0) {
			pipeline.push({ $match: combinedFilter });
		}

		pipeline.push(
			{
				$lookup: {
					from: "users",
					localField: "userId",
					foreignField: "_id",
					as: "userDetails",
				},
			},
			{
				$lookup: {
					from: "products",
					localField: "packageId",
					foreignField: "_id",
					as: "packageDetails",
				},
			},
			{
				$addFields: {
					userName: { $arrayElemAt: ["$userDetails.username", 0] },
					packageName: { $arrayElemAt: ["$packageDetails.name", 0] },
					packageType: { $arrayElemAt: ["$packageDetails.packageType", 0] },
				},
			},
		);

		if (username && username.trim()) {
			pipeline.push({
				$match: { userName: { $regex: username, $options: "i" } },
			});
		}

		if (packageName && packageName.trim()) {
			pipeline.push({
				$match: { packageName: { $regex: packageName, $options: "i" } },
			});
		}

		if (packageType && packageType.trim()) {
			pipeline.push({
				$match: { packageType: packageType },
			});
		}

		pipeline.push({
			$facet: {
				statusCounts: [{ $group: { _id: "$status", count: { $sum: 1 } } }],
				revenueTotals: [
					{
						$group: {
							_id: null,
							totalRevenue: { $sum: "$price" },
							totalRevenueNTE: { $sum: "$priceNTE" },
							count: { $sum: 1 },
						},
					},
				],
				packageTypeCounts: [
					{ $group: { _id: "$packageType", count: { $sum: 1 } } },
				],
			},
		});

		const result = await UserPackage.aggregate(pipeline);

		if (result.length === 0) {
			return {
				success: true,
				stats: {
					totalPackages: 0,
					activePackages: 0,
					inactivePackages: 0,
					deactivatedPackages: 0,
					completedPackages: 0,
					expiredPackages: 0,
					totalRevenue: 0,
					totalRevenueNTE: 0,
					lockedPackages: 0,
					unlockedPackages: 0,
					averagePrice: 0,
					averagePriceNTE: 0,
				},
			};
		}

		const stats = result[0];
		const total = stats.revenueTotals[0]?.count || 0;

		const statusCounts = {};
		stats.statusCounts.forEach((item) => {
			statusCounts[item._id] = item.count;
		});

		const packageTypeCounts = {};
		stats.packageTypeCounts.forEach((item) => {
			packageTypeCounts[item._id] = item.count;
		});

		return {
			success: true,
			stats: {
				totalPackages: total,
				activePackages: statusCounts.active || 0,
				inactivePackages: statusCounts.inactive || 0,
				deactivatedPackages:
					statusCounts.deactivated_for_technical_problem || 0,
				completedPackages: statusCounts.completed || 0,
				expiredPackages: statusCounts.expired || 0,
				totalRevenue: stats.revenueTotals[0]?.totalRevenue || 0,
				totalRevenueNTE: stats.revenueTotals[0]?.totalRevenueNTE || 0,
				lockedPackages: packageTypeCounts.locked || 0,
				unlockedPackages: packageTypeCounts.unlocked || 0,
				averagePrice:
					total > 0 ? (stats.revenueTotals[0]?.totalRevenue || 0) / total : 0,
				averagePriceNTE:
					total > 0 ? (stats.revenueTotals[0]?.totalRevenueNTE || 0) / total : 0,
			},
		};
	},
};

module.exports = { userPackageService };
