const speakeasy = require("speakeasy");
const crypto = require("crypto");
const { logger } = require("../../core/logger/logger");

const SECRET_SALT = process.env.TWO_FA_SECRET_SALT;

// Derive encryption key from salt
const ENCRYPTION_KEY = crypto.scryptSync(SECRET_SALT, "salt", 32);
const ALGORITHM = "aes-256-gcm";

const usedCodesCache = new Map();
const CODE_EXPIRY_MS = 90000; // 90 seconds (3 time windows at 30s each)
const MAX_ATTEMPTS_PER_IP = 10; // Max attempts per IP in time window
const ATTEMPT_WINDOW_MS = 300000; // 5 minutes for rate limiting

const failedAttemptsCache = new Map();
const codeUsageStats = new Map();

// Clean up expired entries every 30 seconds
setInterval(() => {
	const now = Date.now();

	// Clean used codes
	for (const [key, data] of usedCodesCache.entries()) {
		if (now - data.timestamp > CODE_EXPIRY_MS) {
			usedCodesCache.delete(key);
		}
	}

	// Clean failed attempts
	for (const [ip, attempts] of failedAttemptsCache.entries()) {
		const recentAttempts = attempts.filter(
			(time) => now - time < ATTEMPT_WINDOW_MS,
		);
		if (recentAttempts.length === 0) {
			failedAttemptsCache.delete(ip);
		} else {
			failedAttemptsCache.set(ip, recentAttempts);
		}
	}
}, 30000);

// Encrypt 2FA secret before storing
function hash2FASecret(secret) {
	try {
		const iv = crypto.randomBytes(16);
		const cipher = crypto.createCipheriv(ALGORITHM, ENCRYPTION_KEY, iv);

		let encrypted = cipher.update(secret, "utf8", "hex");
		encrypted += cipher.final("hex");

		const authTag = cipher.getAuthTag();

		return `${iv.toString("hex")}:${authTag.toString("hex")}:${encrypted}`;
	} catch (error) {
		logger.error("2FA encryption error:", error);
		throw new Error("Failed to encrypt 2FA secret");
	}
}

// Decrypt 2FA secret for TOTP verification
function decrypt2FASecret(encryptedSecret) {
	try {
		const parts = encryptedSecret.split(":");
		if (parts.length !== 3) {
			throw new Error("Invalid encrypted secret format");
		}

		const iv = Buffer.from(parts[0], "hex");
		const authTag = Buffer.from(parts[1], "hex");
		const encrypted = parts[2];

		const decipher = crypto.createDecipheriv(ALGORITHM, ENCRYPTION_KEY, iv);
		decipher.setAuthTag(authTag);

		let decrypted = decipher.update(encrypted, "hex", "utf8");
		decrypted += decipher.final("utf8");

		return decrypted;
	} catch (error) {
		logger.error("2FA decryption error:", error);
		throw new Error("Failed to decrypt 2FA secret");
	}
}

// Verify a TOTP code against the plain secret
function verify2FACode(
	plainSecret,
	code,
	window = 1,
	userId = null,
	clientIP = null,
	method = null,
	username = null,
) {
	try {
		// Validate input parameters
		if (!plainSecret || typeof plainSecret !== "string") {
			logger.error(
				"Invalid 2FA secret format" + ` for user ${username || "unknown"}`,
			);
			return false;
		}

		// Validate code format before verification
		if (!code || !/^\d{6}$/.test(code)) {
			logger.error("Invalid TOTP code format", {
				codeLength: code?.length,
				ip: clientIP || "unknown",
				username: username || "unknown",
			});
			return false;
		}

		// Require userId for replay prevention
		if (!userId) {
			logger.error(
				"Missing userId for replay prevention" +
					` for user ${username || "unknown"}`,
			);
			return false;
		}

		// Require clientIP for brute force protection
		if (!clientIP) {
			logger.error(
				"Missing clientIP for brute force protection" +
					` for user ${username || "unknown"}`,
			);
			return false;
		}

		// Check brute force attempts from IP
		if (!failedAttemptsCache.has(clientIP)) {
			failedAttemptsCache.set(clientIP, []);
		}

		const attempts = failedAttemptsCache.get(clientIP);
		const now = Date.now();
		const recentAttempts = attempts.filter(
			(time) => now - time < ATTEMPT_WINDOW_MS,
		);

		if (recentAttempts.length >= MAX_ATTEMPTS_PER_IP) {
			logger.warn("Brute force attempt blocked", {
				ip: clientIP,
				attempts: recentAttempts.length,
				window: ATTEMPT_WINDOW_MS / 1000 / 60 + " minutes",
				username: username || "unknown",
			});
			return false;
		}

		// Check if code was recently used (replay attack prevention)
		const cacheKey = `${userId}:${code}`;
		const usedData = usedCodesCache.get(cacheKey);

		if (usedData) {
			const elapsed = now - usedData.timestamp;
			logger.warn("Replay attack detected - code already used", {
				userId: userId.substring(0, 10) + "...",
				code: "***" + code.substring(3),
				usedAgo: Math.floor(elapsed / 1000) + "s ago",
				originalIP: usedData.ip,
				currentIP: clientIP,
				method: method || "unknown",
				username: username || "unknown",
			});

			// Track failed attempt
			const attempts = failedAttemptsCache.get(clientIP) || [];
			attempts.push(now);
			failedAttemptsCache.set(clientIP, attempts);

			return false;
		}

		// TOTP verification with standard 30-second window
		const verified = speakeasy.totp.verify({
			secret: plainSecret,
			encoding: "base32",
			token: code,
			window: window, // 0 = strict (30s only), 1 = ±30s tolerance
		});

		if (!verified) {
			logger.warn("Invalid TOTP code", {
				userId: userId.substring(0, 10) + "...",
				ip: clientIP,
				window: window === 0 ? "strict (30s)" : `±${window * 30}s`,
				username: username || "unknown",
			});

			// Track failed attempt
			const attempts = failedAttemptsCache.get(clientIP) || [];
			attempts.push(now);
			failedAttemptsCache.set(clientIP, attempts);

			return false;
		}

		// Track successfully used code to prevent replay
		usedCodesCache.set(cacheKey, {
			timestamp: now,
			ip: clientIP,
			method: method || "unknown",
		});

		// Track usage statistics
		if (!codeUsageStats.has(userId)) {
			codeUsageStats.set(userId, []);
		}
		const userStats = codeUsageStats.get(userId);
		userStats.push({
			timestamp: now,
			code: "***" + code.substring(3),
			ip: clientIP,
		});
		codeUsageStats.set(userId, userStats.slice(-10)); // Keep last 10 uses

		// Clear failed attempts on successful verification
		if (failedAttemptsCache.has(clientIP)) {
			failedAttemptsCache.delete(clientIP);
		}

		logger.info("2FA verification successful", {
			userId: userId.substring(0, 10) + "...",
			ip: clientIP,
			method: method || "unknown",
			username: username || "unknown",
			window: window === 0 ? "strict (30s)" : `±${window * 30}s`,
		});

		return true;
	} catch (error) {
		logger.error("2FA verification exception", {
			error: error.message,
			ip: clientIP || "unknown",
			username: username || "unknown",
		});
		return false;
	}
}

// Clear used codes for a specific user (on logout/session end)
function clearUsedCodes(userId, clientIP = null) {
	// Clear used codes for this user
	for (const key of usedCodesCache.keys()) {
		if (key.startsWith(`${userId}:`)) {
			usedCodesCache.delete(key);
		}
	}

	// Clear failed attempts from this IP
	if (clientIP && failedAttemptsCache.has(clientIP)) {
		failedAttemptsCache.delete(clientIP);
	}
}

// Get statistics about 2FA usage
function get2FAStats() {
	return {
		usedCodesCount: usedCodesCache.size,
		failedAttemptsIPs: failedAttemptsCache.size,
		totalFailedAttempts: Array.from(failedAttemptsCache.values()).reduce(
			(sum, attempts) => sum + attempts.length,
			0,
		),
		totalUsers: codeUsageStats.size,
		recentActivity: Array.from(codeUsageStats.entries())
			.map(([userId, stats]) => ({
				userId: userId.substring(0, 10) + "...",
				lastUsed: stats[stats.length - 1]?.timestamp,
				totalUses: stats.length,
			}))
			.slice(-5), // Last 5 users
	};
}

module.exports = {
	hash2FASecret,
	decrypt2FASecret,
	verify2FACode,
	clearUsedCodes,
	get2FAStats,
};
