const jwt = require("jsonwebtoken");
const speakeasy = require("speakeasy");
const QRCode = require("qrcode");
const crypto = require("crypto");
const { jwtConfig } = require("../../config/jwt.config");
const { User } = require("./user.model");
const { userService, createWalletHash } = require("./user.service");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const { cacheService } = require("../../core/cache/cache.service");
const systemNotificationService = require("../../utils/system-notification.util");
const {
	decrypt2FASecret,
	verify2FACode,
	hash2FASecret,
} = require("./2fa-encryption");
const generateEmailFromTemplate = require("../../templates/generateEmailFromTemplate");

const TOKEN_TTL = 24 * 60 * 60; // 24 hours in seconds
const BLACKLIST_TTL = 24 * 60 * 60; // 24 hours in seconds

const generateVerificationCode = (length = 7) => {
	const chars = "ABCDEFGHJKLMNPQRSTUVWXYZ23456789";
	let code = "";
	for (let i = 0; i < length; i++) {
		code += chars.charAt(Math.floor(Math.random() * chars.length));
	}
	return code;
};

const authService = {
	// Two-step signup: initial request or verification - params: userData, deviceInfo
	async signup(userData, deviceInfo) {
		if (userData?.action === "verify") {
			return this.completeSignupVerification(userData);
		}
		return this.initiateSignup(userData);
	},

	// Create temp user and send verification code - params: userData
	async initiateSignup(userData) {
		const {
			username,
			email,
			password,
			walletAddress,
			fullName,
			referralCode,
			bio,
		} = userData;

		if (!username || !email || !password || !walletAddress) {
			throw new ApiError(400, "Missing required fields");
		}

		if (!referralCode) {
			throw new ApiError(400, "Referral code is required");
		}

		if (typeof password !== "string" || password.length < 8) {
			throw new ApiError(400, "Password must be at least 8 characters long");
		}

		if (await userService.walletExists(walletAddress)) {
			throw new ApiError(
				409,
				"An account with this wallet address already exists. Please use the login page to access your account.",
			);
		}

		if (await userService.usernameExists(username)) {
			throw new ApiError(409, "Username already taken");
		}

		if (await userService.emailExists(email)) {
			throw new ApiError(409, "Email already registered");
		}

		const referralIsValid = await userService.isValidReferralCode(referralCode);
		if (!referralIsValid) {
			throw new ApiError(400, "Invalid referral code.");
		}

		const verificationCode = generateVerificationCode(7);
		const tempUser = await userService.createTempUser({
			username,
			email,
			password,
			walletAddress,
			fullName,
			referralCode,
			bio,
			verificationCode,
		});

		// Send verification code email
		try {
			const emailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #00e5ff; margin-bottom: 20px; font-size: 24px;">🎉 Welcome to NodeMeta!</h2>
					
					<p style="margin-bottom: 15px;">Hello <strong>${username}</strong>,</p>
					
					<p style="margin-bottom: 20px;">Thank you for signing up! To complete your registration, please verify your email address using the code below.</p>
					
					<div style="background: linear-gradient(135deg, rgba(0, 229, 255, 0.15), rgba(0, 229, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 30px; margin: 30px 0; border-radius: 8px; text-align: center;">
						<p style="margin: 0 0 15px 0; color: #9aa3b2; font-size: 14px; text-transform: uppercase; letter-spacing: 1px;">Your Verification Code</p>
						<div style="background: rgba(0, 229, 255, 0.2); padding: 20px; border-radius: 8px; display: inline-block;">
							<p style="margin: 0; color: #fff; font-size: 36px; font-weight: 700; letter-spacing: 8px; font-family: 'Courier New', monospace;">${verificationCode}</p>
						</div>
						<p style="margin: 15px 0 0 0; color: #9aa3b2; font-size: 12px;">Enter this code to activate your account</p>
					</div>
					
					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>⏰ Time Sensitive</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">This verification code will expire in <strong>10 minutes</strong>. If you didn't create this account, please ignore this email.</p>
					</div>
					
					<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #3b82f6;"><strong>📋 Account Details</strong></p>
						<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
							<tr>
								<td style="padding: 5px 0; color: #9aa3b2;">Username:</td>
								<td style="padding: 5px 0; color: #fff; text-align: right; font-weight: 600;">${username}</td>
							</tr>
							<tr>
								<td style="padding: 5px 0; color: #9aa3b2;">Email:</td>
								<td style="padding: 5px 0; color: #fff; text-align: right;">${email}</td>
							</tr>
							<tr>
								<td style="padding: 5px 0; color: #9aa3b2;">Wallet Address:</td>
								<td style="padding: 5px 0; color: #9aa3b2; text-align: right; font-family: monospace; font-size: 11px; word-break: break-all;">${walletAddress}</td>
							</tr>
						</table>
					</div>
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(emailBody);

			systemNotificationService.sendToUser({
				username,
				email,
				subject: "Verify Your NodeMeta Account - Verification Code Inside",
				title: "Verify Account",
				body: `Welcome to NodeMeta! Your verification code is: ${verificationCode}. This code expires in 10 minutes.`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/auth/verify",
						email,
					},
				},
				sendEmail: true,
				sendPush: false,
			});
		} catch (notificationError) {
			logger.error("Failed to send signup verification email:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				email,
				username,
			});
		}

		logger.info("Signup verification code generated", {
			email: tempUser.email,
			walletAddress,
		});

		return {
			success: true,
			nextStep: "verify",
			email: tempUser.email,
			verificationExpiresAt: tempUser.expiresAt,
		};
	},

	// Verify code and complete user registration - params: email, verificationCode
	async completeSignupVerification({ email, verificationCode, walletAddress }) {
		if (!email || !verificationCode) {
			throw new ApiError(400, "Email and verification code are required");
		}

		const tempUser = await userService.getTempUserByEmail(email);
		if (!tempUser) {
			throw new ApiError(
				404,
				"No pending signup found for this email. Please start the signup process again.",
			);
		}

		if (new Date(tempUser.expiresAt) < new Date()) {
			await userService.deleteTempUser(email);
			throw new ApiError(
				400,
				"Verification code has expired. Please start the signup process again.",
			);
		}

		const cleanCode = String(verificationCode)
			.replace(/\s/g, "")
			.trim()
			.toUpperCase();
		const storedCode = String(tempUser.verificationCode)
			.replace(/\s/g, "")
			.trim()
			.toUpperCase();

		if (storedCode !== cleanCode) {
			logger.warn("Invalid verification code attempt", {
				email,
				received: cleanCode,
				expected: storedCode,
				receivedLength: cleanCode.length,
				expectedLength: storedCode.length,
			});
			throw new ApiError(
				400,
				"Invalid verification code. Please check your email and try again.",
			);
		}

		const createdUser = await userService.create({
			username: tempUser.username,
			email: tempUser.email,
			password: tempUser.password,
			walletAddress: tempUser.walletAddress,
			fullName: tempUser.fullName || tempUser.username,
			referralCode: tempUser.referralCode,
			bio: tempUser.bio,
		});

		/* await userService.deleteTempUser(email);
		 */
		// Send welcome email to user
		try {
			const emailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">✅ Account Verified Successfully!</h2>
					
					<p style="margin-bottom: 15px;">Hello <strong>${
						createdUser.username
					}</strong>,</p>
					
					<p style="margin-bottom: 20px;">Congratulations! Your NodeMeta account has been successfully verified and is now active. Welcome to the NodeMeta community!</p>
					
					<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Your Account Details</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Username:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${
									createdUser.username
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Email:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${
									createdUser.email
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Wallet Address:</td>
								<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 11px; word-break: break-all;">${walletAddress}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Referral Code:</td>
								<td style="padding: 8px 0; color: #00e5ff; text-align: right; font-weight: 600; font-size: 16px;">${
									createdUser.referralCode || "N/A"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
								<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Active</td>
							</tr>
						</table>
					</div>
					
					<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #3b82f6;"><strong>🚀 Next Steps</strong></p>
						<ul style="margin: 10px 0 0 0; padding-left: 20px; color: #e1e4eb;">
							<li style="margin: 8px 0;">Log in to your account and complete your profile</li>
							<li style="margin: 8px 0;">Enable Two-Factor Authentication (2FA) for enhanced security</li>
							<li style="margin: 8px 0;">Explore available packages and investment opportunities</li>
							<li style="margin: 8px 0;">Invite friends using your referral code to earn rewards</li>
						</ul>
					</div>
					
					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>🔒 Security Reminder</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">We recommend enabling 2FA to protect your account. Never share your password or wallet private keys with anyone.</p>
					</div>
					
					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/login" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Login Now</a>
						<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Go to Dashboard</a>
					</div>
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(emailBody);

			systemNotificationService.sendToUser({
				username: createdUser.username,
				email: createdUser.email,
				subject: "Welcome to NodeMeta - Account Verified!",
				title: "Welcome!",
				body: `Your NodeMeta account has been verified successfully. Welcome aboard, ${createdUser.username}!`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/dashboard",
					},
				},
				sendEmail: true,
				sendPush: false,
			});
		} catch (notificationError) {
			logger.error("Failed to send welcome email:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				email: createdUser.email,
				username: createdUser.username,
			});
		}

		// Send admin notification email
		try {
			const adminEmailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #00e5ff; margin-bottom: 20px; font-size: 22px;">🆕 New User Signup</h2>
					<p style="margin-bottom: 15px;">A new user has completed signup and verified their account:</p>

					<div style="background: linear-gradient(135deg, rgba(0, 229, 255, 0.15), rgba(0, 229, 255, 0.05)); border-left: 4px solid #00e5ff; padding: 20px; margin: 20px 0; border-radius: 8px;">
						<h3 style="color: #00e5ff; margin-top: 0; margin-bottom: 15px; font-size: 18px;">👤 User Information</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr><td style="padding: 6px 0; color: #9aa3b2;">User ID:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-family: monospace; font-size: 12px;">${
								createdUser._id
							}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Username:</td><td style="padding: 6px 0; color: #fff; text-align: right; font-weight: 600;">${
								createdUser.username
							}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Email:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${
								createdUser.email
							}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Full Name:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${
								createdUser.fullName || "Not provided"
							}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Bio:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${
								createdUser.bio || "Not provided"
							}</td></tr>
						</table>
					</div>

					<div style="background: linear-gradient(135deg, rgba(139, 92, 246, 0.15), rgba(139, 92, 246, 0.05)); border-left: 4px solid #8b5cf6; padding: 20px; margin: 20px 0; border-radius: 8px;">
						<h3 style="color: #8b5cf6; margin-top: 0; margin-bottom: 15px; font-size: 18px;">💰 Wallet & Referral</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Wallet Address:</td><td style="padding: 6px 0; color: #00e5ff; text-align: right; font-family: monospace; font-size: 11px; word-break: break-all;">${walletAddress}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Referral Code:</td><td style="padding: 6px 0; color: #00e5ff; text-align: right; font-weight: 600; font-size: 16px;">${
								createdUser.referralCode || "N/A"
							}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Referral Used:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${
								tempUser.referralCode
							}</td></tr>
						</table>
					</div>

					<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 20px 0; border-radius: 8px;">
						<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">📊 Account Status</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Status:</td><td style="padding: 6px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Active & Verified</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">2FA Enabled:</td><td style="padding: 6px 0; color: #9aa3b2; text-align: right;">No (Setup Required)</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Account Created:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${new Date(
								createdUser.createdAt,
							).toUTCString()}</td></tr>
							<tr><td style="padding: 6px 0; color: #9aa3b2;">Verification Completed:</td><td style="padding: 6px 0; color: #fff; text-align: right;">${new Date().toUTCString()}</td></tr>
						</table>
					</div>

					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>📈 User Statistics</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">This user is part of the growing NodeMeta community. Consider reaching out to welcome them personally.</p>
					</div>

					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 12px 24px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 6px; font-weight: 600; margin-right: 10px;">View User Profile</a>
						<a href="https://node-meta.com/dashboard/admin" style="display: inline-block; padding: 12px 24px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 6px; font-weight: 600;">User Management</a>
					</div>
				</div>
			`;
			const adminEmailHtml = generateEmailFromTemplate(adminEmailBody);
			systemNotificationService.sendToAdmin({
				subject: `New User Signup: ${createdUser.username}`,
				title: "New User Signup",
				body: `A new user has signed up and verified their account: ${createdUser.username} (${createdUser.email})`,
				html: adminEmailHtml,
				pushPayload: {
					username: createdUser.username,
					email: createdUser.email,
					walletAddress: createdUser.walletAddress,
					referralCode: createdUser.referralCode,
					signupTime: new Date().toISOString(),
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (adminNotificationError) {
			logger.error("Failed to send admin signup notification email:", {
				errorMessage: adminNotificationError.message,
				errorStack: adminNotificationError.stack,
				username: createdUser.username,
				email: createdUser.email,
			});
		}

		logger.info("User signup completed after verification", {
			email: createdUser.email,
			username: createdUser.username,
		});

		return {
			success: true,
			user: createdUser,
		};
	},

	// Authenticate user with password and optional 2FA - params: walletAddress, password, totpCode?, deviceInfo, clientIP
	async login(walletAddress, password, totpCode, deviceInfo, clientIP) {
		console.log("🐞 ~ auth.service.js:434 ~ deviceInfo:", deviceInfo);
		const failedAttempts =
			(await cacheService.get(`login-fail:${walletAddress}`)) || 0;
		if (failedAttempts >= 5) {
			logger.error("Account temporarily locked", {
				walletAddress,
				attempts: failedAttempts,
			});
			throw new ApiError(429, "Account temporarily locked", {
				message:
					"Too many failed login attempts. Please try again in 15 minutes.",
			});
		}

		const user = await userService.findByWalletAddressWithPassword(
			walletAddress,
		);

		if (!user) {
			throw new ApiError(401, "Invalid credentials");
		}

		if (user.isBlocked === true) {
			logger.warn("Blocked user attempted login", { userId: user._id });
			throw new ApiError(403, "Account blocked", {
				message: "Your account has been blocked. Please contact support.",
				blocked: true,
			});
		}

		if (!user.password) {
			throw new ApiError(403, "Password not set", {
				message:
					"This account was created before password authentication was implemented. Please contact support to set up a password for your account.",
				needsPasswordSetup: true,
			});
		}

		const isPasswordValid = await user.comparePassword(password);
		if (!isPasswordValid) {
			const attemptKey = `login-fail:${walletAddress}`;
			const attempts = (await cacheService.get(attemptKey)) || 0;
			await cacheService.set(attemptKey, attempts + 1, 15 * 60);

			logger.warn("Invalid password attempt", {
				walletAddress,
				attempts: attempts + 1,
			});

			throw new ApiError(401, "Invalid password");
		}

		if (user.twoFactorEnabled) {
			if (!totpCode) {
				throw new ApiError(400, "2FA code required", {
					message:
						"This account has 2FA enabled. Please provide your 6-digit authentication code.",
				});
			}

			if (!user.twoFactorSecretHash) {
				logger.error("2FA enabled but no secret hash found", {
					username: user.username,
				});
				throw new ApiError(500, "2FA configuration error", {
					message:
						"2FA is enabled but not properly configured. Please contact support.",
				});
			}

			const decryptedSecret = decrypt2FASecret(user.twoFactorSecretHash);
			const isValidTotp = verify2FACode(
				decryptedSecret,
				totpCode,
				1, // Allow 1 time window tolerance for testing (±30 seconds)
				walletAddress, // User ID for replay prevention
				clientIP, // IP for brute force protection
				"POST", // Method for audit logging
				user.username,
			);
			if (!isValidTotp) {
				const attemptKey = `2fa-fail:${user._id}`;
				const attempts = (await cacheService.get(attemptKey)) || 0;
				await cacheService.set(attemptKey, attempts + 1, 15 * 60);

				if (attempts >= 4) {
					throw new ApiError(401, "Invalid 2FA code", {
						message: "Invalid or expired 2FA code. Please try again.",
					});
				}

				throw new ApiError(401, "Invalid 2FA code", {
					message: "Invalid or expired 2FA code. Please try again.",
				});
			}

			await cacheService.delete(`2fa-fail:${user._id}`);
		}

		await cacheService.delete(`login-fail:${walletAddress}`);

		// SINGLE DEVICE LOGIN: Invalidate ALL other device tokens
		// Devices stay in user.devices[] for history, but only current device has active token
		if (user.devices && user.devices.length > 0) {
			for (const device of user.devices) {
				if (device.tokenId) {
					await this.invalidateTokenById(device.tokenId);
				}
			}
			logger.info(
				"Invalidated all previous device tokens for single-device login",
				{
					userId: user._id,
					previousDevices: user.devices.length,
				},
			);
		}

		const token = this.generateToken({
			...user,
			walletAddress,
		});
		const tokenId = this.registerToken(
			token,
			user._id.toString(),
			user.username,
			deviceInfo,
		);

		await userService.updateLastLogin(user._id, { ...deviceInfo, tokenId });

		// Send login notification email
		try {
			const loginTime = new Date().toLocaleString("en-US", {
				timeZone: "UTC",
				year: "numeric",
				month: "long",
				day: "numeric",
				hour: "2-digit",
				minute: "2-digit",
				second: "2-digit",
				hour12: false,
			});

			const emailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #3b82f6; margin-bottom: 20px; font-size: 24px;">🔐 New Login Detected</h2>
					
					<p style="margin-bottom: 15px;">Hello <strong>${user.username}</strong>,</p>
					
					<p style="margin-bottom: 20px;">A new login to your NodeMeta account was detected. If this was you, no action is needed.</p>
					
					<div style="background: linear-gradient(135deg, rgba(59, 130, 246, 0.15), rgba(59, 130, 246, 0.05)); border-left: 4px solid #3b82f6; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #3b82f6; margin-top: 0; margin-bottom: 15px; font-size: 18px;">Login Details</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Time:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${loginTime} UTC</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">IP Address:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace;">${
									clientIP || "Unknown"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Device:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${
									deviceInfo?.deviceName ||
									deviceInfo?.deviceId ||
									"Unknown Device"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Browser:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right;">${
									deviceInfo?.label || "Unknown"
								}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">2FA Used:</td>
								<td style="padding: 8px 0; color: ${
									user.twoFactorEnabled ? "#10b981" : "#9aa3b2"
								}; text-align: right; font-weight: ${
				user.twoFactorEnabled ? "600" : "normal"
			};">${user.twoFactorEnabled ? "✓ Yes" : "No"}</td>
							</tr>
						</table>
					</div>
					
					<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #ef4444;"><strong>⚠️ Wasn't You?</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">If you didn't log in, please <strong>change your password immediately</strong> and contact our support team. Your account security is important to us.</p>
					</div>
					
					${
						!user.twoFactorEnabled
							? `
					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>🔒 Security Recommendation</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">We noticed you haven't enabled Two-Factor Authentication (2FA) yet. Enable 2FA to add an extra layer of security to your account.</p>
					</div>
					`
							: ""
					}
					
					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Go to Dashboard</a>
						${
							!user.twoFactorEnabled
								? `<a href="https://node-meta.com/dashboard/settings" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Enable 2FA</a>`
								: ""
						}
					</div>
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(emailBody);

			systemNotificationService.sendToUser({
				username: user.username,
				email: user.email,
				subject: "New Login to Your NodeMeta Account",
				title: "Login Detected",
				body: `A new login to your account was detected from IP: ${
					clientIP || "Unknown"
				}`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/settings/security",
					},
				},
				sendEmail: true,
				sendPush: false,
			});
		} catch (notificationError) {
			logger.error("Failed to send login notification email:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				username: user.username,
			});
		}

		logger.info("User logged in", {
			userId: user._id,
			username: user.username,
		});

		return {
			token,
			user,
		};
	},

	// Create JWT access token - params: user, expiresIn (default '24h')
	generateToken(user, expiresIn = "24h") {
		const { username, email, role, _id } = user._doc;
		const payload = {
			_id: _id,
			username: username,
			email: email,
			role: role,
			walletAddress: user.walletAddress,
		};

		// Sign with RSA private key (RS256)
		return jwt.sign(payload, jwtConfig.privateKey, {
			algorithm: jwtConfig.algorithm,
			expiresIn,
			issuer: jwtConfig.issuer,
			audience: jwtConfig.audience,
		});
	},

	// Validate JWT token and check blacklist/active status - params: token
	async verifyToken(token) {
		try {
			// Check if token is blacklisted (logged out)
			const isBlacklisted = await cacheService.exists(
				`token:blacklist:${token}`,
			);
			if (isBlacklisted) {
				throw new ApiError(401, "Token has been revoked");
			}

			// Check if token is still active (single-device enforcement)
			const isActive = await cacheService.exists(`token:active:${token}`);
			if (!isActive) {
				throw new ApiError(401, "Session expired. Please login again.", {
					code: "SESSION_REPLACED",
					message:
						"You have been logged out because you logged in from another device.",
				});
			}

			// Verify with RSA public key (RS256)
			const decoded = jwt.verify(token, jwtConfig.publicKey, {
				algorithms: [jwtConfig.algorithm],
				issuer: jwtConfig.issuer,
				audience: jwtConfig.audience,
			});

			return decoded;
		} catch (err) {
			if (err.name === "TokenExpiredError") {
				throw new ApiError(401, "Token expired");
			}
			if (err.name === "JsonWebTokenError") {
				throw new ApiError(401, "Invalid token");
			}
			throw err;
		}
	},

	// Track active token in cache - params: token, userId, username, metadata
	registerToken(token, userId, username, metadata = {}) {
		const tokenId = crypto.randomBytes(16).toString("hex");
		const tokenData = {
			tokenId,
			userId,
			username,
			deviceId: metadata.deviceId || null,
			createdAt: Date.now(),
			...metadata,
		};

		cacheService.set(`token:active:${token}`, tokenData, TOKEN_TTL);
		cacheService.set(
			`token:${tokenId}`,
			{ token, userId, username, deviceId: metadata.deviceId, ...metadata },
			TOKEN_TTL,
		);

		return tokenId;
	},

	// Invalidate old token by tokenId (used when same device logs in again)
	async invalidateTokenById(tokenId) {
		if (!tokenId) return;
		try {
			const tokenData = await cacheService.get(`token:${tokenId}`);
			if (tokenData && tokenData.token) {
				await cacheService.delete(`token:active:${tokenData.token}`);
				await cacheService.delete(`token:${tokenId}`);
				// Don't blacklist - it's a normal re-login, not a logout
				logger.debug("Old token invalidated for device re-login", { tokenId });
			}
		} catch (err) {
			logger.warn("Failed to invalidate old token", {
				tokenId,
				error: err.message,
			});
		}
	},

	// Blacklist token and remove from active tracking - params: token
	async invalidateToken(token) {
		const tokenData = await cacheService.get(`token:active:${token}`);
		if (tokenData) {
			await cacheService.delete(`token:active:${token}`);
			await cacheService.delete(`token:${tokenData.tokenId}`);
		}
		await cacheService.set(`token:blacklist:${token}`, true, BLACKLIST_TTL);
	},

	// Invalidate user session token - params: token
	async logout(token) {
		await this.invalidateToken(token);
		logger.info("User logged out");
		return { success: true };
	},

	// Generate temporary 2FA QR code for setup - params: userId, userIdentifier, clientIP
	async generate2FASecret(userId, userIdentifier, clientIP) {
		const user = await User.findById(userId).select(
			"+twoFactorSecretTemp +twoFactorEnabled",
		);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (user.twoFactorEnabled === true) {
			logger.warn("2FA regenerate attempt blocked", {
				userId: user._id,
				ip: clientIP,
			});
			throw new ApiError(
				403,
				"2FA is already permanently enabled on this account.",
				{
					isPermanent: true,
					alreadyEnabled: true,
					code: "2FA_ALREADY_ENABLED",
				},
			);
		}

		const secret = speakeasy.generateSecret({
			name: `NodeMeta (${userIdentifier})`,
			issuer: "NodeMeta Platform",
			length: 32,
		});

		const qrCodeDataURL = await QRCode.toDataURL(secret.otpauth_url);

		if (user.twoFactorEnabled !== true) {
			user.twoFactorSecretTemp = secret.base32;
			user.twoFactorSecretTempCreatedAt = new Date();
			user.twoFactorSetupInitiatedIP = clientIP;
			await user.save();
		}

		logger.info("2FA secret generated", { userId: user._id });

		return {
			success: true,
			secret: secret.base32,
			qrCode: qrCodeDataURL,
			message:
				"2FA secret generated. Warning: Once verified, 2FA cannot be disabled.",
			expiresIn: 900000,
		};
	},

	// Verify TOTP code and permanently enable 2FA - params: userId, totpCode, ip
	async verify2FASetup(userId, totpCode, ip, username, walletAddress) {
		const user = await User.findById(userId).select(
			"+twoFactorSecretTemp +twoFactorSecretHash +twoFactorEnabled",
		);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (user.twoFactorEnabled === true) {
			logger.warn("2FA re-verify attempt blocked", {
				userId: user._id,
				ip,
			});
			throw new ApiError(
				403,
				"🔒 2FA is already permanently enabled on this account.",
				{
					isPermanent: true,
					alreadyEnabled: true,
					code: "2FA_ALREADY_ENABLED",
				},
			);
		}

		if (!user.twoFactorSecretTemp) {
			throw new ApiError(
				400,
				"2FA not initialized. Please generate a QR code first.",
			);
		}

		if (user.twoFactorSecretTempCreatedAt) {
			const secretAge =
				Date.now() - new Date(user.twoFactorSecretTempCreatedAt).getTime();
			if (secretAge > 15 * 60 * 1000) {
				throw new ApiError(
					400,
					"2FA setup expired. Please generate a new QR code.",
				);
			}
		}

		const decryptedSecret = user.twoFactorSecretTemp;
		const isValid = verify2FACode(
			decryptedSecret,
			totpCode,
			0, // Allow 1 time window tolerance for testing (±30 seconds)
			walletAddress, // User ID for replay prevention
			ip, // IP for brute force protection
			"POST", // Method for audit logging
			username,
		);
		if (!isValid) {
			logger.warn("Invalid 2FA verification code", { userId: user._id });
			throw new ApiError(400, "Invalid verification code. Please try again.");
		}

		const hashedSecret = hash2FASecret(user.twoFactorSecretTemp);

		user.twoFactorSecretHash = hashedSecret;
		user.twoFactorEnabled = true;
		user.twoFactorIsPermanent = true;
		user.twoFactorEnabledAt = new Date();
		user.twoFactorSetupIP = ip;
		user.twoFactorSecretTemp = undefined;
		user.twoFactorSecretTempCreatedAt = undefined;

		await user.save();

		// Send 2FA enabled confirmation email
		try {
			const enabledTime = new Date().toLocaleString("en-US", {
				timeZone: "UTC",
				year: "numeric",
				month: "long",
				day: "numeric",
				hour: "2-digit",
				minute: "2-digit",
				second: "2-digit",
				hour12: false,
			});

			const emailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #10b981; margin-bottom: 20px; font-size: 24px;">🔒 Two-Factor Authentication Enabled</h2>
					
					<p style="margin-bottom: 15px;">Hello <strong>${username}</strong>,</p>
					
					<p style="margin-bottom: 20px;">Two-Factor Authentication (2FA) has been successfully enabled on your NodeMeta account. Your account is now more secure!</p>
					
					<div style="background: linear-gradient(135deg, rgba(16, 185, 129, 0.15), rgba(16, 185, 129, 0.05)); border-left: 4px solid #10b981; padding: 20px; margin: 25px 0; border-radius: 8px;">
						<h3 style="color: #10b981; margin-top: 0; margin-bottom: 15px; font-size: 18px;">2FA Setup Details</h3>
						<table style="width: 100%; border-collapse: collapse;">
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Enabled On:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-weight: 600;">${enabledTime} UTC</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">IP Address:</td>
								<td style="padding: 8px 0; color: #fff; text-align: right; font-family: monospace;">${ip}</td>
							</tr>
							<tr>
								<td style="padding: 8px 0; color: #9aa3b2;">Status:</td>
								<td style="padding: 8px 0; color: #10b981; text-align: right; font-weight: 600;">✓ Active & Permanent</td>
							</tr>
						</table>
					</div>
					
					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>⚠️ Important Information</strong></p>
						<ul style="margin: 10px 0 0 0; padding-left: 20px; color: #e1e4eb;">
							<li style="margin: 8px 0;">2FA is now <strong>permanently enabled</strong> and cannot be disabled through normal means</li>
							<li style="margin: 8px 0;">You will need your authenticator app to log in from now on</li>
							<li style="margin: 8px 0;">Keep your authenticator device secure and backed up</li>
							<li style="margin: 8px 0;">If you lose access to your authenticator, contact support immediately</li>
						</ul>
					</div>
					
					<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #ef4444;"><strong>🚨 Wasn't You?</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">If you didn't enable 2FA, your account may be compromised. Contact our support team immediately.</p>
					</div>
					
					<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #3b82f6;"><strong>💡 Tips for Using 2FA</strong></p>
						<ul style="margin: 10px 0 0 0; padding-left: 20px; color: #e1e4eb;">
							<li style="margin: 8px 0;">Save backup codes in a secure location (if provided by your app)</li>
							<li style="margin: 8px 0;">Consider setting up 2FA on multiple devices</li>
							<li style="margin: 8px 0;">Never share your 2FA codes with anyone</li>
							<li style="margin: 8px 0;">Test your 2FA setup before logging out</li>
						</ul>
					</div>
					
					<div style="text-align: center; margin-top: 30px;">
						<a href="https://node-meta.com/dashboard" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #00d9ff 0%, #0099cc 100%); color: #0a0e1a; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px; margin-right: 10px;">Go to Dashboard</a>
						<a href="https://node-meta.com/dashboard/settings" style="display: inline-block; padding: 14px 32px; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); color: #fff; text-decoration: none; border-radius: 8px; font-weight: 600; font-size: 15px;">Security Settings</a>
					</div>
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(emailBody);

			systemNotificationService.sendToUser({
				username,
				email: user.email,
				subject: "2FA Enabled on Your NodeMeta Account",
				title: "2FA Enabled",
				body: `Two-Factor Authentication has been permanently enabled on your account. Your account is now more secure!`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/settings/security",
					},
				},
				sendEmail: true,
				sendPush: true,
			});
		} catch (notificationError) {
			logger.error("Failed to send 2FA enabled confirmation email:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				username,
			});
		}

		logger.info("2FA enabled successfully", {
			userId: user._id,
			ip,
		});

		return {
			success: true,
			message: "2FA has been permanently enabled!",
			warning:
				"Your authenticator app is now the ONLY way to access your account.",
			isPermanent: true,
			twoFactorEnabled: true,
			twoFactorEnabledAt: user.twoFactorEnabledAt,
		};
	},

	// Reject 2FA disable attempts (permanent security feature) - params: userId, password, totpCode, clientIP
	async disable2FA(userId, password, totpCode, clientIP) {
		logger.warn("2FA disable attempt blocked", {
			userId,
			ip: clientIP,
			timestamp: new Date().toISOString(),
		});

		throw new ApiError(
			403,
			"2FA cannot be disabled once enabled. This is a permanent security feature. Only a nodemeta administrator can manually remove 2FA from the nodemeta.",
			{
				isPermanent: true,
				code: "2FA_DISABLE_NOT_ALLOWED",
				contact:
					"Please contact support if you have lost access to your authenticator.",
			},
		);
	},

	// Reject 2FA reset attempts (permanent security feature) - params: userId, clientIP
	async reset2FA(userId, clientIP) {
		logger.warn("2FA reset attempt blocked", {
			userId,
			ip: clientIP,
			timestamp: new Date().toISOString(),
		});

		throw new ApiError(
			403,
			"2FA cannot be reset once enabled. This is a permanent security feature. Only a nodemeta administrator can manually remove 2FA from the nodemeta.",
			{
				isPermanent: true,
				code: "2FA_RESET_NOT_ALLOWED",
				contact:
					"Please contact support if you have lost access to your authenticator.",
			},
		);
	},

	// Verify TOTP code for testing purposes - params: userId, totpCode
	async test2FA(userId, totpCode, walletAddress, clientIP, username) {
		const user = await User.findById(userId).select(
			"+twoFactorSecretHash +twoFactorEnabled",
		);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (!user.twoFactorEnabled || !user.twoFactorSecretHash) {
			throw new ApiError(400, "2FA is not enabled on this account");
		}

		if (!/^\d{6}$/.test(totpCode)) {
			throw new ApiError(400, "Invalid code format. Must be 6 digits.");
		}
		const decryptedSecret = decrypt2FASecret(user.twoFactorSecretHash);

		const isValid = verify2FACode(
			decryptedSecret,
			totpCode,
			1, // Allow 1 time window tolerance for testing (±30 seconds)
			walletAddress, // User ID for replay prevention
			clientIP, // IP for brute force protection
			"POST", // Method for audit logging
			username,
		);

		if (!isValid) {
			return {
				success: false,
				message:
					"Invalid verification code, already used, or too many attempts from your IP",
				timestamp: new Date().toISOString(),
			};
		}

		return {
			success: true,
			message: "✅ TOTP verification successful!",
			timestamp: new Date().toISOString(),
			codeUsed: totpCode,
		};
	},

	// Request password setup verification code - params: walletAddress, clientIP
	async requestPasswordSetup(walletAddress, clientIP) {
		const bcrypt = require("bcryptjs");
		const { PasswordSetupRequest } = require("./temp-user.model");

		const user = await userService.findByWalletAddressWithPassword(
			walletAddress,
		);
		if (!user) {
			throw new ApiError(404, "User not found", {
				message: "No account found with this wallet address.",
			});
		}

		if (user.password && user.password.length > 0) {
			throw new ApiError(403, "Password already set", {
				message:
					"This account already has a password. You cannot set a new password using this method. If you need to change your password, please use the change password feature or contact support.",
				hasPassword: true,
			});
		}

		const verificationCode = generateVerificationCode(7);

		// Send verification code via email
		try {
			const emailBody = `
				<div style="color: #e1e4eb; line-height: 1.8;">
					<h2 style="color: #3b82f6; margin-bottom: 20px; font-size: 24px;">🔑 Password Setup Verification</h2>
					
					<p style="margin-bottom: 15px;">Hello,</p>
					
					<p style="margin-bottom: 20px;">You've requested to set up a password for your NodeMeta account. Use the verification code below to complete the setup process.</p>
					
					<div style="background: linear-gradient(135deg, rgba(59, 130, 246, 0.15), rgba(59, 130, 246, 0.05)); border-left: 4px solid #3b82f6; padding: 30px; margin: 30px 0; border-radius: 8px; text-align: center;">
						<p style="margin: 0 0 15px 0; color: #9aa3b2; font-size: 14px; text-transform: uppercase; letter-spacing: 1px;">Your Verification Code</p>
						<div style="background: rgba(59, 130, 246, 0.2); padding: 20px; border-radius: 8px; display: inline-block;">
							<p style="margin: 0; color: #fff; font-size: 36px; font-weight: 700; letter-spacing: 8px; font-family: 'Courier New', monospace;">${verificationCode}</p>
						</div>
						<p style="margin: 15px 0 0 0; color: #9aa3b2; font-size: 12px;">Enter this code to continue</p>
					</div>
					
					<div style="background: rgba(234, 179, 8, 0.1); border-left: 4px solid #eab308; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #eab308;"><strong>⏰ Important</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">This verification code will expire in <strong>5 minutes</strong>. If you didn't request this, please ignore this email or contact support if you're concerned about your account security.</p>
					</div>
					
					<div style="background: rgba(59, 130, 246, 0.1); border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #3b82f6;"><strong>📋 Account Details</strong></p>
						<table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
							<tr>
								<td style="padding: 5px 0; color: #9aa3b2;">Wallet Address:</td>
								<td style="padding: 5px 0; color: #fff; text-align: right; font-family: monospace; font-size: 11px; word-break: break-all;">${walletAddress}</td>
							</tr>
							<tr>
								<td style="padding: 5px 0; color: #9aa3b2;">Email:</td>
								<td style="padding: 5px 0; color: #fff; text-align: right;">${user.email}</td>
							</tr>
							<tr>
								<td style="padding: 5px 0; color: #9aa3b2;">Request IP:</td>
								<td style="padding: 5px 0; color: #9aa3b2; text-align: right; font-family: monospace;">${
									clientIP || "N/A"
								}</td>
							</tr>
						</table>
					</div>
					
					<div style="background: rgba(239, 68, 68, 0.1); border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0; border-radius: 8px;">
						<p style="margin: 0; color: #ef4444;"><strong>🔒 Security Notice</strong></p>
						<p style="margin: 10px 0 0 0; color: #e1e4eb;">Never share your verification code with anyone. NodeMeta staff will never ask for your verification code or password.</p>
					</div>
					
				
				</div>
			`;

			const emailHtml = generateEmailFromTemplate(emailBody);

			systemNotificationService.sendToUser({
				username: user.username,
				email: user.email,
				subject: "Password Setup Verification Code - NodeMeta",
				title: "Password Setup",
				body: `Your password setup verification code is: ${verificationCode}. This code expires in 5 minutes.`,
				html: emailHtml,
				pushPayload: {
					icon: "/icons/icon-192x192.png",
					badge: "/icons/badge-72x72.png",
					data: {
						url: "/auth/password-setup",
						walletAddress,
					},
				},
				sendEmail: true,
				sendPush: false,
			});
		} catch (notificationError) {
			logger.error("Failed to send password setup verification email:", {
				errorMessage: notificationError.message,
				errorStack: notificationError.stack,
				walletAddress,
				email: user.email,
			});
		}

		const hashedVerificationCode = await bcrypt.hash(verificationCode, 10);

		const walletHash = createWalletHash(walletAddress);

		await PasswordSetupRequest.deleteMany({ walletHash });
		const setupRequest = new PasswordSetupRequest({
			walletHash,
			walletAddress,
			email: user.email,
			verificationCode: hashedVerificationCode,
			expiresAt: new Date(Date.now() + 5 * 60 * 1000),
		});

		await setupRequest.save();

		logger.info("Password setup code generated", {
			walletAddress,
			email: user.email,
		});

		return {
			success: true,
			message: "Verification code sent to your email",
			email: user.email,
			verificationCode:
				process.env.NODE_ENV === "production" ? undefined : verificationCode,
		};
	},

	// Verify code and set new password - params: walletAddress, verificationCode, newPassword
	async completePasswordSetup(walletAddress, verificationCode, newPassword) {
		const bcrypt = require("bcryptjs");
		const { PasswordSetupRequest } = require("./temp-user.model");

		const walletHash = createWalletHash(walletAddress);

		const setupRequest = await PasswordSetupRequest.findOne({ walletHash });

		if (!setupRequest) {
			throw new ApiError(404, "No password setup request found", {
				message: "Please request a verification code first.",
			});
		}
		if (new Date(setupRequest.expiresAt) < new Date()) {
			await PasswordSetupRequest.deleteOne({ walletHash });
			throw new ApiError(400, "Verification code has expired", {
				message: "The code has expired. Please request a new one.",
			});
		}

		const cleanCode = String(verificationCode).trim();
		const isCodeValid = await bcrypt.compare(
			cleanCode,
			setupRequest.verificationCode,
		);

		if (!isCodeValid) {
			throw new ApiError(400, "Invalid verification code", {
				message:
					"The code you entered is incorrect. Please check and try again.",
			});
		}

		const user = await userService.findByWalletAddressWithPassword(
			walletAddress,
		);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		if (user.password && user.password.length > 0) {
			throw new ApiError(403, "Password already set", {
				message: "This account already has a password.",
			});
		}

		if (newPassword.length < 8) {
			throw new ApiError(400, "Password too weak", {
				message: "Password must be at least 8 characters long.",
			});
		}

		// hash and set new password

		const hashedPassword = await bcrypt.hash(newPassword, 10);

		const userDoc = await User.findById(user._id);
		userDoc.password = hashedPassword;
		await userDoc.save();

		await PasswordSetupRequest.deleteOne({ walletHash });

		logger.info("Password set successfully", {
			userId: user._id,
		});

		return {
			success: true,
			message:
				"Password set successfully. You can now login with your password.",
		};
	},

	// Return token cache stats - params: none
	getTokenStats() {
		const cacheStats = cacheService.getStats();
		return {
			totalCachedItems: cacheStats.size,
			note: "Token counts included in total cache size",
		};
	},
};

module.exports = { authService };
