const mongoose = require("mongoose");
const bcrypt = require("bcryptjs");

const tempUserSchema = new mongoose.Schema(
	{
		username: { type: String, required: true, lowercase: true, trim: true },
		email: { type: String, required: true, lowercase: true, trim: true },
		password: { type: String, required: true },
		walletAddress: { type: String, required: true },
		walletHash: { type: String, required: true },
		fullName: { type: String, default: "" },
		referralCode: { type: String, required: true, default: null },
		bio: { type: String, default: "" },
		verificationCode: { type: String, required: true },
		expiresAt: { type: Date, required: true },
	},
	{
		timestamps: true,
	},
);

// Hash password before saving temp user
tempUserSchema.pre("save", async function (next) {
	if (!this.isModified("password")) return next();

	try {
		const salt = await bcrypt.genSalt(12);
		this.password = await bcrypt.hash(this.password, salt);
		next();
	} catch (error) {
		next(error);
	}
});

// Auto-remove expired temp users
tempUserSchema.index({ expiresAt: 1 }, { expireAfterSeconds: 0 });
tempUserSchema.index({ email: 1 });
tempUserSchema.index({ walletHash: 1 });

const TempUser = mongoose.model("TempUser", tempUserSchema);

const passwordSetupRequestSchema = new mongoose.Schema(
	{
		walletHash: { type: String, required: true, trim: true },
		walletAddress: { type: String, required: true, trim: true },
		email: { type: String, required: true, trim: true },
		verificationCode: { type: String, required: true, trim: true },
		expiresAt: { type: Date, required: true },
	},
	{
		timestamps: true,
		collection: "password_setup_requests",
	},
);

passwordSetupRequestSchema.index({ expiresAt: 1 }, { expireAfterSeconds: 0 });
passwordSetupRequestSchema.index({ walletHash: 1 });

const PasswordSetupRequest = mongoose.model(
	"PasswordSetupRequest",
	passwordSetupRequestSchema,
);

module.exports = { TempUser, PasswordSetupRequest };
