const { userService } = require("./user.service");
const { authService } = require("./auth.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");
const UAParser = require("ua-parser-js");
const crypto = require("crypto");

// Generate a consistent device fingerprint from request
const generateDeviceFingerprint = (req) => {
	const components = [
		req.headers["user-agent"] || "",
		req.headers["accept-language"] || "",
		req.headers["accept-encoding"] || "",
	].join("|");
	
	return crypto.createHash("md5").update(components).digest("hex").substring(0, 16);
};

// Extract device info from request headers - params: req
const getDeviceInfo = (req) => {
	const userAgent = req.headers["user-agent"] || "";
	const parser = new UAParser();
	parser.setUA(userAgent);

	const browser = parser.getBrowser();
	const os = parser.getOS();
	const device = parser.getDevice();

	// Get client IP with proper fallback chain
	const clientIp = req.clientIp || req.ip || req.connection?.remoteAddress || "unknown";

	// Generate device ID: prefer header, then fingerprint-based (consistent), fallback to random
	let deviceId = req.headers["x-device-id"];
	if (!deviceId) {
		// Create consistent fingerprint from request characteristics
		const fingerprint = generateDeviceFingerprint(req);
		deviceId = `${fingerprint}-${clientIp}`;
	}

	// Better label detection
	let label;
	if (browser.name && os.name) {
		label = `${browser.name} on ${os.name}`;
	} else if (userAgent.toLowerCase().includes("postman")) {
		label = "Postman API Client";
	} else if (userAgent.toLowerCase() === "node" || userAgent.toLowerCase().includes("node-fetch")) {
		label = "Node.js Client";
	} else if (userAgent.toLowerCase().includes("curl")) {
		label = "cURL Client";
	} else if (userAgent.toLowerCase().includes("insomnia")) {
		label = "Insomnia API Client";
	} else if (userAgent) {
		// Try to extract something useful from user agent
		label = userAgent.substring(0, 50);
	} else {
		label = "Unknown Device";
	}

	return {
		deviceId,
		deviceType: device.type || (device.vendor ? "mobile" : "desktop"),
		ip: clientIp,
		label,
		userAgent: userAgent || "unknown",
		tokenId: "",
	};
};

const userController = {
	// Handle two-step signup: verification request or completion - params: req.body (action, userData)
	async signup(req, res, next) {
		try {
			const deviceInfo = getDeviceInfo(req);
			const action = req.body?.action;

			if (action === "verify") {
				const result = await authService.signup(req.body, deviceInfo);
				return res.status(201).json(
					new ApiResponse(
						201,
						{
							success: true,
							user: result.user,
						},
						"Account created successfully",
					),
				);
			}

			const result = await authService.signup(req.body, deviceInfo);
			res
				.status(200)
				.json(
					new ApiResponse(200, result, "Verification code sent to your email"),
				);
		} catch (err) {
			next(err);
		}
	},

	// Authenticate user with password and 2FA - params: req.body (walletAddress, password, totpCode?)
	async login(req, res, next) {
		try {
			const { walletAddress, password, totpCode } = req.body;

			if (!walletAddress || !password) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Wallet address and password are required",
						),
					);
			}

			if (!password) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Password is required"));
			}

			const deviceInfo = getDeviceInfo(req);

			const result = await authService.login(
				walletAddress,
				password,
				totpCode,
				deviceInfo,
				deviceInfo.ip,
			);

			res.json(new ApiResponse(200, result, "Login successful"));
		} catch (err) {
			next(err);
		}
	},

	// Invalidate user session token - params: req.headers.authorization or req.body.token
	async logout(req, res, next) {
		try {
			const headerToken = req.headers.authorization?.split(" ")[1];
			const bodyToken = req.body?.token;
			const token = headerToken || bodyToken;

			if (!token) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Token required"));
			}

			await authService.logout(token);
			res.json(
				new ApiResponse(200, { success: true }, "Logged out successfully"),
			);
		} catch (err) {
			next(err);
		}
	},

	// Generate JWT token for wallet address - params: req.body (walletAddress)
	async generateToken(req, res, next) {
		try {
			const { walletAddress } = req.body;

			if (!walletAddress) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Wallet address required"));
			}

			const user = await userService.findByWalletAddress(walletAddress);
			if (!user) {
				return res
					.status(404)
					.json(new ApiResponse(404, null, "User not found or blocked"));
			}

			const deviceInfo = getDeviceInfo(req);

			// SINGLE DEVICE LOGIN: Invalidate ALL other device tokens
			if (user.devices && user.devices.length > 0) {
				for (const device of user.devices) {
					if (device.tokenId) {
						await authService.invalidateTokenById(device.tokenId);
					}
				}
			}

			const token = authService.generateToken(user);
			const tokenId = authService.registerToken(token, user._id.toString(), user.username, deviceInfo);

			// Update device with new tokenId
			await userService.updateLastLogin(user._id, { ...deviceInfo, tokenId });

			res.json(
				new ApiResponse(
					200,
					{
						success: true,
						token,
						loginTimestamp: Date.now(),
						user: {
							username: user.username,
							email: user.email,
							role: user.role,
							walletHash: user.walletHash,
						},
					},
					"Token generated",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Validate JWT token and return decoded payload - params: req.body (token)
	async verifyToken(req, res, next) {
		try {
			const { token } = req.body;

			if (!token) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Token required"));
			}

			const decoded = await authService.verifyToken(token);
			res.json(
				new ApiResponse(200, { valid: true, user: decoded }, "Token is valid"),
			);
		} catch (err) {
			if (err instanceof ApiError && err.statusCode === 401) {
				return res
					.status(401)
					.json(new ApiResponse(401, { valid: false }, err.message));
			}
			next(err);
		}
	},

	// Check if user exists and has password/2FA - params: req.query (address)
	async checkUser(req, res, next) {
		try {
			const { address } = req.query;

			if (!address) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Address required"));
			}

			const user = await userService.findByWalletAddressWithPassword(address);

			if (!user) {
				return res.json(new ApiResponse(200, { exists: false, user: null }));
			}

			res.json(
				new ApiResponse(200, {
					exists: true,
					user: {
						hasPassword: !!user.password,
						twoFactorEnabled: !!user.twoFactorEnabled,
					},
				}),
			);
		} catch (err) {
			next(err);
		}
	},

	// Check user existence and authentication status - params: req.query (address)
	async checkLogin(req, res, next) {
		try {
			const { address } = req.query;

			if (!address) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Address required"));
			}

			const user = await userService.findByWalletAddressWithPassword(address);

			res.json(
				new ApiResponse(200, {
					exists: !!user,
					hasPassword: user ? !!user.password : false,
					twoFactorEnabled: user ? !!user.twoFactorEnabled : false,
				}),
			);
		} catch (err) {
			next(err);
		}
	},

	// Update user profile with validation for protected fields - params: req.body (fullName, bio, notifications)
	async updateProfile(req, res, next) {
		try {
			const {
				walletAddress,
				fullName,
				username,
				email,
				referralCode,
				bio,
				notifications,
			} = req.body;
			const userId = req.user._id;

			if (!walletAddress) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Wallet address is required"));
			}

			const existingUser = await userService.findById(userId);

			if (
				username &&
				username.toLowerCase().trim() !== existingUser.username.toLowerCase()
			) {
				logger.warn("Username update attempt blocked", {
					userId,
					oldUsername: existingUser.username,
					attemptedUsername: username,
				});
				return res.status(403).json(
					new ApiResponse(
						403,
						{
							code: "USERNAME_UPDATE_FORBIDDEN",
						},
						"Username cannot be changed after account creation. This field is permanent.",
					),
				);
			}

			if (
				email &&
				email.toLowerCase().trim() !== existingUser.email.toLowerCase()
			) {
				logger.warn("Email update attempt blocked", {
					userId,
					oldEmail: existingUser.email,
					attemptedEmail: email,
				});
				return res.status(403).json(
					new ApiResponse(
						403,
						{
							code: "EMAIL_UPDATE_FORBIDDEN",
						},
						"Email cannot be changed after account creation. Please contact support if you need to update your email.",
					),
				);
			}

			if (
				referralCode &&
				referralCode.toLowerCase().trim() !==
					(existingUser.referralCode || "").toLowerCase()
			) {
				logger.warn("Referral code update attempt blocked", {
					userId,
					oldReferralCode: existingUser.referralCode,
					attemptedReferralCode: referralCode,
				});
				return res.status(403).json(
					new ApiResponse(
						403,
						{
							code: "REFERRAL_CODE_UPDATE_FORBIDDEN",
						},
						"Referral code cannot be changed after account creation. This field is permanent.",
					),
				);
			}

			const protectedFields = [
				"twoFactorEnabled",
				"twoFactorSecretHash",
				"twoFactorSecretTemp",
				"twoFactorQRCode",
				"twoFactorEnabledAt",
				"twoFactorBackupCodes",
			];

			const attemptedProtectedFields = protectedFields.filter((field) =>
				req.body.hasOwnProperty(field),
			);

			if (attemptedProtectedFields.length > 0) {
				logger.warn("Protected fields update attempt blocked", {
					userId,
					attemptedFields: attemptedProtectedFields,
				});
				return res.status(403).json(
					new ApiResponse(
						403,
						{
							code: "PROTECTED_FIELDS",
							attemptedFields: attemptedProtectedFields,
						},
						"Cannot modify protected fields via profile update. These fields are permanently protected.",
					),
				);
			}

			const user = await userService.updateProfile(userId, {
				fullName,
				bio,
				notifications,
			});

			res.json(new ApiResponse(200, user, "Profile updated successfully"));
		} catch (err) {
			next(err);
		}
	},

	// Two-step password setup: request or verify - params: req.body (walletAddress, action?, verificationCode?, newPassword?)
	async setPassword(req, res, next) {
		try {
			const { walletAddress, action, verificationCode, newPassword } = req.body;

			if (action === "verify") {
				if (!walletAddress || !verificationCode || !newPassword) {
					return res
						.status(400)
						.json(
							new ApiResponse(
								400,
								null,
								"Wallet address, verification code, and new password are required",
							),
						);
				}

				const result = await authService.completePasswordSetup(
					walletAddress,
					verificationCode,
					newPassword,
				);
				res.json(new ApiResponse(200, result));
			} else {
				if (!walletAddress) {
					return res
						.status(400)
						.json(new ApiResponse(400, null, "Wallet address is required"));
				}

				const deviceInfo = getDeviceInfo(req);
				const result = await authService.requestPasswordSetup(
					walletAddress,
					deviceInfo.ip,
				);
				res.json(new ApiResponse(200, result));
			}
		} catch (err) {
			next(err);
		}
	},

	// Get active and blacklisted token statistics - params: none
	async getTokenStats(req, res, next) {
		try {
			const stats = authService.getTokenStats();
			res.json(new ApiResponse(200, stats));
		} catch (err) {
			next(err);
		}
	},

	// Generate 2FA QR code for authenticator app - params: req.body (walletAddress)
	async generate2FA(req, res, next) {
		try {
			const userId = req.user._id;
			const user = await userService.findById(userId);
			const { walletAddress } = req.body;

			if (!walletAddress) {
				return res
					.status(400)
					.json(new ApiResponse(400, null, "Wallet address is required"));
			}

			const deviceInfo = getDeviceInfo(req);
			const result = await authService.generate2FASecret(
				userId,
				user.email || user.username,
				deviceInfo.ip,
			);

			res.json(new ApiResponse(200, result));
		} catch (err) {
			next(err);
		}
	},

	// Verify and permanently enable 2FA - params: req.body (code, walletAddress)
	async verify2FA(req, res, next) {
		try {
			const { code, walletAddress } = req.body;
			const userId = req.user._id;

			const deviceInfo = getDeviceInfo(req);
			const username = req.user.username;

			if (!walletAddress || !code) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Wallet address and verification code are required",
						),
					);
			}

			if (!/^\d{6}$/.test(code)) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Invalid code format. Must be 6 digits.",
						),
					);
			}

			const result = await authService.verify2FASetup(
				userId,
				code,
				deviceInfo.ip,
				username,
				walletAddress,
			);
			res.json(new ApiResponse(200, result));
		} catch (err) {
			next(err);
		}
	},

	// Reject 2FA disable attempts (permanent security) - params: none
	async disable2FA(req, res, next) {
		try {
			const deviceInfo = getDeviceInfo(req);
			const userId = req.user._id;
			await authService.disable2FA(userId, null, null, deviceInfo.ip);
		} catch (err) {
			next(err);
		}
	},

	// Reject 2FA reset attempts (permanent security) - params: none
	async reset2FA(req, res, next) {
		try {
			const deviceInfo = getDeviceInfo(req);
			const userId = req.user._id;
			await authService.reset2FA(userId, deviceInfo.ip);
		} catch (err) {
			next(err);
		}
	},

	// Test 2FA code for validation - params: req.body (code, walletAddress)
	async test2FA(req, res, next) {
		try {
			const { code, walletAddress } = req.body;
			const userId = req.user._id;

			const deviceInfo = getDeviceInfo(req);
			const username = req.user.username;

			if (!walletAddress || !code) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Wallet address and verification code are required",
						),
					);
			}

			if (!/^\d{6}$/.test(code)) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Invalid code format. Must be 6 digits.",
						),
					);
			}

			const result = await authService.test2FA(
				userId,
				code,
				walletAddress,
				deviceInfo.ip,
				username,
			);
			res.json(new ApiResponse(200, result));
		} catch (err) {
			next(err);
		}
	},

	// List users with filtering and pagination - params: req.query (status?, role?, search?, limit?, skip?)
	async listUsers(req, res, next) {
		try {
			const { status, role, search, limit, skip } = req.query;
			const result = await userService.list({
				status,
				role,
				search,
				limit,
				skip,
			});
			res.json(new ApiResponse(200, result));
		} catch (err) {
			next(err);
		}
	},

	// Get user by ID - params: req.params (id)
	async getUser(req, res, next) {
		try {
			const user = await userService.findById(req.params.id);
			res.json(new ApiResponse(200, user.toSafeObject()));
		} catch (err) {
			next(err);
		}
	},
	async getUserProfile(req, res, next) {
		try {
			const user = await userService.findByWalletAddress(
				req.user.walletAddress,
			);
			res.json(new ApiResponse(200, user.toSafeObject()));
		} catch (err) {
			next(err);
		}
	},

	// Block user account - params: req.params (id), req.body (reason)
	async blockUser(req, res, next) {
		try {
			const { reason } = req.body;
			const user = await userService.blockUser(
				req.params.id,
				reason,
				req.user.username,
			);
			res.json(new ApiResponse(200, user, "User blocked"));
		} catch (err) {
			next(err);
		}
	},

	// Unblock user account - params: req.params (id)
	async unblockUser(req, res, next) {
		try {
			const user = await userService.unblockUser(req.params.id);
			res.json(new ApiResponse(200, user, "User unblocked"));
		} catch (err) {
			next(err);
		}
	},

	// Remove device from user account - params: req.params (id, deviceId)
	async removeDevice(req, res, next) {
		try {
			const user = await userService.removeDevice(
				req.params.id,
				req.params.deviceId,
			);
			res.json(new ApiResponse(200, user, "Device removed"));
		} catch (err) {
			next(err);
		}
	},

	// Get referral statistics with packages - params: req.query (walletAddress?, username?, timePeriod?, maxLevels?)
	async getReferralStats(req, res, next) {
		try {
			const {
				walletAddress,
				username,
				timePeriod = "all",
				maxLevels,
			} = req.query;

			const parsedMaxLevels = maxLevels ? parseInt(maxLevels) : Infinity;

			if (!walletAddress && !username) {
				return res
					.status(400)
					.json(
						new ApiResponse(
							400,
							null,
							"Wallet address or username is required",
						),
					);
			}

			let user;
			if (walletAddress) {
				user = await userService.findByWalletAddress(walletAddress);
			} else if (username) {
				user = await userService.findByUsername(username);
			}

			if (!user) {
				return res
					.status(404)
					.json(new ApiResponse(404, null, "User not found"));
			}

			const { getUserReferralsWithPackagesFast } = require("./user.service");
			const referralData = await getUserReferralsWithPackagesFast(
				user.username,
				timePeriod,
				parsedMaxLevels,
			);

			const stats = {
				totalReferrals: referralData.totalReferrals || 0,
				activeReferrals: referralData.activeReferrals || 0,
				referralCode: user.username,
				referredUsers: referralData.referredUsers || [],
				packageStats: referralData.packageStats || {},
				timePeriod: timePeriod,
				queryMaxLevels: parsedMaxLevels,
			};

			res.json(
				new ApiResponse(
					200,
					stats,
					"Referral statistics retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Get user network (upline + downline)
	async getUserNetwork(req, res, next) {
		try {
			const { username } = req.query;

			if (!username) {
				throw new ApiError(400, "Username is required");
			}

			const user = await userService.findByUsername(username);
			if (!user) {
				throw new ApiError(404, "User not found");
			}

			const { getUserUpline, getUserDownline } = require("./user.service");

			const [upline, downline] = await Promise.all([
				getUserUpline(user.username),
				getUserDownline(user.username),
			]);

			res.json(
				new ApiResponse(
					200,
					{
						user: {
							username: user.username,
							fullName: user.fullName,
							referralCode: user.referralCode,
							isActive: user.isActive,
						},
						upline: {
							total: upline.length,
							users: upline,
						},
						downline: {
							total: downline.length,
							users: downline,
						},
					},
					"User network retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { userController };
