const mongoose = require("mongoose");
const bcrypt = require("bcryptjs");

const deviceSchema = new mongoose.Schema(
	{
		deviceId: { type: String, required: true },
		deviceType: { type: String, required: true },
		ip: { type: String, required: true },
		label: { type: String, required: true },
		userAgent: { type: String, required: true },
		tokenId: { type: String, default: null },
		lastUsed: { type: Date, required: true, default: Date.now },
		createdAt: { type: Date, required: true, default: Date.now },
	},
	{ _id: false },
);

const rankHistorySchema = new mongoose.Schema(
	{
		level: { type: Number, required: true },
		achievedAt: { type: Date, required: true },
		endedAt: { type: Date, default: null },
	},
	{ _id: false },
);

const notificationSettingsSchema = new mongoose.Schema(
	{
		email: { type: Boolean, required: true, default: true },
		browser: { type: Boolean, required: true, default: true },
	},
	{ _id: false },
);

const userSchema = new mongoose.Schema(
	{
		username: {
			type: String,
			required: true,
			unique: true,
			lowercase: true,
			trim: true,
			index: true,
		},
		email: {
			type: String,
			required: true,
			unique: true,
			lowercase: true,
			trim: true,
			index: true,
		},
		password: {
			type: String,
			required: true,
			select: false,
		},
		walletHash: {
			type: String,
			required: true,
			unique: true,
			index: true,
		},

		fullName: { type: String, default: null },
		bio: { type: String, default: null },
		role: {
			type: String,
			required: true,
			enum: ["user", "admin", "support"],
			default: "user",
			index: true,
		},

		isActive: { type: Boolean, required: true, default: true },
		isBlocked: { type: Boolean, default: false, index: true },
		blockedAt: { type: Date },
		blockedBy: { type: String },
		blockReason: { type: String },

		lastLogin: { type: Date, required: true, default: Date.now },
		loginCount: { type: Number, required: true, default: 0 },

		devices: [deviceSchema],

		twoFactorEnabled: { type: Boolean, default: false },
		twoFactorEnabledAt: { type: Date, default: null },
		twoFactorIsPermanent: { type: Boolean, default: false },
		twoFactorSecretHash: { type: String, select: false },
		twoFactorSecretTemp: { type: String, select: false },
		twoFactorSecretTempCreatedAt: { type: Date },
		twoFactorSetupInitiatedIP: { type: String },
		twoFactorSetupIP: { type: String },

		notifications: {
			type: notificationSettingsSchema,
			required: true,
			default: () => ({ email: true, browser: true }),
		},

		referralCode: { type: String, default: null },

		currentRankLevel: { type: Number, default: 0 },
		rankAchievedAt: { type: Date, default: null },
		rankHistory: [rankHistorySchema],

		walletBalance: { type: Number, default: 0 },
		walletBalanceUpdatedAt: { type: Date },
		walletBalanceUpdatedBy: { type: String },
		withdrawalInProgress: { type: Boolean, default: false },
		withdrawalsDisabled: { type: Boolean, default: false },
		withdrawalsDisabledAt: { type: Date },
		withdrawalsDisabledBy: { type: String },
		withdrawalDisableReason: { type: String },
		lastWithdrawalAt: { type: Date },
		lastRefundAt: { type: Date },

		// Balance locking for concurrent withdrawal prevention
		balanceLocked: { type: Boolean, default: false, index: true },
		balanceLockedAt: { type: Date },
		balanceLockedBy: { type: String },
		balanceLockedAmount: { type: Number, default: 0 },

		balanceSnapshot: { type: Number }, // Balance at lock time
		balanceChecksum: { type: String }, // Cryptographic verification
		balanceLockNonce: { type: String }, // Unique lock identifier
	},
	{
		timestamps: true,
		collection: "users",
	},
);

userSchema.index({ createdAt: -1 });
userSchema.index({ lastLogin: -1 });
userSchema.index({ isActive: 1, isBlocked: 1 });
userSchema.index({ "devices.tokenId": 1 });
userSchema.index({ referralCode: 1 });

userSchema.methods.comparePassword = async function (candidatePassword) {
	return bcrypt.compare(candidatePassword, this.password);
};

userSchema.methods.has2FA = function () {
	return this.twoFactorEnabled === true && this.twoFactorIsPermanent === true;
};

userSchema.methods.addDevice = function (deviceInfo) {
	const existingDeviceIndex = this.devices.findIndex(
		(d) => d.deviceId === deviceInfo.deviceId,
	);

	if (existingDeviceIndex >= 0) {
		this.devices[existingDeviceIndex] = {
			...this.devices[existingDeviceIndex].toObject(),
			...deviceInfo,
			lastUsed: new Date(),
		};
	} else {
		this.devices.push({
			deviceId: deviceInfo.deviceId,
			deviceType: deviceInfo.deviceType || "unknown",
			ip: deviceInfo.ip || "unknown",
			label: deviceInfo.label || "unknown",
			userAgent: deviceInfo.userAgent || "unknown",
			tokenId: deviceInfo.tokenId || null,
			createdAt: new Date(),
			lastUsed: new Date(),
		});
	}

	// Keep only the last 10 devices (sorted by lastUsed, most recent first)
	if (this.devices.length > 10) {
		this.devices = this.devices
			.sort((a, b) => new Date(b.lastUsed) - new Date(a.lastUsed))
			.slice(0, 10);
	}
};

userSchema.methods.removeDevice = function (deviceId) {
	this.devices = this.devices.filter((d) => d.deviceId !== deviceId);
};

userSchema.methods.toSafeObject = function () {
	const obj = this.toObject();
	delete obj.password;
	delete obj.twoFactorSecretHash;
	delete obj.twoFactorSecretTemp;
	delete obj.__v;
	return obj;
};

const User = mongoose.model("User", userSchema);

module.exports = { User };
