const { User } = require("./user.model");
const { TempUser } = require("./temp-user.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const crypto = require("crypto");
const { validators } = require("../../utils/validators");

function hashWalletAddress(
	walletAddress,
	salt = process.env.WALLET_HASH_SALT || "nodemeta-default-salt",
) {
	if (!walletAddress || typeof walletAddress !== "string") {
		throw new Error("Invalid wallet address");
	}

	const normalizedAddress = walletAddress.toLowerCase().trim();

	const hash = crypto.createHash("sha256");
	hash.update(normalizedAddress + salt);

	return hash.digest("hex");
}

function createWalletHash(walletAddress) {
	return hashWalletAddress(
		walletAddress,
		process.env.WALLET_HASH_SALT || "nodemeta-lookup-salt",
	);
}

const userService = {
	// Create new user after signup verification - params: {username, email, password, walletAddress, fullName?, referralCode?, bio?}
	async create(userData) {
		const {
			username,
			email,
			password,
			walletAddress,
			fullName,
			referralCode,
			bio,
		} = userData;

		if (!username || !email || !password || !walletAddress) {
			throw new ApiError(400, "Missing required fields");
		}

		if (!validators.isValidEmail(email)) {
			throw new ApiError(400, "Invalid email format");
		}

		const walletHash = createWalletHash(walletAddress);
		const existing = await User.findOne({
			$or: [
				{ email: email.toLowerCase() },
				{ username: username.toLowerCase() },
				{ walletHash },
			],
		});

		if (existing) {
			if (existing.email === email.toLowerCase()) {
				throw new ApiError(409, "Email already registered");
			}
			if (existing.username === username.toLowerCase()) {
				throw new ApiError(409, "Username already taken");
			}
			if (existing.walletHash === walletHash) {
				throw new ApiError(409, "Wallet address already registered");
			}
		}

		if (referralCode) {
			const referrer = await User.findOne({
				username: referralCode.toLowerCase(),
			});
			if (!referrer) {
				throw new ApiError(400, "Invalid referral code");
			}
		}

		const user = new User({
			username: username.toLowerCase().trim(),
			email: email.toLowerCase().trim(),
			password,
			walletHash,
			fullName: fullName || username,
			referralCode: referralCode ? referralCode.toUpperCase() : null,
			bio: bio || null,
			role: "user",
			isActive: true,
			loginCount: 0,
			lastLogin: new Date(),
		});

		await user.save();
		logger.info("User created", { userId: user._id, username: user.username });

		return user.toSafeObject();
	},

	// Store signup data temporarily before email verification - params: {username, email, password, walletAddress, verificationCode, expiresInMinutes?}
	async createTempUser(tempData) {
		const {
			username,
			email,
			password,
			walletAddress,
			fullName,
			referralCode,
			bio,
			verificationCode,
			expiresInMinutes = 10,
		} = tempData;

		if (
			!username ||
			!email ||
			!password ||
			!walletAddress ||
			!verificationCode
		) {
			throw new ApiError(400, "Missing required fields for temporary user");
		}

		const walletHash = createWalletHash(walletAddress);

		await TempUser.deleteMany({
			$or: [{ email: email.toLowerCase() }, { walletHash }],
		});

		const tempUser = new TempUser({
			username: username.toLowerCase().trim(),
			email: email.toLowerCase().trim(),
			password,
			walletAddress,
			walletHash,
			fullName: fullName || username,
			referralCode: referralCode ? referralCode.toUpperCase() : null,
			bio: bio || "",
			verificationCode,
			expiresAt: new Date(Date.now() + expiresInMinutes * 60 * 1000),
		});

		await tempUser.save();
		return tempUser;
	},

	// Get temp user by email - params: email
	async getTempUserByEmail(email) {
		if (!email) return null;
		return TempUser.findOne({ email: email.toLowerCase() });
	},

	// Delete temp user after verification or expiry - params: email
	async deleteTempUser(email) {
		if (!email) return;
		await TempUser.deleteMany({ email: email.toLowerCase() });
	},

	// Check if referral code belongs to active user - params: code
	async isValidReferralCode(code) {
		if (!code) return false;
		const referrer = await User.findOne({
			username: code.toLowerCase(),
			isBlocked: { $ne: true },
			isActive: true,
		});
		return !!referrer;
	},

	// Get user by ID - params: userId
	async findById(userId) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}
		return user;
	},

	// Get user by wallet hash - params: walletHash
	async findByWalletHash(walletHash) {
		const user = await User.findOne({ walletHash, isBlocked: { $ne: true } });
		return user;
	},

	// Get user by wallet address - params: walletAddress
	async findByWalletAddress(walletAddress) {
		const walletHash = createWalletHash(walletAddress);
		return await this.findByWalletHash(walletHash);
	},

	// Get user with password field for authentication - params: walletAddress
	async findByWalletAddressWithPassword(walletAddress) {
		const walletHash = createWalletHash(walletAddress);
		return await User.findOne({ walletHash, isBlocked: { $ne: true } }).select(
			"+password +twoFactorSecretHash",
		);
	},

	// Get user by email with sensitive fields - params: email
	async findByEmail(email) {
		const user = await User.findOne({
			email: email.toLowerCase(),
			isBlocked: { $ne: true },
		}).select("+password +twoFactorSecretHash");
		return user;
	},

	// Get user by username with sensitive fields - params: username
	async findByUsername(username) {
		const user = await User.findOne({
			username: username.toLowerCase(),
			isBlocked: { $ne: true },
		}).select("+password +twoFactorSecretHash");
		return user;
	},

	// Check if email is already taken - params: email
	async emailExists(email) {
		const count = await User.countDocuments({ email: email.toLowerCase() });
		return count > 0;
	},

	// Check if username is already taken - params: username
	async usernameExists(username) {
		const count = await User.countDocuments({
			username: username.toLowerCase(),
		});
		return count > 0;
	},

	// Check if wallet address is already registered - params: walletAddress
	async walletExists(walletAddress) {
		const walletHash = createWalletHash(walletAddress);
		const count = await User.countDocuments({ walletHash });
		return count > 0;
	},

	// Update login timestamp and device info - params: userId, deviceInfo
	async updateLastLogin(userId, deviceInfo) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.lastLogin = new Date();
		user.loginCount += 1;

		if (deviceInfo) {
			user.addDevice(deviceInfo);
		}

		await user.save();
		return user;
	},

	// Update allowed profile fields only - params: userId, {fullName?, bio?, notifications?}
	async updateProfile(userId, updates) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		const allowedFields = ["fullName", "bio", "notifications"];

		for (const field of allowedFields) {
			if (updates[field] !== undefined) {
				user[field] = updates[field];
			}
		}

		await user.save();
		logger.info("Profile updated", { userId: user._id });

		return user.toSafeObject();
	},

	// Change password with old password verification - params: userId, oldPassword, newPassword
	async updatePassword(userId, oldPassword, newPassword) {
		const user = await User.findById(userId).select("+password");
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		const isMatch = await user.comparePassword(oldPassword);
		if (!isMatch) {
			throw new ApiError(401, "Current password is incorrect");
		}

		user.password = newPassword;
		await user.save();

		logger.info("Password updated", { userId: user._id });
		return { success: true };
	},

	// Set password without old password check - params: userId, newPassword
	async setPassword(userId, newPassword) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.password = newPassword;
		await user.save();

		logger.info("Password set", { userId: user._id });
		return { success: true };
	},

	// Remove device from user's login devices - params: userId, deviceId
	async removeDevice(userId, deviceId) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.removeDevice(deviceId);
		await user.save();

		logger.info("Device removed", { userId: user._id, deviceId });
		return user.toSafeObject();
	},

	// Block user account - params: userId, reason, blockedBy
	async blockUser(userId, reason, blockedBy) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.isBlocked = true;
		user.blockedAt = new Date();
		user.blockReason = reason;
		user.blockedBy = blockedBy;

		await user.save();
		logger.warn("User blocked", { userId: user._id, reason, blockedBy });

		return user.toSafeObject();
	},

	// Unblock user account - params: userId
	async unblockUser(userId) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.isBlocked = false;
		user.blockedAt = null;
		user.blockReason = null;
		user.blockedBy = null;

		await user.save();
		logger.info("User unblocked", { userId: user._id });

		return user.toSafeObject();
	},

	// Get users list with filters and pagination - params: {status?, role?, search?, limit?, skip?}
	async list({ status, role, search, limit = 50, skip = 0 } = {}) {
		const query = {};

		if (status === "active") {
			query.isActive = true;
			query.isBlocked = false;
		} else if (status === "blocked") {
			query.isBlocked = true;
		} else if (status === "inactive") {
			query.isActive = false;
		}

		if (role) {
			query.role = role;
		}

		if (search) {
			query.$or = [
				{ username: new RegExp(search, "i") },
				{ email: new RegExp(search, "i") },
				{ fullName: new RegExp(search, "i") },
			];
		}

		const pagination = validators.parsePagination(limit, skip);

		const [items, total] = await Promise.all([
			User.find(query)
				.sort({ createdAt: -1 })
				.skip(pagination.skip)
				.limit(pagination.limit),
			User.countDocuments(query),
		]);

		return {
			items: items.map((u) => u.toSafeObject()),
			total,
			...pagination,
		};
	},

	// Update user's wallet balance - params: userId, amount, updatedBy
	async updateWalletBalance(userId, amount, updatedBy) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.walletBalance = amount;
		user.walletBalanceUpdatedAt = new Date();
		user.walletBalanceUpdatedBy = updatedBy;

		await user.save();
		logger.info("Wallet balance updated", {
			userId: user._id,
			amount,
			updatedBy,
		});

		return user.toSafeObject();
	},

	// Block user from making withdrawals - params: userId, reason, disabledBy
	async disableWithdrawals(userId, reason, disabledBy) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.withdrawalsDisabled = true;
		user.withdrawalsDisabledAt = new Date();
		user.withdrawalDisableReason = reason;
		user.withdrawalsDisabledBy = disabledBy;

		await user.save();
		logger.warn("Withdrawals disabled", {
			userId: user._id,
			reason,
			disabledBy,
		});

		return user.toSafeObject();
	},

	// Allow user to make withdrawals again - params: userId
	async enableWithdrawals(userId) {
		const user = await User.findById(userId);
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		user.withdrawalsDisabled = false;
		user.withdrawalsDisabledAt = null;
		user.withdrawalDisableReason = null;
		user.withdrawalsDisabledBy = null;

		await user.save();
		logger.info("Withdrawals enabled", { userId: user._id });

		return user.toSafeObject();
	},
};

// Get user upline referrers
async function getUserUpline(username) {
	try {
		const user = await User.findOne({ username: new RegExp(`^${username}$`, "i") });
		if (!user) {
			return [];
		}

		const result = await User.aggregate([
			{ $match: { username: user.username } },
			{
				$graphLookup: {
					from: "users",
					startWith: "$referralCode",
					connectFromField: "referralCode",
					connectToField: "username",
					as: "upline",
					depthField: "level",
					maxDepth: 9,
				},
			},
			{
				$project: {
					upline: 1,
				},
			},
		]);

		if (!result.length) {
			return [];
		}

		// Map and format the upline array - SORT BY LEVEL TO GET CORRECT ORDER
		const formattedUpline = result[0]?.upline
			?.sort((a, b) => a.level - b.level) // Sort by level ascending (1, 2, 3, 4...)
			?.map((ref, idx) => ({
				username: ref.username,
				fullName: ref.fullName,
				level: ref.level + 1, // level is 0-based
				referredUsername:
					idx === 0 ? username : result[0].upline[idx - 1].username,
				createdAt: ref.createdAt,
				isActive: ref.isActive,
			}));

		return formattedUpline || [];
	} catch (error) {
		logger.error("Error getting user upline:", error);
		return [];
	}
}

// Get user downline (all referrals recursively)
async function getUserDownline(username) {
	try {
		const user = await User.findOne({ username: new RegExp(`^${username}$`, "i") });
		if (!user) {
			return [];
		}

		const result = await User.aggregate([
			{ $match: { username: user.username } },
			{
				$graphLookup: {
					from: "users",
					startWith: "$username",
					connectFromField: "username",
					connectToField: "referralCode",
					as: "downline",
					depthField: "level",
					maxDepth: 9,
				},
			},
			{
				$project: {
					downline: 1,
				},
			},
		]);

		if (!result.length) {
			return [];
		}

		const formattedDownline = result[0]?.downline
			?.sort((a, b) => a.level - b.level)
			?.map((ref) => ({
				username: ref.username,
				fullName: ref.fullName,
				level: ref.level + 1,
				referrerUsername: ref.referralCode,
				createdAt: ref.createdAt,
				isActive: ref.isActive,
			}));

		return formattedDownline || [];
	} catch (error) {
		logger.error("Error getting user downline:", error);
		return [];
	}
}

// Get user referrals with packages (fast) - params: username, timePeriod?, maxLevels?
async function getUserReferralsWithPackagesFast(
	username,
	timePeriod = "all",
	maxLevels = Infinity,
) {
	try {
		// Build date filter
		let dateFilter = {};
		if (timePeriod !== "all") {
			const now = new Date();
			const cutoffDate = new Date();

			switch (timePeriod) {
				case "7d":
					cutoffDate.setDate(now.getDate() - 7);
					break;
				case "30d":
					cutoffDate.setDate(now.getDate() - 30);
					break;
				case "90d":
					cutoffDate.setDate(now.getDate() - 90);
					break;
			}

			if (["7d", "30d", "90d"].includes(timePeriod)) {
				dateFilter.createdAt = { $gte: cutoffDate };
			}
		}

		const queue = [{ username: username.toLowerCase(), level: 0 }];
		const visitedUsernames = new Set();
		const allReferredUsersDetails = [];
		const activeReferredUsernames = new Set();

		visitedUsernames.add(username.toLowerCase());

		while (queue.length > 0) {
			const { username: currentUsername, level: currentLevel } = queue.shift();

			if (currentLevel >= maxLevels && maxLevels !== Infinity) continue;

			const directReferrals = await User.aggregate([
				{
					$match: {
						referralCode: { $regex: new RegExp(`^${currentUsername}$`, "i") },
						...dateFilter,
					},
				},
				{
					$lookup: {
						from: "userPackages",
						let: { userId: "$_id" },
						pipeline: [
							{
								$match: {
									$expr: { $eq: ["$userId", "$$userId"] },
								},
							},
							{
								$lookup: {
									from: "products",
									localField: "packageId",
									foreignField: "_id",
									as: "productData",
								},
							},
							{
								$project: {
									_id: 1,
									productName: {
										$ifNull: [
											{ $arrayElemAt: ["$productData.name", 0] },
											"Basic Package",
										],
									},
									price: {
										$ifNull: ["$price", "$amount", "$cost", 100],
									},
									priceNTE: {
										$ifNull: [
											"$priceNTE",
											"$ntePrice",
											"$nteValue",
											"$nte",
											50,
										],
									},
									status: {
										$ifNull: ["$status", "$state", "active"],
									},
									isActive: {
										$ifNull: ["$isActive", "$active", true],
									},
									purchaseDate: {
										$ifNull: [
											"$purchaseDate",
											"$createdAt",
											"$dateCreated",
											"$startDate",
											new Date(),
										],
									},
									productId: "$productId",
								},
							},
							{ $sort: { purchaseDate: -1 } },
						],
						as: "packages",
					},
				},
				{
					$addFields: {
						packageStats: {
							totalPackages: { $size: "$packages" },
							activePackages: {
								$size: {
									$filter: {
										input: "$packages",
										as: "pkg",
										cond: {
											$and: [
												{ $eq: ["$$pkg.isActive", true] },
												{ $eq: ["$$pkg.status", "active"] },
											],
										},
									},
								},
							},
							packages: "$packages",
						},
					},
				},
				{
					$project: {
						username: 1,
						fullName: 1,
						createdAt: 1,
						isActive: 1,
						packageStats: 1,
						referralCode: 1,
					},
				},
			]);

			for (const user of directReferrals) {
				if (!visitedUsernames.has(user.username.toLowerCase())) {
					visitedUsernames.add(user.username.toLowerCase());

					allReferredUsersDetails.push({
						username: user.username,
						fullName: user.fullName,
						createdAt: user.createdAt,
						isActive: user.isActive,
						level: currentLevel + 1,
						referrerUsername: currentUsername,
						packageStats: user.packageStats,
					});

					// Count as active referral only if they have purchased at least one package
					if (user.packageStats.totalPackages > 0) {
						activeReferredUsernames.add(user.username.toLowerCase());
					}

					queue.push({
						username: user.username.toLowerCase(),
						level: currentLevel + 1,
					});
				}
			}
		}

		// Calculate aggregate stats
		const totalPackageStats = allReferredUsersDetails.reduce(
			(acc, user) => {
				acc.totalPackages += user.packageStats.totalPackages;
				acc.activePackages += user.packageStats.activePackages;
				return acc;
			},
			{ totalPackages: 0, activePackages: 0 },
		);

		return {
			totalReferrals: allReferredUsersDetails.length,
			activeReferrals: activeReferredUsernames.size,
			referredUsers: allReferredUsersDetails,
			packageStats: totalPackageStats,
		};
	} catch (error) {
		logger.error("Error getting user referrals with packages (fast):", error);
		throw error;
	}
}

module.exports = {
	userService,
	createWalletHash,
	getUserUpline,
	getUserDownline,
	getUserReferralsWithPackagesFast,
};
