const { voteService } = require('./vote.service');
const { ApiResponse } = require('../../utils/ApiResponse');
const { ApiError } = require('../../utils/ApiError');

const voteController = {
  // List all votes (admin only)
  async listVotes(req, res, next) {
    try {
      const options = {
        page: parseInt(req.query.page) || 1,
        limit: parseInt(req.query.limit) || 20,
        search: req.query.search || '',
        sortBy: req.query.sortBy || 'startDate',
        sortOrder: req.query.sortOrder || 'desc',
      };

      const result = await voteService.listVotes(options);

      res.json(
        new ApiResponse(200, result, 'Votes retrieved successfully')
      );
    } catch (err) {
      next(err);
    }
  },

  // Get public votes (authenticated users)
  async getPublicVotes(req, res, next) {
    try {
      const username = req.user?.username;
      const result = await voteService.getPublicVotes(username);

      res.json(
        new ApiResponse(200, result, 'Public votes retrieved successfully')
      );
    } catch (err) {
      next(err);
    }
  },

  // Get vote by ID (admin only)
  async getVoteById(req, res, next) {
    try {
      const { id } = req.params;
      const vote = await voteService.getVoteById(id);

      res.json(
        new ApiResponse(200, { vote }, 'Vote retrieved successfully')
      );
    } catch (err) {
      next(err);
    }
  },

  // Create new vote (admin only)
  async createVote(req, res, next) {
    try {
      const { title, description, startDate, endDate, status, icon } = req.body;

      const voteData = {
        title,
        description,
        startDate,
        endDate,
        status,
        icon,
      };

      const vote = await voteService.createVote(voteData);

      res.status(201).json(
        new ApiResponse(201, { vote }, 'Vote created successfully')
      );
    } catch (err) {
      next(err);
    }
  },

  // Update vote (admin only)
  async updateVote(req, res, next) {
    try {
      const { id } = req.params;
      const { title, description, startDate, endDate, status, icon } = req.body;

      const updateData = {};
      if (title !== undefined) updateData.title = title;
      if (description !== undefined) updateData.description = description;
      if (startDate !== undefined) updateData.startDate = startDate;
      if (endDate !== undefined) updateData.endDate = endDate;
      if (status !== undefined) updateData.status = status;
      if (icon !== undefined) updateData.icon = icon;

      const vote = await voteService.updateVote(id, updateData);

      res.json(
        new ApiResponse(200, { vote }, 'Vote updated successfully')
      );
    } catch (err) {
      next(err);
    }
  },

  // Toggle vote status (admin only)
  async toggleVoteStatus(req, res, next) {
    try {
      const { id } = req.params;
      const { active } = req.body;

      if (typeof active !== 'boolean') {
        throw new ApiError(400, 'active field must be a boolean');
      }

      const vote = await voteService.toggleVoteStatus(id, active);

      res.json(
        new ApiResponse(
          200,
          { vote },
          `Vote ${active ? 'activated' : 'deactivated'} successfully`
        )
      );
    } catch (err) {
      next(err);
    }
  },

  // Delete vote (admin only)
  async deleteVote(req, res, next) {
    try {
      const { id } = req.params;
      const result = await voteService.deleteVote(id);

      res.json(
        new ApiResponse(200, result, 'Vote deleted successfully')
      );
    } catch (err) {
      next(err);
    }
  },

  // Cast a vote (authenticated users)
  async castVote(req, res, next) {
    try {
      const { id } = req.params;
      const { address } = req.body;
      const username = req.user?.username;

      if (!username) {
        throw new ApiError(401, 'User authentication required');
      }

      if (!address) {
        throw new ApiError(400, 'address is required');
      }

      const voteData = {
        username,
        address,
      };

      const result = await voteService.castVote(id, voteData);

      res.status(201).json(
        new ApiResponse(201, result, 'Vote cast successfully')
      );
    } catch (err) {
      next(err);
    }
  },
};

module.exports = { voteController };
