const { Vote } = require("./vote.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");

const voteService = {
	// List all votes with pagination and filters (admin only)
	async listVotes(options) {
		const {
			page = 1,
			limit = 20,
			search = "",
			sortBy = "startDate",
			sortOrder = "desc",
		} = options;

		const skip = (page - 1) * limit;
		const filter = {};

		if (search) {
			filter.$or = [
				{ title: { $regex: search, $options: "i" } },
				{ description: { $regex: search, $options: "i" } },
			];
		}

		const sortOptions = {};
		sortOptions[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [votes, totalVotes] = await Promise.all([
			Vote.find(filter).sort(sortOptions).skip(skip).limit(limit).lean(),
			Vote.countDocuments(filter),
		]);

		// Add vote count for each vote
		const votesWithCount = votes.map((v) => ({
			...v,
			voteCount: Array.isArray(v.votes) ? v.votes.length : 0,
		}));

		return {
			votes: votesWithCount,
			pagination: {
				page,
				limit,
				total: totalVotes,
				totalPages: Math.ceil(totalVotes / limit),
			},
		};
	},

	// Get public votes (active and upcoming only)
	async getPublicVotes(username = null) {
		const votes = await Vote.find({
			active: true,
			status: { $in: ["active", "upcoming"] },
		})
			.select("title description startDate endDate status icon votes")
			.lean();

		const votesWithCount = votes.map((v) => ({
			...v,
			voteCount: Array.isArray(v.votes) ? v.votes.length : 0,
		}));

		// Get last 12 votes from all votes
		const last12Votes = [];
		for (
			let i = votesWithCount.length - 1;
			i >= 0 && last12Votes.length < 12;
			i--
		) {
			const item = votesWithCount[i];
			const voteArray = Array.isArray(item.votes) ? item.votes : [];

			const needed = 12 - last12Votes.length;
			const recent = voteArray.slice(-needed);

			last12Votes.unshift(...recent);
		}

		const recentVotes = last12Votes.map((vote) => ({
			address: vote.address,
			votedAt: vote.votedAt,
		}));

		// find if user voted or not
		let voted;
		if (username) {
			voted = await Vote.find({
				active: true,
				status: { $in: ["active", "upcoming"] },
				"votes.username": username,
			}).lean();
		}

		return {
			votes: votesWithCount,
			last12Votes: recentVotes,
			voted: voted?.map((vote) => ({
				voteFor: vote._id,
			})),
		};
	},

	// Get vote by ID
	async getVoteById(id) {
		const vote = await Vote.findById(id).lean();
		if (!vote) {
			throw new ApiError(404, "Vote not found");
		}

		return {
			...vote,
			voteCount: Array.isArray(vote.votes) ? vote.votes.length : 0,
		};
	},

	// Create new vote (admin only)
	async createVote(voteData) {
		const { title, description, startDate, endDate, status, icon } = voteData;

		if (!title || !description || !startDate || !endDate || !status || !icon) {
			throw new ApiError(
				400,
				"Missing required fields: title, description, startDate, endDate, status, icon",
			);
		}

		const vote = new Vote({
			title: title.trim(),
			description: description.trim(),
			startDate,
			endDate,
			status,
			icon: icon.trim(),
			active: true,
			votes: [],
		});

		await vote.save();

		logger.info("Vote created", {
			voteId: vote._id,
			title: vote.title,
			status: vote.status,
		});

		return vote.toObject();
	},

	// Update vote (admin only)
	async updateVote(id, updateData) {
		const { title, description, startDate, endDate, status, icon } = updateData;

		const vote = await Vote.findById(id);
		if (!vote) {
			throw new ApiError(404, "Vote not found");
		}

		if (title !== undefined) vote.title = title.trim();
		if (description !== undefined) vote.description = description.trim();
		if (startDate !== undefined) vote.startDate = startDate;
		if (endDate !== undefined) vote.endDate = endDate;
		if (status !== undefined) vote.status = status;
		if (icon !== undefined) vote.icon = icon.trim();

		await vote.save();

		logger.info("Vote updated", {
			voteId: id,
			title: vote.title,
		});

		return vote.toObject();
	},

	// Toggle vote active status (admin only)
	async toggleVoteStatus(id, active) {
		const vote = await Vote.findById(id);
		if (!vote) {
			throw new ApiError(404, "Vote not found");
		}

		vote.active = active;
		await vote.save();

		logger.info("Vote status toggled", {
			voteId: id,
			active,
		});

		return vote.toObject();
	},

	// Delete vote (admin only)
	async deleteVote(id) {
		const vote = await Vote.findById(id);
		if (!vote) {
			throw new ApiError(404, "Vote not found");
		}

		await Vote.deleteOne({ _id: id });

		logger.info("Vote deleted", { voteId: id });

		return { deletedId: id };
	},

	// Cast a vote (authenticated users)
	async castVote(voteId, voteData) {
		const { username, address } = voteData;

		if (!username || !address) {
			throw new ApiError(400, "username and address are required");
		}

		const vote = await Vote.findById(voteId);
		if (!vote) {
			throw new ApiError(404, "Vote not found");
		}

		// Check if user already voted
		const votes = vote.votes || [];
		const existingVote = votes.find((v) => v.username === username);
		if (existingVote) {
			throw new ApiError(403, "User already voted");
		}

		const voteInfo = {
			username,
			address,
			votedAt: new Date().toISOString(),
		};

		vote.votes = votes;
		vote.votes.push(voteInfo);
		await vote.save();

		logger.info("Vote cast", {
			voteId,
			username,
			totalVotes: vote.votes.length,
		});

		return {
			success: true,
			voteCount: vote.votes.length,
		};
	},
};

module.exports = { voteService };
