const { ethers } = require("ethers");
const mongoose = require("mongoose");
const { WalletTransaction } = require("./walletTransaction.model");
const { WithdrawalRequest } = require("../withdrawals/withdrawalRequest.model");
const { Withdrawal } = require("../withdrawals/withdrawal.model");
const { User } = require("../users/user.model");
const { walletService } = require("./wallet.service");
const { logger } = require("../../core/logger/logger");
const { generateTraceId } = require("../purchase/services/userPackage.service");

const isSimulate = process.env.BLOCKCHAIN_SIMULATION_MODE === "true";

if (process.env.NODE_ENV === "production" && isSimulate) {
	logger.error("FATAL: Blockchain simulation mode is ENABLED in PRODUCTION!");
	throw new Error(
		"FATAL: Cannot run blockchain simulation mode in production. Set BLOCKCHAIN_SIMULATION_MODE=false",
	);
}

if (isSimulate) {
	logger.warn(
		"Blockchain simulation mode enabled - no real transfers will occur",
	);
} else {
	logger.info(
		"Blockchain transfers enabled - real transactions will be executed",
	);
}

const BSC_RPC_URL =
	process.env.BSC_RPC_URL || "https://bsc-dataseed1.binance.org/";

// Company wallet private keys for different categories
const COMPANY_WALLET_PRIVATE_KEYS = {
	core: process.env.COMPANY_WALLET_PRIVATE_KEY_CORE,
	elite: process.env.COMPANY_WALLET_PRIVATE_KEY_ELITE,
	stake: process.env.COMPANY_WALLET_PRIVATE_KEY_STAKE,
	meta_pulse: process.env.COMPANY_WALLET_PRIVATE_KEY_META_PULSE,
	others: process.env.COMPANY_WALLET_PRIVATE_KEY_OTHER,
};

const TOKEN_CONTRACT_ADDRESS = process.env.TOKEN_CONTRACT_ADDRESS;

// NTE Token ABI (ERC20 standard functions)
const TOKEN_ABI = [
	"function transfer(address to, uint256 amount) returns (bool)",
	"function balanceOf(address owner) view returns (uint256)",
	"function decimals() view returns (uint8)",
];

const companyWalletService = {
	/**
	 * Transfer NTE tokens from company wallet to user wallet
	 */
	async transferFromCompanyWallet(
		toAddress,
		amountNTE,
		sourceBalanceType = "others",
		metadata = {},
	) {
		const withdrawalRequestId = metadata.withdrawalRequestId;

		try {
			logger.info(
				`COMPANY TRANSFER | Initiating transfer of ${amountNTE} NTE to ${toAddress}`,
			);

			// If we have a withdrawalRequestId, verify this transfer hasn't been executed
			if (withdrawalRequestId) {
				const existingRequest =
					await WithdrawalRequest.findById(withdrawalRequestId);

				if (!existingRequest) {
					throw new Error(
						"INVALID_WITHDRAWAL_REQUEST:Withdrawal request not found",
					);
				}

				// Check if transfer already executed
				if (existingRequest.transferExecuted === true) {
					logger.warn(
						`COMPANY TRANSFER | Transfer already executed for withdrawal ${withdrawalRequestId}`,
					);

					if (existingRequest.transactionHash) {
						return {
							success: true,
							transactionHash: existingRequest.transactionHash,
							blockNumber: existingRequest.blockNumber,
							message: "Transfer already executed (idempotency check)",
							previouslyExecuted: true,
						};
					}

					throw new Error(
						"DUPLICATE_TRANSFER:Transfer already executed for this withdrawal request",
					);
				}

				// Check if blockchain transfer completed
				if (existingRequest.blockchainTransferCompleted === true) {
					logger.warn(
						`COMPANY TRANSFER | Blockchain transfer already completed for withdrawal ${withdrawalRequestId}`,
					);

					if (existingRequest.blockchainTransactionHash) {
						return {
							success: true,
							transactionHash: existingRequest.blockchainTransactionHash,
							blockNumber: existingRequest.blockchainBlockNumber,
							message:
								"Blockchain transfer already completed (idempotency check)",
							previouslyExecuted: true,
						};
					}
				}

				// Atomically mark as executing to prevent duplicate submissions
				const lockResult = await WithdrawalRequest.findOneAndUpdate(
					{
						_id: withdrawalRequestId,
						transferExecuted: { $ne: true },
					},
					{
						transferExecuted: true,
						transferInitiatedAt: new Date(),
					},
					{ new: true },
				);

				if (!lockResult) {
					throw new Error(
						"TRANSFER_LOCK_FAILED:Failed to acquire transfer execution lock",
					);
				}

				logger.info(
					`COMPANY TRANSFER | Marked withdrawal ${withdrawalRequestId} as transferExecuted=true`,
				);
			}

			// Validate inputs
			if (!this.isValidAddress(toAddress)) {
				throw new Error("INVALID_ADDRESS:Invalid recipient wallet address");
			}

			if (amountNTE <= 0) {
				throw new Error("INVALID_AMOUNT:Transfer amount must be positive");
			}

			// Get the appropriate private key for the source balance category
			const privateKey = COMPANY_WALLET_PRIVATE_KEYS[sourceBalanceType];
			if (!privateKey && !isSimulate) {
				throw new Error(
					`MISSING_WALLET_KEY:No private key configured for balance type: ${sourceBalanceType}`,
				);
			}

			if (!TOKEN_CONTRACT_ADDRESS && !isSimulate) {
				throw new Error(
					"MISSING_CONTRACT:Token contract address not configured",
				);
			}

			logger.info(
				`COMPANY TRANSFER | Using ${sourceBalanceType.toUpperCase()} wallet for withdrawal`,
			);

			// Check blockchain history for recent duplicate transactions
			if (withdrawalRequestId) {
				const recentDuplicates = await WithdrawalRequest.find({
					toAddress,
					amountNTE,
					status: { $in: ["completed", "processing"] },
					transactionHash: { $exists: true, $ne: null }, // Only check completed blockchain transfers
					createdAt: { $gte: new Date(Date.now() - 15 * 60 * 1000) }, // Last 15 minutes (replay protection)
					_id: { $ne: withdrawalRequestId },
				}).limit(5);

				if (recentDuplicates.length > 0) {
					logger.warn(
						`COMPANY TRANSFER | Potential duplicate/replay attack detected:`,
						{
							current: withdrawalRequestId,
							duplicatesFound: recentDuplicates.length,
							duplicates: recentDuplicates.map((d) => ({
								id: d._id,
								txHash: d.transactionHash,
								createdAt: d.createdAt,
							})),
						},
					);

					throw new Error(
						`DUPLICATE_DETECTED : Identical withdrawal detected in last 15 minutes (${recentDuplicates.length} duplicates found).`,
					);
				}
			}

			// Execute transfer
			logger.info(`COMPANY TRANSFER | Executing transfer...`);

			let tx, receipt, walletAddress;

			if (isSimulate) {
				// ==================== SIMULATION MODE ====================
				logger.warn(
					`COMPANY TRANSFER | 🔶 SIMULATION MODE - No actual blockchain transfer`,
				);
				const mockTxHash = `0x${Math.random()
					.toString(16)
					.substring(2)}${Math.random()
					.toString(16)
					.substring(2)}${Math.random().toString(16).substring(2)}`;
				const mockBlockNumber = Math.floor(Math.random() * 1000000) + 30000000;
				const mockFromAddress = `0x${Math.random()
					.toString(16)
					.substring(2)
					.padEnd(40, "0")}`;

				walletAddress = mockFromAddress;

				tx = {
					hash: mockTxHash,
					from: mockFromAddress,
					to: toAddress,
					value: ethers.parseUnits(amountNTE.toString(), 18),
				};

				receipt = {
					transactionHash: mockTxHash,
					blockNumber: mockBlockNumber,
					gasUsed: BigInt(65000),
					gasPrice: BigInt(5000000000),
					status: 1,
				};

				await new Promise((resolve) => setTimeout(resolve, 500));

				logger.info(`COMPANY TRANSFER | 🔶 SIMULATED Transaction: ${tx.hash}`);
				logger.info(
					`COMPANY TRANSFER | 🔶 SIMULATED Block: ${receipt.blockNumber}`,
				);
			} else {
				// ==================== REAL BLOCKCHAIN TRANSFER ====================
				const provider = new ethers.JsonRpcProvider(BSC_RPC_URL);
				const wallet = new ethers.Wallet(privateKey, provider);
				const tokenContract = new ethers.Contract(
					TOKEN_CONTRACT_ADDRESS,
					TOKEN_ABI,
					wallet,
				);

				walletAddress = wallet.address;
				logger.info(
					`COMPANY TRANSFER | Company wallet address: ${walletAddress}`,
				);

				// Get token decimals
				const decimals = await tokenContract.decimals();
				const amountInWei = ethers.parseUnits(amountNTE.toString(), decimals);

				logger.info(
					`COMPANY TRANSFER | Amount in wei: ${amountInWei.toString()}`,
				);

				// Check company wallet balance
				const companyBalance = await tokenContract.balanceOf(wallet.address);
				const companyBalanceFormatted = ethers.formatUnits(
					companyBalance,
					decimals,
				);

				logger.info(
					`COMPANY TRANSFER | Company balance: ${companyBalanceFormatted} NTE`,
				);

				if (companyBalance < amountInWei) {
					logger.error(
						`COMPANY TRANSFER | Insufficient company wallet balance`,
						{
							required: amountNTE,
							available: companyBalanceFormatted,
						},
					);
					throw new Error(
						`INSUFFICIENT_COMPANY_BALANCE:Company wallet has insufficient balance. Required: ${amountNTE} NTE, Available: ${companyBalanceFormatted} NTE`,
					);
				}

				// Execute real blockchain transfer
				tx = await tokenContract.transfer(toAddress, amountInWei);
				logger.info(`COMPANY TRANSFER | Transaction submitted: ${tx.hash}`);

				// Wait for confirmation
				receipt = await tx.wait();
				logger.info(
					`COMPANY TRANSFER | Transaction confirmed in block ${receipt.blockNumber}`,
				);
			}

			// Update withdrawal request with transaction details
			if (withdrawalRequestId) {
				await WithdrawalRequest.findByIdAndUpdate(withdrawalRequestId, {
					transactionHash: tx.hash,
					blockNumber: receipt.blockNumber,
					blockchainTransactionHash: tx.hash,
					blockchainBlockNumber: receipt.blockNumber,
					blockchainTransferCompleted: true,
					blockchainCompletedAt: new Date(),
				});
			}

			return {
				success: true,
				transactionHash: tx.hash,
				blockNumber: receipt.blockNumber,
				gasUsed: receipt.gasUsed.toString(),
				effectiveGasPrice: receipt.gasPrice
					? receipt.gasPrice.toString()
					: null,
				from: walletAddress,
				to: toAddress,
				amount: amountNTE,
				message: "Transfer completed successfully",
			};
		} catch (error) {
			logger.error(`COMPANY TRANSFER | Error:`, {
				error: error.message,
				toAddress,
				amountNTE,
				sourceBalanceType,
			});

			// Update withdrawal request with error
			if (withdrawalRequestId) {
				await WithdrawalRequest.findByIdAndUpdate(withdrawalRequestId, {
					transferError: error.message,
					transferFailedAt: new Date(),
					status: "failed",
					failedAt: new Date(),
					failureReason: error.message,
				});
			}

			const [errorCode, errorMessage] = error.message.split(":");
			if (errorMessage) {
				throw new Error(error.message);
			}

			throw new Error(`TRANSFER_FAILED:${error.message}`);
		}
	},

	/**
	 * Validate if address is a valid Ethereum/BSC address format
	 */
	isValidAddress(address) {
		const ethAddressRegex = /^0x[a-fA-F0-9]{40}$/;
		return ethAddressRegex.test(address);
	},

	/**
	 * Determine the source balance type for withdrawal based on user's balance breakdown
	 */
	async determineSourceBalanceType(
		username,
		amountNTE,
		selectedWithdrawalType = null,
	) {
		try {
			const balanceResult = await walletService.getBalanceBreakdown(
				username,
				true,
			);

			if (!balanceResult || !balanceResult.availableBreakdown) {
				throw new Error("BALANCE_ERROR:Unable to retrieve balance breakdown");
			}

			const available = balanceResult.availableBreakdown;
			const withdrawalType = selectedWithdrawalType || "total";

			let sourceBalanceType = "others";
			let packageName = "Unknown";
			let packageType = "Unknown";
			let availableBalance = 0;
			let isProportional = false;

			switch (withdrawalType.toLowerCase()) {
				case "core":
					sourceBalanceType = "core";
					packageName = "Core Package";
					packageType = "core";
					availableBalance = available.core || 0;
					break;
				case "elite":
					sourceBalanceType = "elite";
					packageName = "Elite Package";
					packageType = "elite";
					availableBalance = available.elite || 0;
					break;
				case "stake":
					sourceBalanceType = "stake";
					packageName = "Stake Package";
					packageType = "stake";
					availableBalance = available.stake || 0;
					break;
				case "meta_pulse":
				case "metapulse":
					sourceBalanceType = "meta_pulse";
					packageName = "Meta Pulse Package";
					packageType = "meta_pulse";
					availableBalance = available.meta_pulse || 0;
					break;
				default:
					// For "total" or others, use proportional withdrawal
					sourceBalanceType = "others";
					packageName = "Multiple Packages";
					packageType = "proportional";
					availableBalance = available.total || 0;
					isProportional = true;
					break;
			}

			if (availableBalance < amountNTE) {
				throw new Error(
					`INSUFFICIENT_BALANCE:Insufficient ${packageName} balance. Available: ${availableBalance.toFixed(
						6,
					)} NTE, Requested: ${amountNTE} NTE`,
				);
			}

			return {
				sourceBalanceType,
				packageName,
				packageType,
				availableBalance,
				isProportional,
			};
		} catch (error) {
			logger.error(`Error determining source balance type:`, error);

			const [errorCode, errorMessage] = error.message.split(":");
			if (errorMessage) {
				throw error;
			}

			throw new Error(`BALANCE_CHECK_FAILED:${error.message}`);
		}
	},

	/**
	 * Create a new withdrawal request with atomic guarantees and idempotency
	 */
	async createWithdrawalRequest(requestData) {
		try {
			const {
				userId,
				username,
				toAddress,
				amountNTE,
				amountUSD,
				selectedWithdrawalType,
				metadata,
				sourceBalanceInfo,
				claimRequestId = null,
				walletBalanceBefore,
				idempotencyKey,
			} = requestData;

			// Check for existing withdrawal with this idempotency key
			if (idempotencyKey) {
				const existingWithdrawal = await WithdrawalRequest.findOne({
					idempotencyKey,
				});
				if (existingWithdrawal) {
					logger.warn(`IDEMPOTENCY | Duplicate withdrawal request prevented`, {
						idempotencyKey,
						existingId: existingWithdrawal._id,
						status: existingWithdrawal.status,
					});
					throw new Error(
						"DUPLICATE_WITHDRAWAL:Withdrawal request with this idempotency key already exists",
					);
				}
			}

			const traceId = generateTraceId("withdrawal", username);
			const withdrawalRequest = new WithdrawalRequest({
				userId,
				username,
				toAddress,
				amountNTE,
				amountUSD,
				selectedWithdrawalType,
				status: "pending",
				traceId,
				walletBalanceBefore: walletBalanceBefore || 0,
				metadata,
				sourceBalanceInfo,
				processingStartedAt: new Date(),
				validationStartedAt: new Date(),
				transferStartedAt: new Date(),
				createdAt: new Date(),
				updatedAt: new Date(),
				transferExecuted: false,
				claimRequestId,
				idempotencyKey: idempotencyKey || null,
			});

			await withdrawalRequest.save();

			logger.info(
				`✅ WITHDRAWAL REQUEST CREATED | ID: ${withdrawalRequest._id}`,
				{
					username,
					amountNTE,
					idempotencyKey: idempotencyKey || "none",
				},
			);

			logger.info(
				`Withdrawal request created: ${withdrawalRequest._id} for ${username}`,
			);

			return withdrawalRequest;
		} catch (error) {
			logger.error(`Error creating withdrawal request:`, error);
			throw new Error(`CREATE_REQUEST_FAILED:${error.message}`);
		}
	},

	/**
	 * Get withdrawal request by ID
	 */
	async getWithdrawalRequest(requestId) {
		try {
			const request = await WithdrawalRequest.findById(requestId);
			return request;
		} catch (error) {
			logger.error(`Error getting withdrawal request:`, error);
			throw new Error(`GET_REQUEST_FAILED:${error.message}`);
		}
	},

	/**
	 * Update withdrawal request status and metadata
	 */
	async updateWithdrawalRequest(requestId, updates) {
		try {
			const updatedRequest = await WithdrawalRequest.findByIdAndUpdate(
				requestId,
				{ ...updates, updatedAt: new Date() },
				{ new: true },
			);

			return updatedRequest;
		} catch (error) {
			logger.error(`Error updating withdrawal request:`, error);
			throw new Error(`UPDATE_REQUEST_FAILED:${error.message}`);
		}
	},

	/**
	 * Log withdrawal transaction across all collections
	 */
	async logWithdrawalTransaction(withdrawalData) {
		try {
			const {
				username,
				userId,
				toAddress,
				amountNTE,
				amountUSD,
				transactionHash,
				blockNumber,
				gasUsed,
				sourceBalanceInfo,
				metadata,
				withdrawalRequestId,
				balanceBefore,
				balanceAfter,
			} = withdrawalData;

			// Create wallet transaction record
			const walletTransaction = new WalletTransaction({
				userId,
				username,
				type: "withdrawal",
				subType: "fund_transfer",
				amountNTE: -Math.abs(amountNTE),
				amountUSD: -Math.abs(amountUSD) || 0,
				balanceBefore: balanceBefore || 0,
				balanceAfter: balanceAfter || 0,
				status: "completed",
				description: `Withdrawal to ${toAddress}`,
				transactionHash,
				processedAt: new Date(),
				createdAt: new Date(),
				metadata: {
					bonusTransactionId: new mongoose.Types.ObjectId(),
					distributionId: new mongoose.Types.ObjectId(),
					packageName: sourceBalanceInfo?.packageName || "Unknown Package",
					packageType: sourceBalanceInfo?.packageType || "others",
					amountUSD,
					toAddress,
					gasUsed,
					blockNumber,
					sourceBalanceType: sourceBalanceInfo?.sourceBalanceType,
					withdrawalRequestId,
					...metadata,
				},
			});

			await walletTransaction.save();

			logger.info(`Withdrawal transaction logged: ${walletTransaction._id}`);

			return {
				walletTransactionId: walletTransaction._id,
				transactionHash,
				blockNumber,
			};
		} catch (error) {
			logger.error(`Error logging withdrawal transaction:`, error);
			throw new Error(`LOG_TRANSACTION_FAILED:${error.message}`);
		}
	},

	/**
	 * Process complete withdrawal from company wallet to user wallet
	 */
	async processWithdrawal(withdrawalRequest) {
		let transferExecuted = false;
		let transferResult = null;

		try {
			logger.info(
				`Processing withdrawal ${withdrawalRequest._id} for ${withdrawalRequest.username}`,
			);

			// 1. Validate withdrawal request
			if (withdrawalRequest.status === "completed") {
				logger.warn(`Withdrawal ${withdrawalRequest._id} already completed`);
				return {
					success: true,
					message: "Withdrawal already completed",
					withdrawalRequestId: withdrawalRequest._id,
					transactionHash: withdrawalRequest.transactionHash,
				};
			}

			// 2. Update status to processing
			await this.updateWithdrawalRequest(withdrawalRequest._id, {
				status: "processing",
				processingStartedAt: new Date(),
			});

			// 3. Determine source balance type
			const sourceBalanceInfo =
				withdrawalRequest.sourceBalanceInfo ||
				(await this.determineSourceBalanceType(
					withdrawalRequest.username,
					withdrawalRequest.amountNTE,
					withdrawalRequest.selectedWithdrawalType,
				));

			// 4. Execute blockchain transfer
			logger.info(`Initiating blockchain transfer...`);

			await this.updateWithdrawalRequest(withdrawalRequest._id, {
				status: "transferring",
				transferStartedAt: new Date(),
			});

			transferResult = await this.transferFromCompanyWallet(
				withdrawalRequest.toAddress,
				withdrawalRequest.amountNTE,
				sourceBalanceInfo.sourceBalanceType,
				{
					withdrawalRequestId: withdrawalRequest._id,
					username: withdrawalRequest.username,
					selectedWithdrawalType: withdrawalRequest.selectedWithdrawalType,
				},
			);

			transferExecuted = true;

			if (!transferResult.success) {
				throw new Error(
					`TRANSFER_FAILED:${
						transferResult.message || "Blockchain transfer failed"
					}`,
				);
			}

			logger.info(
				`Blockchain transfer completed: ${transferResult.transactionHash}`,
			);

			// 5. Deduct balance from user
			const user = await User.findById(withdrawalRequest.userId);
			if (!user) {
				throw new Error("USER_NOT_FOUND:User not found");
			}

			// Use MongoDB server-side arithmetic for precise decrement and floor at 0.
			// This works with both Number and Decimal128 walletBalance types.
			const updatedUser = await User.findByIdAndUpdate(
				withdrawalRequest.userId,
				[
					{
						$set: {
							walletBalance: {
								$max: [
									0,
									{
										$subtract: ["$walletBalance", withdrawalRequest.amountNTE],
									},
								],
							},
							lastWithdrawalAt: new Date(),
						},
					},
				],
				{ new: true },
			);

			const newBalance = updatedUser.walletBalance;

			logger.info(
				`User balance updated: ${user.walletBalance} -> ${newBalance} NTE`,
			);

			const completionTimestamp = new Date();

			// 6. Log transaction
			const walletLogResult = await this.logWithdrawalTransaction({
				username: withdrawalRequest.username,
				userId: withdrawalRequest.userId,
				toAddress: withdrawalRequest.toAddress,
				amountNTE: withdrawalRequest.amountNTE,
				amountUSD: withdrawalRequest.amountUSD,
				transactionHash: transferResult.transactionHash,
				blockNumber: transferResult.blockNumber,
				gasUsed: transferResult.gasUsed,
				sourceBalanceInfo,
				metadata: withdrawalRequest.metadata,
				withdrawalRequestId: withdrawalRequest._id,
				balanceBefore: user.walletBalance,
				balanceAfter: newBalance,
			});

			// 7. Log to Withdrawals collection
			const requestedAt =
				withdrawalRequest.requestedAt ||
				withdrawalRequest.createdAt ||
				completionTimestamp;

			await Withdrawal.create({
				userId: withdrawalRequest.userId,
				username: withdrawalRequest.username,
				amountNTE: withdrawalRequest.amountNTE,
				amountUSD: withdrawalRequest.amountUSD,
				withdrawalType:
					withdrawalRequest.selectedWithdrawalType ||
					withdrawalRequest.withdrawalType ||
					"total",
				status: "completed",
				toAddress: withdrawalRequest.toAddress,
				transactionHash: transferResult.transactionHash,
				blockNumber: transferResult.blockNumber,
				gasPrice: transferResult.effectiveGasPrice || "0",
				gasUsed: transferResult.gasUsed || "0",
				requestedAt,
				processedAt: completionTimestamp,
				confirmedAt: completionTimestamp,
				metadata: {
					availableBalance: sourceBalanceInfo.availableBalance,
					bonusTransactionId: new mongoose.Types.ObjectId(),
					confirmedAt: completionTimestamp,
					distributionId: new mongoose.Types.ObjectId(),
					fromAddress: transferResult.from,
					packageName: sourceBalanceInfo.packageName,
					packageType: sourceBalanceInfo.packageType,
					sourceBalanceType: sourceBalanceInfo.sourceBalanceType,
					tokenValue: withdrawalRequest.amountNTE,
					traceId: withdrawalRequest.traceId,
					walletBalanceBefore: user.walletBalance,
					walletTransactionId: walletLogResult.walletTransactionId,
					isProportional: sourceBalanceInfo.isProportional,
					requestId: withdrawalRequest.claimRequestId || null,
					withdrawalRequestId: withdrawalRequest._id,
				},
			});

			// 8. Update withdrawal request to completed
			await this.updateWithdrawalRequest(withdrawalRequest._id, {
				status: "completed",
				completedAt: completionTimestamp,
				transactionHash: transferResult.transactionHash,
				blockNumber: transferResult.blockNumber,
				transferResult: {
					success: true,
					message: "Transfer completed successfully",
					error: null,
				},
			});

			logger.info(`Withdrawal ${withdrawalRequest._id} completed successfully`);

			return {
				success: true,
				message: "Withdrawal completed successfully",
				withdrawalRequestId: withdrawalRequest._id,
				transactionHash: transferResult.transactionHash,
				blockNumber: transferResult.blockNumber,
				newBalance,
			};
		} catch (error) {
			logger.error(`Withdrawal processing failed:`, error);

			// Update withdrawal request with failure
			await this.updateWithdrawalRequest(withdrawalRequest._id, {
				status: "failed",
				failedAt: new Date(),
				failureReason: error.message,
				transferError: error.message,
				transferResult: {
					success: false,
					message: error.message,
					error: error.message,
				},
			});

			const [errorCode, errorMessage] = error.message.split(":");
			if (errorMessage) {
				throw error;
			}

			throw new Error(`WITHDRAWAL_FAILED:${error.message}`);
		}
	},

	/**
	 * Get withdrawal history for a user
	 */
	async getWithdrawalHistory(username, params = {}) {
		try {
			const { page = 1, limit = 10, status = null } = params;
			const query = { username };

			if (status) {
				query.status = status;
			}

			const withdrawals = await WithdrawalRequest.find(query)
				.sort({ createdAt: -1 })
				.skip((page - 1) * limit)
				.limit(limit)
				.lean();

			const total = await WithdrawalRequest.countDocuments(query);

			return {
				withdrawals,
				pagination: {
					page,
					limit,
					total,
					totalPages: Math.ceil(total / limit),
				},
			};
		} catch (error) {
			logger.error(`Error getting withdrawal history:`, error);
			throw new Error(`GET_HISTORY_FAILED:${error.message}`);
		}
	},
};

module.exports = { companyWalletService };
