const { walletService } = require("./wallet.service");
const { ApiResponse } = require("../../utils/ApiResponse");
const { ApiError } = require("../../utils/ApiError");

const walletController = {
	// Get balance breakdown
	async getBalanceBreakdown(req, res, next) {
		try {
			const { username } = req.query;
			const forceRecalculate = req.query.recalculate === "true";
			const rebuild = req.query.rebuild === "true";

			if (!username) {
				throw new ApiError(400, "Username is required");
			}

			const result = await walletService.getBalanceBreakdown(
				username,
				forceRecalculate,
				rebuild,
			);

			// Build category details
			const categories = [
				{
					name: "Core Node Bonuses",
					key: "core",
					value: result.breakdown.core,
					availableValue: result.availableBreakdown.core,
					flaggedAmount: result.flaggedInfo.flaggedByCategory.core,
					description: "Earnings from Core Node package daily bonuses",
				},
				{
					name: "Elite Node Bonuses",
					key: "elite",
					value: result.breakdown.elite,
					availableValue: result.availableBreakdown.elite,
					flaggedAmount: result.flaggedInfo.flaggedByCategory.elite,
					description: "Earnings from Elite Node package daily bonuses",
				},
				{
					name: "Meta Pulse Bonuses",
					key: "meta_pulse",
					value: result.breakdown.meta_pulse,
					availableValue: result.availableBreakdown.meta_pulse,
					flaggedAmount: result.flaggedInfo.flaggedByCategory.meta_pulse,
					description: "Earnings from Meta Pulse package daily bonuses",
				},
				{
					name: "Staking Bonuses",
					key: "stake",
					value: result.breakdown.stake,
					availableValue: result.availableBreakdown.stake,
					flaggedAmount: result.flaggedInfo.flaggedByCategory.stake,
					description: "Earnings from staking bonuses and final claims",
				},
				{
					name: "Other Earnings",
					key: "others",
					value: result.breakdown.others,
					availableValue: result.availableBreakdown.others,
					flaggedAmount: result.flaggedInfo.flaggedByCategory.others,
					description: "Referral bonuses, monthly bonuses, and other earnings",
				},
			];

			res.json(
				new ApiResponse(
					200,
					{
						username,
						timestamp: new Date().toISOString(),
						data: {
							breakdown: result.breakdown,
							availableBreakdown: result.availableBreakdown,
							categories,
							total: result.breakdown.total,
							availableTotal: result.availableBreakdown.total,
							lastUpdated: result.lastUpdated,
							metadata: result.metadata,
							version: result.version,
						},
						validation: result.validation,
						adjustments: {
							hasPendingAdjustments: result.flaggedInfo.totalCount > 0,
							pendingAdjustmentCount: result.flaggedInfo.totalCount,
							totalPendingAmount: result.flaggedInfo.flaggedByCategory.total,
							flaggedByCategory: result.flaggedInfo.flaggedByCategory,
							availableForWithdrawal: result.availableBreakdown.total,
							restrictions:
								result.flaggedInfo.totalCount > 0
									? {
											message: `You have ${
												result.flaggedInfo.totalCount
											} pending balance adjustment${
												result.flaggedInfo.totalCount > 1 ? "s" : ""
											} totaling ${result.flaggedInfo.flaggedByCategory.total.toFixed(
												6,
											)} NTE. Your available balance for withdrawal is reduced by category.`,
											restrictedAmount:
												result.flaggedInfo.flaggedByCategory.total,
											canWithdraw: result.availableBreakdown.total > 0,
									  }
									: null,
						},
					},
					"Balance breakdown retrieved successfully",
				),
			);
		} catch (err) {
			next(err);
		}
	},

	// Fix wallet balances
	async fixBalances(req, res, next) {
		try {
			const targetName = req.query.name;
			const username = targetName && targetName !== "all" ? targetName : null;
			const doFix = req.query.fix === "true";
			const checkDuplicate = req.query.checkDuplicate === "true";
			const summaryOnly = req.query.summaryOnly === "true";
			const limit = parseInt(req.query.limit) || 50;
			const offset = parseInt(req.query.offset) || 0;

			if (checkDuplicate) {
				const result = await walletService.checkDuplicateWithdrawals({
					limit,
					offset,
				});
				return res.json(
					new ApiResponse(
						200,
						{
							duplicatesFound: result.duplicatesFound,
							results: result.results,
							total: result.total,
							limit: result.limit,
							offset: result.offset,
						},
						"Duplicate withdrawal transactions check completed",
					),
				);
			}

			const result = await walletService.fixBalances({
				username,
				doFix,
				limit,
				offset,
			});

			let paginatedResults = result.results;
			let paginatedErrors = result.errors;
			if (summaryOnly) {
				paginatedResults = [];
				paginatedErrors = [];
			}

			res.json(
				new ApiResponse(
					200,
					{
						fixApplied: doFix,
						summary: {
							totalUsers: result.summary.totalProcessed,
							target: targetName || "all",
							updated: result.summary.updated,
							wouldUpdate: doFix
								? 0
								: result.results.filter((r) => r.action === "would_update")
										.length,
							skipped: result.summary.skipped,
							errors: result.summary.errors,
							completedAt: new Date(),
						},
						results: paginatedResults,
						errors: paginatedErrors,
						totalResults: result.results.length,
						totalErrors: result.errors.length,
						limit: summaryOnly ? 0 : limit,
						offset: summaryOnly ? 0 : offset,
					},
					"Wallet balance update process completed",
				),
			);
		} catch (err) {
			next(err);
		}
	},
};

module.exports = { walletController };
