const { BalanceBreakdown } = require("./balanceBreakdown.model");
const { User } = require("../users/user.model");
const { WalletTransaction } = require("./walletTransaction.model");
const { ApiError } = require("../../utils/ApiError");
const { logger } = require("../../core/logger/logger");
const {
	precisionAdd,
	precisionSubtract,
} = require("../../utils/precisionUtils");

const BALANCE_CATEGORIES = ["core", "elite", "stake", "meta_pulse", "others"];
const VALIDATION_EPSILON = 0.000001;
const AGGREGATION_THRESHOLD = 1000;
const BATCH_SIZE = 100;

const categorizeTransaction = (transaction) => {
	const { type, subType, metadata } = transaction;

	if (subType === "package_purchase") {
		return null;
	}

	if (subType === "fund_adjustment" || type === "fund_adjustment") {
		return null;
	}

	// Manual balance adjustment - categorize by packageName
	if (subType === "manual_balance_adjustment") {
		const packageName = metadata?.packageName?.toLowerCase() || "";

		if (packageName.includes("core")) return "core";
		if (packageName.includes("elite")) return "elite";
		if (packageName.includes("stake")) return "stake";
		if (packageName.includes("meta") || packageName.includes("pulse")) {
			return "meta_pulse";
		}
		return "others";
	}

	if (subType === "fund_transfer" || subType === "withdrawal") {
		if (metadata?.sourceBalanceType) {
			return metadata.sourceBalanceType;
		}
		return "others";
	}

	const stakingTypes = [
		"staking_bonus",
		"flexible_staking_bonus",
		"final_bonus_claim",
	];
	const packageBasedTypes = ["daily_bonus", ...stakingTypes];

	if (!packageBasedTypes.includes(subType)) {
		return "others";
	}

	if (metadata?.packageType) {
		if (metadata.packageType === "locked" && stakingTypes.includes(subType)) {
			return "stake";
		}

		if (metadata.packageType === "unlocked" && subType === "daily_bonus") {
			const packageName = metadata.packageName?.toLowerCase() || "";

			if (packageName.includes("core")) return "core";
			if (packageName.includes("elite")) return "elite";
			if (packageName.includes("meta") || packageName.includes("pulse")) {
				return "meta_pulse";
			}
		}
	}

	return "others";
};

const calculateTotal = (balances) => {
	return (
		balances.core +
		balances.elite +
		balances.stake +
		balances.meta_pulse +
		balances.others
	);
};

const buildCategoryExpression = () => {
	return {
		$switch: {
			branches: [
				// Exclude package_purchase
				{
					case: { $eq: ["$subType", "package_purchase"] },
					then: null,
				},
				// Exclude fund_adjustment
				{
					case: {
						$or: [
							{ $eq: ["$subType", "fund_adjustment"] },
							{ $eq: ["$type", "fund_adjustment"] },
						],
					},
					then: null,
				},
				// Manual balance adjustment - categorize by packageName
				{
					case: { $eq: ["$subType", "manual_balance_adjustment"] },
					then: {
						$cond: [
							{
								$regexMatch: {
									input: {
										$toLower: { $ifNull: ["$metadata.packageName", ""] },
									},
									regex: "core",
								},
							},
							"core",
							{
								$cond: [
									{
										$regexMatch: {
											input: {
												$toLower: { $ifNull: ["$metadata.packageName", ""] },
											},
											regex: "elite",
										},
									},
									"elite",
									{
										$cond: [
											{
												$regexMatch: {
													input: {
														$toLower: {
															$ifNull: ["$metadata.packageName", ""],
														},
													},
													regex: "stake",
												},
											},
											"stake",
											{
												$cond: [
													{
														$or: [
															{
																$regexMatch: {
																	input: {
																		$toLower: {
																			$ifNull: ["$metadata.packageName", ""],
																		},
																	},
																	regex: "meta",
																},
															},
															{
																$regexMatch: {
																	input: {
																		$toLower: {
																			$ifNull: ["$metadata.packageName", ""],
																		},
																	},
																	regex: "pulse",
																},
															},
														],
													},
													"meta_pulse",
													"others",
												],
											},
										],
									},
								],
							},
						],
					},
				},
				// fund_transfer/withdrawal with sourceBalanceType
				{
					case: {
						$and: [
							{
								$in: ["$subType", ["fund_transfer", "withdrawal"]],
							},
							{ $ne: ["$metadata.sourceBalanceType", null] },
							{ $ne: ["$metadata.sourceBalanceType", ""] },
						],
					},
					then: "$metadata.sourceBalanceType",
				},
				// fund_transfer/withdrawal without sourceBalanceType
				{
					case: {
						$in: ["$subType", ["fund_transfer", "withdrawal"]],
					},
					then: "others",
				},
				{
					case: {
						$not: {
							$in: [
								"$subType",
								[
									"daily_bonus",
									"staking_bonus",
									"flexible_staking_bonus",
									"final_bonus_claim",
								],
							],
						},
					},
					then: "others",
				},
				// Staking types with locked package
				{
					case: {
						$and: [
							{
								$in: [
									"$subType",
									[
										"staking_bonus",
										"flexible_staking_bonus",
										"final_bonus_claim",
									],
								],
							},
							{ $eq: ["$metadata.packageType", "locked"] },
						],
					},
					then: "stake",
				},
				// daily_bonus with unlocked core
				{
					case: {
						$and: [
							{ $eq: ["$subType", "daily_bonus"] },
							{ $eq: ["$metadata.packageType", "unlocked"] },
							{
								$regexMatch: {
									input: {
										$toLower: { $ifNull: ["$metadata.packageName", ""] },
									},
									regex: "core",
								},
							},
						],
					},
					then: "core",
				},
				// daily_bonus with unlocked elite
				{
					case: {
						$and: [
							{ $eq: ["$subType", "daily_bonus"] },
							{ $eq: ["$metadata.packageType", "unlocked"] },
							{
								$regexMatch: {
									input: {
										$toLower: { $ifNull: ["$metadata.packageName", ""] },
									},
									regex: "elite",
								},
							},
						],
					},
					then: "elite",
				},
				// daily_bonus with unlocked meta/pulse
				{
					case: {
						$and: [
							{ $eq: ["$subType", "daily_bonus"] },
							{ $eq: ["$metadata.packageType", "unlocked"] },
							{
								$or: [
									{
										$regexMatch: {
											input: {
												$toLower: {
													$ifNull: ["$metadata.packageName", ""],
												},
											},
											regex: "meta",
										},
									},
									{
										$regexMatch: {
											input: {
												$toLower: {
													$ifNull: ["$metadata.packageName", ""],
												},
											},
											regex: "pulse",
										},
									},
								],
							},
						],
					},
					then: "meta_pulse",
				},
			],
			default: "others",
		},
	};
};

const walletService = {
	async getBalanceBreakdown(
		username,
		forceRecalculate = false,
		rebuild = false,
	) {
		if (!username) {
			throw new ApiError(400, "Username is required");
		}

		const user = await User.findOne({ username }, { _id: 1, walletBalance: 1 });
		if (!user) {
			throw new ApiError(404, "User not found");
		}

		logger.info("Getting balance breakdown", {
			username,
			forceRecalculate,
			rebuild,
		});

		let breakdown;

		if (rebuild) {
			breakdown = await this.rebuildBalanceBreakdown(username, user._id);
		} else if (forceRecalculate) {
			breakdown = await this.calculateBalanceBreakdown(username, user._id);
		} else {
			breakdown = await BalanceBreakdown.findOne({ username }).lean();

			if (!breakdown) {
				breakdown = await this.calculateBalanceBreakdown(username, user._id);
			} else {
				const latestTransaction = await WalletTransaction.findOne({
					username,
					$or: [
						{ refundStatus: { $exists: false } },
						{ refundStatus: "none" },
						{ refundStatus: "pending" },
						{ refundStatus: null },
					],
				})
					.sort({ createdAt: -1 })
					.limit(1)
					.lean();

				if (latestTransaction) {
					const lastProcessedDate = breakdown.lastTransactionProcessed;
					const latestTransactionDate = latestTransaction.createdAt;

					if (!lastProcessedDate || latestTransactionDate > lastProcessedDate) {
						breakdown = await this.incrementalUpdateBreakdown(
							username,
							user._id,
							breakdown,
							lastProcessedDate,
						);
					}
				}
			}
		}

		const flaggedInfo = await this.getUnpaidFlaggedTransactionsByCategory(
			username,
		);

		const originalBalances = breakdown.balances;
		const flaggedByCategory = flaggedInfo.flaggedByCategory;

		const availableBalances = {
			core: Math.max(
				0,
				precisionSubtract(originalBalances.core, flaggedByCategory.core),
			),
			elite: Math.max(
				0,
				precisionSubtract(originalBalances.elite, flaggedByCategory.elite),
			),
			stake: Math.max(
				0,
				precisionSubtract(originalBalances.stake, flaggedByCategory.stake),
			),
			meta_pulse: Math.max(
				0,
				precisionSubtract(
					originalBalances.meta_pulse,
					flaggedByCategory.meta_pulse,
				),
			),
			others: Math.max(
				0,
				precisionSubtract(originalBalances.others, flaggedByCategory.others),
			),
			total: 0,
		};

		availableBalances.total = calculateTotal(availableBalances);

		return {
			breakdown: originalBalances,
			availableBreakdown: availableBalances,
			flaggedInfo,
			lastUpdated: breakdown.lastUpdated,
			metadata: breakdown.metadata,
			version: breakdown.version || 1,
			validation: {
				walletBalance: user.walletBalance || 0,
				calculatedTotal: originalBalances.total,
				difference: Math.abs(
					(user.walletBalance || 0) - originalBalances.total,
				),
				isValid:
					Math.abs((user.walletBalance || 0) - originalBalances.total) <
					VALIDATION_EPSILON,
			},
		};
	},

	async calculateBalanceBreakdown(username, userId) {
		const txCount = await WalletTransaction.countDocuments({
			username,
			$or: [
				{ refundStatus: { $exists: false } },
				{ refundStatus: "none" },
				{ refundStatus: "pending" },
				{ refundStatus: null },
			],
		});

		if (txCount >= AGGREGATION_THRESHOLD) {
			logger.info("Using aggregation pipeline for large dataset", {
				username,
				transactionCount: txCount,
			});
			return await this.calculateBalanceBreakdownAggregation(username, userId);
		}

		const transactions = await WalletTransaction.find({
			username,
			$or: [
				{ refundStatus: { $exists: false } },
				{ refundStatus: "none" },
				{ refundStatus: "pending" },
				{ refundStatus: null },
			],
		})
			.sort({ createdAt: 1 })
			.lean();

		const balances = {
			core: 0,
			elite: 0,
			stake: 0,
			meta_pulse: 0,
			others: 0,
			total: 0,
		};

		let lastTransactionDate = null;
		let processedCount = 0;

		for (const tx of transactions) {
			const category = categorizeTransaction(tx);

			if (!category) continue;

			const amount = tx.amountNTE || 0;

			if (BALANCE_CATEGORIES.includes(category)) {
				balances[category] = precisionAdd(balances[category], amount);
			}

			lastTransactionDate = tx.createdAt;
			processedCount++;
		}

		balances.total = calculateTotal(balances);
		const breakdown = await BalanceBreakdown.findOneAndUpdate(
			{ username },
			{
				userId,
				username,
				balances,
				lastTransactionProcessed: lastTransactionDate,
				lastUpdated: new Date(),
				metadata: {
					calculationType: "full",
					lastRecalculationReason: "force_recalculate",
					totalTransactionsProcessed: processedCount,
				},
				version: 1,
			},
			{ upsert: true, new: true },
		);

		logger.info("Balance breakdown calculated", {
			username,
			total: balances.total,
			transactionsProcessed: processedCount,
		});

		return breakdown;
	},

	async calculateBalanceBreakdownAggregation(username, userId) {
		const pipeline = [
			// Stage 1: Match transactions
			{
				$match: {
					username,
					$or: [
						{ refundStatus: { $exists: false } },
						{ refundStatus: "none" },
						{ refundStatus: "pending" },
						{ refundStatus: null },
					],
				},
			},
			// Stage 2: Sort by creation date
			{
				$sort: { createdAt: 1 },
			},
			// Stage 3: Add computed category field
			{
				$addFields: {
					category: buildCategoryExpression(),
				},
			},
			// Stage 4: Filter out null categories (excluded transactions)
			{
				$match: {
					category: { $ne: null },
				},
			},
			// Stage 5: Facet for parallel processing
			{
				$facet: {
					// Group by category and sum amounts
					balancesByCategory: [
						{
							$group: {
								_id: "$category",
								total: { $sum: { $ifNull: ["$amountNTE", 0] } },
							},
						},
					],
					// Get last transaction date
					lastTransaction: [
						{ $sort: { createdAt: -1 } },
						{ $limit: 1 },
						{ $project: { createdAt: 1 } },
					],
					// Count processed transactions
					count: [{ $count: "total" }],
				},
			},
		];

		const [result] = await WalletTransaction.aggregate(pipeline);

		// Initialize balances
		const balances = {
			core: 0,
			elite: 0,
			stake: 0,
			meta_pulse: 0,
			others: 0,
			total: 0,
		};

		// Map aggregation results to balance categories
		if (result.balancesByCategory) {
			for (const { _id: category, total } of result.balancesByCategory) {
				if (BALANCE_CATEGORIES.includes(category)) {
					balances[category] = total;
				}
			}
		}

		balances.total = calculateTotal(balances);

		const lastTransactionDate = result.lastTransaction?.[0]?.createdAt || null;
		const processedCount = result.count?.[0]?.total || 0;

		const breakdown = await BalanceBreakdown.findOneAndUpdate(
			{ username },
			{
				userId,
				username,
				balances,
				lastTransactionProcessed: lastTransactionDate,
				lastUpdated: new Date(),
				metadata: {
					calculationType: "full_aggregation",
					lastRecalculationReason: "force_recalculate",
					totalTransactionsProcessed: processedCount,
				},
				version: 1,
			},
			{ upsert: true, new: true },
		);

		logger.info("Balance breakdown calculated via aggregation", {
			username,
			total: balances.total,
			transactionsProcessed: processedCount,
		});

		return breakdown;
	},

	async incrementalUpdateBreakdown(
		username,
		userId,
		existingBreakdown,
		fromDate,
	) {
		const query = {
			username,
			$or: [
				{ refundStatus: { $exists: false } },
				{ refundStatus: "none" },
				{ refundStatus: "pending" },
				{ refundStatus: null },
			],
		};

		if (fromDate) {
			query.createdAt = { $gt: fromDate };
		}

		const newTransactions = await WalletTransaction.find(query)
			.sort({ createdAt: 1 })
			.lean();

		if (newTransactions.length === 0) {
			return existingBreakdown;
		}

		const balances = { ...existingBreakdown.balances };
		let lastTransactionDate = existingBreakdown.lastTransactionProcessed;
		let processedCount = 0;

		for (const tx of newTransactions) {
			const category = categorizeTransaction(tx);

			if (!category) continue;

			const amount = tx.amountNTE || 0;

			if (BALANCE_CATEGORIES.includes(category)) {
				balances[category] = precisionAdd(balances[category], amount);
			}

			lastTransactionDate = tx.createdAt;
			processedCount++;
		}

		balances.total = calculateTotal(balances);
		const breakdown = await BalanceBreakdown.findOneAndUpdate(
			{ username },
			{
				balances,
				lastTransactionProcessed: lastTransactionDate,
				lastUpdated: new Date(),
				"metadata.calculationType": "incremental",
				"metadata.lastRecalculationReason": "automatic_incremental_update",
				$inc: {
					version: 1,
					"metadata.totalTransactionsProcessed": processedCount,
				},
			},
			{ new: true },
		);

		logger.info("Incremental balance breakdown update", {
			username,
			newTransactionsProcessed: processedCount,
			total: balances.total,
		});

		return breakdown;
	},

	async rebuildBalanceBreakdown(username, userId) {
		logger.info("Rebuilding balance breakdown", { username });

		await BalanceBreakdown.deleteOne({ username });
		return await this.calculateBalanceBreakdown(username, userId);
	},

	async getUnpaidFlaggedTransactionsByCategory(username) {
		const pipeline = [
			{
				$match: {
					username,
					flaggedForRefund: true,
					refundStatus: { $in: ["pending", "none"] },
				},
			},
			{
				$addFields: {
					category: {
						$cond: [
							{ $eq: ["$subType", "manual_balance_adjustment"] },
							{
								$cond: [
									{
										$regexMatch: {
											input: {
												$toLower: { $ifNull: ["$metadata.packageName", ""] },
											},
											regex: "core",
										},
									},
									"core",
									{
										$cond: [
											{
												$regexMatch: {
													input: {
														$toLower: {
															$ifNull: ["$metadata.packageName", ""],
														},
													},
													regex: "elite",
												},
											},
											"elite",
											{
												$cond: [
													{
														$regexMatch: {
															input: {
																$toLower: {
																	$ifNull: ["$metadata.packageName", ""],
																},
															},
															regex: "stake",
														},
													},
													"stake",
													{
														$cond: [
															{
																$or: [
																	{
																		$regexMatch: {
																			input: {
																				$toLower: {
																					$ifNull: [
																						"$metadata.packageName",
																						"",
																					],
																				},
																			},
																			regex: "meta",
																		},
																	},
																	{
																		$regexMatch: {
																			input: {
																				$toLower: {
																					$ifNull: [
																						"$metadata.packageName",
																						"",
																					],
																				},
																			},
																			regex: "pulse",
																		},
																	},
																],
															},
															"meta_pulse",
															"others",
														],
													},
												],
											},
										],
									},
								],
							},
							{ $ifNull: ["$metadata.sourceBalanceType", "others"] },
						],
					},
				},
			},
			{
				$group: {
					_id: "$category",
					total: {
						$sum: {
							$abs: { $ifNull: ["$amountNTE", 0] },
						},
					},
					count: { $sum: 1 },
				},
			},
		];

		const aggregationResult = await WalletTransaction.aggregate(pipeline);

		const flaggedByCategory = {
			core: 0,
			elite: 0,
			stake: 0,
			meta_pulse: 0,
			others: 0,
			total: 0,
		};

		let totalCount = 0;

		for (const { _id: category, total, count } of aggregationResult) {
			if (["core", "elite", "stake", "meta_pulse"].includes(category)) {
				flaggedByCategory[category] = total;
			} else {
				flaggedByCategory.others += total;
			}
			flaggedByCategory.total += total;
			totalCount += count;
		}

		const flaggedTransactions = await WalletTransaction.find({
			username,
			flaggedForRefund: true,
			refundStatus: { $in: ["pending", "none"] },
		}).lean();

		return {
			flaggedByCategory,
			totalCount,
			transactions: flaggedTransactions,
		};
	},

	async fixBalances(options = {}) {
		const { username, doFix = false } = options;

		const query = username ? { username } : {};
		const users = await User.find(query, {
			username: 1,
			walletBalance: 1,
		}).lean();

		if (username && users.length === 0) {
			throw new ApiError(404, `User '${username}' not found`);
		}

		let updatedCount = 0;
		let skippedCount = 0;
		let errorCount = 0;
		const results = [];
		const errors = [];

		logger.info("Starting balance fix", {
			userCount: users.length,
			doFix,
			username: username || "all",
		});

		if (username || users.length <= BATCH_SIZE) {
			for (const user of users) {
				const result = await this.processUserBalanceFix(user, doFix);

				if (result.action === "updated") updatedCount++;
				else if (result.action === "skipped") skippedCount++;
				else if (result.action === "error") {
					errorCount++;
					errors.push({ username: user.username, error: result.error });
				}

				if (result.action !== "error") {
					results.push(result);
				}
			}
		} else {
			for (let i = 0; i < users.length; i += BATCH_SIZE) {
				const batch = users.slice(i, i + BATCH_SIZE);
				logger.info(
					`Processing batch ${Math.floor(i / BATCH_SIZE) + 1}/${Math.ceil(
						users.length / BATCH_SIZE,
					)}`,
					{
						batchSize: batch.length,
					},
				);

				const batchResults = await Promise.allSettled(
					batch.map((user) => this.processUserBalanceFix(user, doFix)),
				);

				for (const promiseResult of batchResults) {
					if (promiseResult.status === "fulfilled") {
						const result = promiseResult.value;

						if (result.action === "updated") updatedCount++;
						else if (result.action === "skipped") skippedCount++;
						else if (result.action === "error") {
							errorCount++;
							errors.push({ username: result.username, error: result.error });
						}

						if (result.action !== "error") {
							results.push(result);
						}
					} else {
						errorCount++;
						errors.push({
							username: "unknown",
							error: promiseResult.reason?.message || "Unknown error",
						});
					}
				}
			}
		}

		return {
			summary: {
				totalUsers: users.length,
				target: username || "all",
				updated: updatedCount,
				skipped: skippedCount,
				errors: errorCount,
			},
			results,
			errors,
		};
	},

	async processUserBalanceFix(user, doFix) {
		try {
			const breakdown = await BalanceBreakdown.findOne({
				username: user.username,
			}).lean();

			if (!breakdown) {
				return {
					username: user.username,
					action: "error",
					error: "No breakdown found",
				};
			}

			const flaggedInfo = await this.getUnpaidFlaggedTransactionsByCategory(
				user.username,
			);

			const balances = breakdown.balances || breakdown;

			const negativeFields = [];
			for (const key of [
				"core",
				"elite",
				"stake",
				"meta_pulse",
				"others",
				"total",
			]) {
				const val = balances?.[key];
				if (typeof val === "number" && val < 0) {
					negativeFields.push({ field: key, value: val });
				}
			}

			if (negativeFields.length > 0) {
				logger.warn(
					`NEGATIVE BALANCE DETECTED | ${user.username} | negative fields:`,
					negativeFields,
					"| PROCEEDING WITH UPDATE",
				);
			}

			const calculatedTotal =
				(typeof breakdown.total === "number" ? breakdown.total : undefined) ??
				(typeof balances?.total === "number" ? balances.total : 0);

			// Calculate available balance after flagged deductions
			const flaggedByCategory = flaggedInfo.flaggedByCategory;
			const originalBreakdown = balances;

			// Calculate available balance per category
			const availableBreakdown = {
				core: originalBreakdown.core - flaggedByCategory.core,
				elite: originalBreakdown.elite - flaggedByCategory.elite,
				stake: originalBreakdown.stake - flaggedByCategory.stake,
				meta_pulse: originalBreakdown.meta_pulse - flaggedByCategory.meta_pulse,
				others: originalBreakdown.others - flaggedByCategory.others,
			};

			availableBreakdown.total =
				availableBreakdown.core +
				availableBreakdown.elite +
				availableBreakdown.stake +
				availableBreakdown.meta_pulse +
				availableBreakdown.others;

			const correctWalletBalance = availableBreakdown.total;
			const currentBalance = user.walletBalance || 0;

			if (correctWalletBalance === currentBalance) {
				return {
					username: user.username,
					action: "skipped",
					currentBalance,
					calculatedTotal,
					availableBalance: correctWalletBalance,
					flaggedAmount: flaggedByCategory.total,
					difference: 0,
				};
			}

			if (!doFix) {
				const difference = correctWalletBalance - currentBalance;
				logger.info(
					`WOULD UPDATE | ${
						user.username
					} | ${currentBalance} → ${correctWalletBalance} (${
						difference >= 0 ? "+" : ""
					}${difference}) | Flagged: ${flaggedByCategory.total}`,
				);
				return {
					username: user.username,
					action: "would_update",
					previousBalance: currentBalance,
					newBalance: correctWalletBalance,
					calculatedTotal,
					availableBalance: correctWalletBalance,
					flaggedAmount: flaggedByCategory.total,
					difference,
				};
			}

			await User.updateOne(
				{ username: user.username },
				{
					$set: {
						walletBalance: correctWalletBalance,
					},
				},
			);

			const difference = correctWalletBalance - currentBalance;
			logger.info(
				`UPDATED | ${
					user.username
				} | ${currentBalance} → ${correctWalletBalance} (${
					difference >= 0 ? "+" : ""
				}${difference}) | Flagged: ${flaggedByCategory.total}`,
			);

			return {
				username: user.username,
				action: "updated",
				previousBalance: currentBalance,
				newBalance: correctWalletBalance,
				calculatedTotal,
				availableBalance: correctWalletBalance,
				flaggedAmount: flaggedByCategory.total,
				difference,
			};
		} catch (error) {
			logger.error("Balance fix error", {
				username: user.username,
				error: error.message,
			});
			return {
				username: user.username,
				action: "error",
				error: error.message,
			};
		}
	},

	async checkDuplicateWithdrawals(options = {}) {
		const { limit = 50, offset = 0 } = options;

		const pipeline = [
			{
				$match: {
					type: "withdrawal",
					amountUSD: { $exists: true },
					amountNTE: { $exists: true },
					username: { $exists: true },
					balanceBefore: { $exists: true },
					balanceAfter: { $exists: true },
					refundStatus: { $ne: "completed" },
				},
			},
			{
				$group: {
					_id: {
						username: "$username",
						amountUSD: "$amountUSD",
						amountNTE: "$amountNTE",
						balanceBefore: "$balanceBefore",
						balanceAfter: "$balanceAfter",
					},
					count: { $sum: 1 },
					docs: { $push: "$$ROOT" },
				},
			},
			{ $match: { count: { $gte: 2 } } },
			{ $sort: { count: -1 } },
		];

		const allResults = await WalletTransaction.aggregate(pipeline);
		const paginatedResults = allResults.slice(offset, offset + limit);

		return {
			duplicatesFound: allResults.length,
			results: paginatedResults,
			total: allResults.length,
			limit,
			offset,
		};
	},
};

module.exports = { walletService };
