const { WalletTransaction } = require('./walletTransaction.model');
const { ApiError } = require('../../utils/ApiError');
const { logger } = require('../../core/logger/logger');

const walletTransactionService = {
  // Get wallet transactions list with comprehensive filtering and pagination
  async listWalletTransactions(options) {
    const {
      username,
      type,
      subType,
      status,
      flaggedForRefund,
      refundStatus,
      startDate,
      endDate,
      minAmount,
      maxAmount,
      page = 1,
      limit = 20,
      sortBy = 'createdAt',
      sortOrder = 'desc',
    } = options;

    const skip = (page - 1) * limit;
    const filter = {};

    // Username filter (required)
    if (username) {
      filter.username = username;
    }

    // Transaction type filter (optional)
    if (type) {
      filter.type = type;
    }

    // Transaction subType filter (optional)
    if (subType) {
      filter.subType = subType;
    }

    // Status filter (optional)
    if (status && status !== 'all') {
      filter.status = status;
    }

    // Flagged for refund filter (optional)
    if (flaggedForRefund !== undefined) {
      filter.flaggedForRefund = flaggedForRefund === 'true';
    }

    // Refund status filter (optional)
    if (refundStatus && refundStatus !== 'all') {
      filter.refundStatus = refundStatus;
    }

    // Date range filter (optional)
    if (startDate || endDate) {
      filter.createdAt = {};
      if (startDate) {
        filter.createdAt.$gte = new Date(startDate);
      }
      if (endDate) {
        filter.createdAt.$lte = new Date(endDate);
      }
    }

    // Amount range filter (optional)
    if (minAmount || maxAmount) {
      filter.amountNTE = {};
      if (minAmount) {
        filter.amountNTE.$gte = parseFloat(minAmount);
      }
      if (maxAmount) {
        filter.amountNTE.$lte = parseFloat(maxAmount);
      }
    }

    const sortOptions = {};
    sortOptions[sortBy] = sortOrder === 'asc' ? 1 : -1;

    const [transactions, total] = await Promise.all([
      WalletTransaction.find(filter)
        .sort(sortOptions)
        .skip(skip)
        .limit(limit)
        .lean(),
      WalletTransaction.countDocuments(filter),
    ]);

    // Calculate summary statistics
    const summaryPipeline = [
      { $match: filter },
      {
        $group: {
          _id: null,
          totalTransactions: { $sum: 1 },
          creditTransactions: {
            $sum: {
              $cond: [{ $gt: ['$amountNTE', 0] }, 1, 0]
            }
          },
          debitTransactions: {
            $sum: {
              $cond: [{ $lt: ['$amountNTE', 0] }, 1, 0]
            }
          },
          totalCreditNTE: {
            $sum: {
              $cond: [{ $gt: ['$amountNTE', 0] }, '$amountNTE', 0]
            }
          },
          totalDebitNTE: {
            $sum: {
              $cond: [{ $lt: ['$amountNTE', 0] }, { $abs: '$amountNTE' }, 0]
            }
          },
          totalCreditUSD: {
            $sum: {
              $cond: [{ $gt: ['$amountUSD', 0] }, '$amountUSD', 0]
            }
          },
          totalDebitUSD: {
            $sum: {
              $cond: [{ $lt: ['$amountUSD', 0] }, { $abs: '$amountUSD' }, 0]
            }
          },
          netNTE: { $sum: '$amountNTE' },
          netUSD: { $sum: '$amountUSD' },
        }
      }
    ];

    const summaryResult = await WalletTransaction.aggregate(summaryPipeline);
    const summary = summaryResult.length > 0 
      ? {
          totalTransactions: summaryResult[0].totalTransactions,
          creditTransactions: summaryResult[0].creditTransactions,
          debitTransactions: summaryResult[0].debitTransactions,
          totalCreditNTE: summaryResult[0].totalCreditNTE,
          totalDebitNTE: summaryResult[0].totalDebitNTE,
          totalCreditUSD: summaryResult[0].totalCreditUSD,
          totalDebitUSD: summaryResult[0].totalDebitUSD,
          netNTE: summaryResult[0].netNTE,
          netUSD: summaryResult[0].netUSD,
        }
      : {
          totalTransactions: 0,
          creditTransactions: 0,
          debitTransactions: 0,
          totalCreditNTE: 0,
          totalDebitNTE: 0,
          totalCreditUSD: 0,
          totalDebitUSD: 0,
          netNTE: 0,
          netUSD: 0,
        };

    const totalPages = Math.ceil(total / limit);

    return {
      username,
      transactions,
      summary,
      pagination: {
        total,
        page,
        limit,
        totalPages,
        hasNext: page < totalPages,
        hasPrev: page > 1,
      },
    };
  },

  // Find wallet transaction by ID
  async getWalletTransactionById(id) {
    const transaction = await WalletTransaction.findById(id).lean();
    if (!transaction) {
      throw new ApiError(404, 'Wallet transaction not found');
    }

    return transaction;
  },
};

module.exports = { walletTransactionService };
