const { WalletTransaction } = require("./walletTransaction.model");
const { getTokenValue } = require("../config/globalConfig.service");

const walletTransactionHistoryService = {
	async getTransactionHistory(queryParams) {
		const {
			page = 1,
			limit = 10,
			sortBy = "createdAt",
			sortOrder = "desc",
			search,
			username,
			type,
			status,
			startDate,
			endDate,
			minAmount,
			maxAmount,
			timezone = 0,
		} = queryParams;

		const parsedLimit = Math.min(Math.max(Number(limit) || 10, 1), 100);
		const parsedPage = Math.max(1, Number(page) || 1);
		const skip = (parsedPage - 1) * parsedLimit;

		const searchQuery = {};

		if (username && username.trim()) {
			searchQuery.username = { $regex: username, $options: "i" };
		}

		// Type can match either type or subType
		if (type && type.trim()) {
			searchQuery.$or = [{ type: type }, { subType: type }];
		}

		if (status && status.trim()) {
			searchQuery.status = status;
		}

		// Date range with timezone handling
		if ((startDate && startDate.trim()) || (endDate && endDate.trim())) {
			searchQuery.createdAt = {};
			const clientTimezoneOffset = parseInt(timezone) || 0;
			const offsetMs = -clientTimezoneOffset * 60 * 1000;

			const parseLocalDateToUTC = (dateStr, endOfDay = false) => {
				const [year, month, day] = dateStr.split("-").map(Number);
				const utcDate = new Date(
					Date.UTC(
						year,
						month - 1,
						day,
						endOfDay ? 23 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 999 : 0,
					),
				);
				return new Date(utcDate.getTime() - offsetMs);
			};

			if (startDate && startDate.trim()) {
				searchQuery.createdAt.$gte = parseLocalDateToUTC(startDate, false);
			}

			if (endDate && endDate.trim()) {
				searchQuery.createdAt.$lte = parseLocalDateToUTC(endDate, true);
			}
		}

		// Amount range
		if (minAmount && String(minAmount).trim()) {
			searchQuery.amountNTE = { ...searchQuery.amountNTE, $gte: parseFloat(minAmount) };
		}
		if (maxAmount && String(maxAmount).trim()) {
			searchQuery.amountNTE = { ...searchQuery.amountNTE, $lte: parseFloat(maxAmount) };
		}

		// Text search across multiple fields
		if (search && search.trim()) {
			const searchConditions = [
				{ username: { $regex: search, $options: "i" } },
				{ type: { $regex: search, $options: "i" } },
				{ subType: { $regex: search, $options: "i" } },
				{ description: { $regex: search, $options: "i" } },
				{ "metadata.packageName": { $regex: search, $options: "i" } },
				{ "metadata.paymentMethod": { $regex: search, $options: "i" } },
			];

			// Combine with existing type filter if present
			if (searchQuery.$or) {
				searchQuery.$and = [{ $or: searchQuery.$or }, { $or: searchConditions }];
				delete searchQuery.$or;
			} else {
				searchQuery.$or = searchConditions;
			}
		}

		const sortObj = {};
		sortObj[sortBy] = sortOrder === "asc" ? 1 : -1;

		const tokenValue = await getTokenValue();

		const [transactions, totalCount, summaryStats] = await Promise.all([
			WalletTransaction.find(searchQuery).sort(sortObj).skip(skip).limit(parsedLimit).lean(),
			WalletTransaction.countDocuments(searchQuery),
			this._getSummaryStats(searchQuery, tokenValue),
		]);

		const totalPages = Math.ceil(totalCount / parsedLimit);

		// Process transactions with USD calculation
		const processedTransactions = transactions.map((tx) => ({
			...tx,
			_id: tx._id.toString(),
			userId: tx.userId?.toString(),
			amountNTE: tx.amountNTE || 0,
			amountUSD: this._calculateUSD(tx.amountNTE, tx.amountUSD, tokenValue),
		}));

		const safeStats = {
			totalUsers: summaryStats.totalUsers || 0,
			totalAmountNTE: summaryStats.totalAmountNTE || 0,
			totalAmountUSD: summaryStats.totalAmountUSD || 0,
			averageAmountNTE: summaryStats.averageAmountNTE || 0,
			averageAmountUSD: summaryStats.averageAmountUSD || 0,
			minAmountNTE: summaryStats.minAmountNTE || 0,
			maxAmountNTE: summaryStats.maxAmountNTE || 0,
			uniqueTypes: summaryStats.uniqueTypes || 0,
			uniqueSubTypes: summaryStats.uniqueSubTypes || 0,
			totalTransactions: summaryStats.totalTransactions || 0,
			transactionsWithOriginalUSD: summaryStats.transactionsWithOriginalUSD || 0,
			transactionsWithCalculatedUSD: summaryStats.transactionsWithCalculatedUSD || 0,
		};

		return {
			success: true,
			transactions: processedTransactions,
			pagination: {
				total: totalCount,
				page: parsedPage,
				limit: parsedLimit,
				totalPages,
				hasNext: parsedPage < totalPages,
				hasPrev: parsedPage > 1,
			},
			summary: {
				totalTransactions: totalCount,
				totalUsers: safeStats.totalUsers,
				totalAmountNTE: safeStats.totalAmountNTE,
				totalAmountUSD: safeStats.totalAmountUSD,
				averageAmountNTE: safeStats.averageAmountNTE,
				averageAmountUSD: safeStats.averageAmountUSD,
				minAmountNTE: safeStats.minAmountNTE,
				maxAmountNTE: safeStats.maxAmountNTE,
				uniqueTypes: safeStats.uniqueTypes,
				uniqueSubTypes: safeStats.uniqueSubTypes,
			},
		};
	},

	_calculateUSD(amountNTE, amountUSD, tokenValue) {
		if (!amountUSD || amountUSD === 0 || isNaN(amountUSD)) {
			return (amountNTE || 0) * tokenValue;
		}
		return amountUSD;
	},

	async _getSummaryStats(searchQuery, tokenValue) {
		const result = await WalletTransaction.aggregate([
			{ $match: searchQuery },
			{
				$addFields: {
					actualAmountUSD: {
						$cond: {
							if: {
								$or: [
									{ $eq: ["$amountUSD", null] },
									{ $eq: ["$amountUSD", 0] },
									{ $not: ["$amountUSD"] },
									{ $eq: [{ $type: "$amountUSD" }, "missing"] },
								],
							},
							then: { $multiply: [{ $ifNull: ["$amountNTE", 0] }, tokenValue] },
							else: { $ifNull: ["$amountUSD", 0] },
						},
					},
					calculationAmountNTE: {
						$cond: {
							if: { $eq: ["$subType", "package_purchase"] },
							then: { $abs: { $ifNull: ["$amountNTE", 0] } },
							else: { $ifNull: ["$amountNTE", 0] },
						},
					},
				},
			},
			{
				$addFields: {
					calculationAmountUSD: {
						$cond: {
							if: { $eq: ["$subType", "package_purchase"] },
							then: { $abs: { $ifNull: ["$actualAmountUSD", 0] } },
							else: { $ifNull: ["$actualAmountUSD", 0] },
						},
					},
				},
			},
			{
				$group: {
					_id: null,
					totalUsers: { $addToSet: "$username" },
					totalAmountNTE: { $sum: { $ifNull: ["$calculationAmountNTE", 0] } },
					totalAmountUSD: { $sum: { $ifNull: ["$calculationAmountUSD", 0] } },
					averageAmountNTE: { $avg: { $ifNull: ["$calculationAmountNTE", 0] } },
					averageAmountUSD: { $avg: { $ifNull: ["$calculationAmountUSD", 0] } },
					minAmountNTE: { $min: { $ifNull: ["$calculationAmountNTE", 0] } },
					maxAmountNTE: { $max: { $ifNull: ["$calculationAmountNTE", 0] } },
					transactionTypes: { $addToSet: "$type" },
					transactionSubTypes: { $addToSet: "$subType" },
					totalTransactions: { $sum: 1 },
					transactionsWithOriginalUSD: {
						$sum: {
							$cond: [
								{
									$and: [
										{ $ne: ["$amountUSD", null] },
										{ $ne: ["$amountUSD", 0] },
										{ $ne: [{ $type: "$amountUSD" }, "missing"] },
									],
								},
								1,
								0,
							],
						},
					},
					transactionsWithCalculatedUSD: {
						$sum: {
							$cond: [
								{
									$or: [
										{ $eq: ["$amountUSD", null] },
										{ $eq: ["$amountUSD", 0] },
										{ $not: ["$amountUSD"] },
										{ $eq: [{ $type: "$amountUSD" }, "missing"] },
									],
								},
								1,
								0,
							],
						},
					},
				},
			},
			{
				$project: {
					totalUsers: { $size: "$totalUsers" },
					totalAmountNTE: { $ifNull: ["$totalAmountNTE", 0] },
					totalAmountUSD: { $ifNull: ["$totalAmountUSD", 0] },
					averageAmountNTE: { $ifNull: ["$averageAmountNTE", 0] },
					averageAmountUSD: { $ifNull: ["$averageAmountUSD", 0] },
					minAmountNTE: { $ifNull: ["$minAmountNTE", 0] },
					maxAmountNTE: { $ifNull: ["$maxAmountNTE", 0] },
					uniqueTypes: { $size: "$transactionTypes" },
					uniqueSubTypes: { $size: "$transactionSubTypes" },
					totalTransactions: 1,
					transactionsWithOriginalUSD: { $ifNull: ["$transactionsWithOriginalUSD", 0] },
					transactionsWithCalculatedUSD: { $ifNull: ["$transactionsWithCalculatedUSD", 0] },
				},
			},
		]);

		return result[0] || {};
	},
};

module.exports = { walletTransactionHistoryService };
