const mongoose = require('mongoose');

const withdrawalMetadataSchema = new mongoose.Schema(
  {
    availableBalance: {
      type: Number,
      required: true,
    },
    bonusTransactionId: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
    },
    confirmedAt: {
      type: Date,
      required: true,
    },
    distributionId: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
    },
    fromAddress: {
      type: String,
      required: true,
      trim: true,
    },
    packageName: {
      type: String,
      required: true,
      trim: true,
    },
    packageType: {
      type: String,
      required: true,
      trim: true,
    },
    sourceBalanceType: {
      type: String,
      required: true,
      trim: true,
    },
    tokenValue: {
      type: Number,
      required: true,
    },
    traceId: {
      type: String,
      required: true,
      trim: true,
    },
    walletBalanceBefore: {
      type: Number,
      required: true,
    },
    walletTransactionId: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
    },
    isProportional: {
      type: Boolean,
    },
    requestId: {
      type: mongoose.Schema.Types.Mixed,
    },
    withdrawalRequestId: {
      type: mongoose.Schema.Types.ObjectId,
    },
  },
  { _id: false }
);

const withdrawalSchema = new mongoose.Schema(
  {
    userId: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
      ref: 'User',
      index: true,
    },
    username: {
      type: String,
      required: true,
      trim: true,
      index: true,
    },
    amountNTE: {
      type: Number,
      required: true,
      min: 0,
    },
    amountUSD: {
      type: Number,
      required: true,
      min: 0,
    },
    withdrawalType: {
      type: String,
      required: true,
      index: true,
    },
    status: {
      type: String,
      required: true,
      index: true,
    },
    toAddress: {
      type: String,
      required: true,
      trim: true,
    },
    transactionHash: {
      type: String,
      required: true,
      trim: true,
      index: true,
    },
    blockNumber: {
      type: Number,
      required: true,
      default: 0,
    },
    gasPrice: {
      type: String,
      required: true,
      default: '0',
    },
    gasUsed: {
      type: String,
      required: true,
      default: '0',
    },
    requestedAt: {
      type: Date,
      required: true,
      default: Date.now,
      index: true,
    },
    processedAt: {
      type: Date,
      required: true,
      default: Date.now,
    },
    confirmedAt: {
      type: Date,
      required: true,
      default: Date.now,
    },
    metadata: {
      type: withdrawalMetadataSchema,
      required: true,
    },
  },
  {
    timestamps: true,
    collection: 'withdrawals',
  }
);

// Indexes for query optimization
withdrawalSchema.index({ userId: 1, status: 1 });
withdrawalSchema.index({ username: 1, requestedAt: -1 });
withdrawalSchema.index({ status: 1, requestedAt: -1 });
withdrawalSchema.index({ withdrawalType: 1, status: 1 });
withdrawalSchema.index({ transactionHash: 1 });
withdrawalSchema.index({ createdAt: -1 });

// Calculate total withdrawal cost including gas
withdrawalSchema.methods.getTotalCost = function () {
  const gasCost = parseFloat(this.gasPrice) * parseFloat(this.gasUsed);
  return this.amountNTE + gasCost;
};

// Check if withdrawal is completed
withdrawalSchema.methods.isCompleted = function () {
  return this.status === 'completed';
};

// Check if withdrawal can be cancelled
withdrawalSchema.methods.canCancel = function () {
  return ['pending', 'failed'].includes(this.status);
};

// Safe object for API response
withdrawalSchema.methods.toSafeObject = function () {
  const obj = this.toObject();
  delete obj.__v;
  return obj;
};

const Withdrawal = mongoose.model('Withdrawal', withdrawalSchema);

module.exports = { Withdrawal };
