const { Withdrawal } = require("./withdrawal.model");

const withdrawalHistoryService = {
	async getWithdrawalHistory(queryParams) {
		const {
			page = 1,
			limit = 20,
			sortBy = "requestedAt",
			sortOrder = "desc",
			search,
			username,
			status,
			withdrawalType,
			startDate,
			endDate,
			minAmount,
			maxAmount,
			timezone = 0,
		} = queryParams;

		const parsedLimit = Math.min(Math.max(Number(limit) || 20, 1), 100);
		const parsedPage = Math.max(1, Number(page) || 1);
		const skip = (parsedPage - 1) * parsedLimit;

		const query = {};

		if (username && username.trim()) {
			query.username = { $regex: username.trim(), $options: "i" };
		}

		if (status && status.trim() && status !== "all") {
			query.status = status;
		}

		if (withdrawalType && withdrawalType.trim() && withdrawalType !== "all") {
			query.withdrawalType = withdrawalType;
		}

		// Date range with timezone handling
		if ((startDate && startDate.trim()) || (endDate && endDate.trim())) {
			query.requestedAt = {};
			const clientTimezoneOffset = parseInt(timezone) || 0;
			const offsetMs = -clientTimezoneOffset * 60 * 1000;

			const parseLocalDateToUTC = (dateStr, endOfDay = false) => {
				const [year, month, day] = dateStr.split("-").map(Number);
				const utcDate = new Date(
					Date.UTC(
						year,
						month - 1,
						day,
						endOfDay ? 23 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 59 : 0,
						endOfDay ? 999 : 0,
					),
				);
				return new Date(utcDate.getTime() - offsetMs);
			};

			if (startDate && startDate.trim()) {
				query.requestedAt.$gte = parseLocalDateToUTC(startDate, false);
			}

			if (endDate && endDate.trim()) {
				query.requestedAt.$lte = parseLocalDateToUTC(endDate, true);
			}
		}

		// Amount range
		if (minAmount && String(minAmount).trim()) {
			query.amountNTE = { ...query.amountNTE, $gte: parseFloat(minAmount) };
		}
		if (maxAmount && String(maxAmount).trim()) {
			query.amountNTE = { ...query.amountNTE, $lte: parseFloat(maxAmount) };
		}

		// Search across multiple fields
		if (search && search.trim()) {
			const searchRegex = { $regex: search.trim(), $options: "i" };
			query.$or = [
				{ toAddress: searchRegex },
				{ transactionHash: searchRegex },
				{ "metadata.traceId": searchRegex },
			];
		}

		const sortObj = {};
		sortObj[sortBy] = sortOrder === "asc" ? 1 : -1;

		const [withdrawals, totalCount, summaryStats] = await Promise.all([
			Withdrawal.find(query).sort(sortObj).skip(skip).limit(parsedLimit).lean(),
			Withdrawal.countDocuments(query),
			this._getSummaryStats(query),
		]);

		const totalPages = Math.ceil(totalCount / parsedLimit);

		const processedWithdrawals = withdrawals.map((w) => ({
			...w,
			_id: w._id.toString(),
			userId: w.userId.toString(),
		}));

		return {
			success: true,
			withdrawals: processedWithdrawals,
			summary: {
				totalWithdrawals: summaryStats.totalWithdrawals || 0,
				totalAmountNTE: summaryStats.totalAmountNTE || 0,
				totalAmountUSD: summaryStats.totalAmountUSD || 0,
				completedCount: summaryStats.completedCount || 0,
				pendingCount: summaryStats.pendingCount || 0,
				failedCount: summaryStats.failedCount || 0,
			},
			pagination: {
				total: totalCount,
				page: parsedPage,
				limit: parsedLimit,
				totalPages,
				hasNext: parsedPage < totalPages,
				hasPrev: parsedPage > 1,
			},
			filters: {
				username,
				status,
				withdrawalType,
				startDate,
				endDate,
				minAmount,
				maxAmount,
				search,
			},
		};
	},

	async _getSummaryStats(query) {
		const result = await Withdrawal.aggregate([
			{ $match: query },
			{
				$group: {
					_id: null,
					totalWithdrawals: { $sum: 1 },
					totalAmountNTE: { $sum: "$amountNTE" },
					totalAmountUSD: { $sum: "$amountUSD" },
					completedCount: {
						$sum: { $cond: [{ $eq: ["$status", "completed"] }, 1, 0] },
					},
					pendingCount: {
						$sum: { $cond: [{ $eq: ["$status", "pending"] }, 1, 0] },
					},
					failedCount: {
						$sum: { $cond: [{ $eq: ["$status", "failed"] }, 1, 0] },
					},
				},
			},
		]);

		return result[0] || {};
	},
};

module.exports = { withdrawalHistoryService };
