const { WithdrawalRequest } = require("./withdrawalRequest.model");

/**
 * Parse a local date string (YYYY-MM-DD) to UTC based on timezone offset
 * @param {string} dateStr - Date string in YYYY-MM-DD format
 * @param {number} timezoneOffset - Timezone offset in minutes (e.g., -480 for UTC+8)
 * @param {boolean} endOfDay - If true, set time to 23:59:59.999
 * @returns {Date} UTC Date object
 */
function parseLocalDateToUTC(dateStr, timezoneOffset = 0, endOfDay = false) {
	const [year, month, day] = dateStr.split("-").map(Number);
	const offsetMs = -timezoneOffset * 60 * 1000;

	const utcDate = new Date(
		Date.UTC(
			year,
			month - 1,
			day,
			endOfDay ? 23 : 0,
			endOfDay ? 59 : 0,
			endOfDay ? 59 : 0,
			endOfDay ? 999 : 0
		)
	);

	return new Date(utcDate.getTime() - offsetMs);
}

const withdrawalRequestHistoryService = {
	/**
	 * Get withdrawal requests with filters, pagination, and summary stats
	 * @param {Object} params - Query parameters
	 * @returns {Promise<Object>} Paginated withdrawal requests with summary
	 */
	async getWithdrawalRequests(params = {}) {
		const {
			page = 1,
			limit = 10,
			sortBy = "createdAt",
			sortOrder = "desc",
			search = "",
			username,
			status,
			selectedWithdrawalType,
			startDate,
			endDate,
			minAmount,
			maxAmount,
			timezone = 0,
		} = params;

		const pageNum = parseInt(page, 10) || 1;
		const limitNum = parseInt(limit, 10) || 10;
		const timezoneOffset = parseInt(timezone, 10) || 0;

		// Build search query
		const searchQuery = {};

		// Username filter
		if (username && username.trim()) {
			searchQuery.username = { $regex: username.trim(), $options: "i" };
		}

		// Status filter
		if (status && status.trim()) {
			searchQuery.status = status.trim();
		}

		// Withdrawal type filter
		if (selectedWithdrawalType && selectedWithdrawalType.trim()) {
			searchQuery.selectedWithdrawalType = selectedWithdrawalType.trim();
		}

		// Date range filter
		if (startDate || endDate) {
			searchQuery.createdAt = {};
			if (startDate) {
				searchQuery.createdAt.$gte = parseLocalDateToUTC(startDate, timezoneOffset, false);
			}
			if (endDate) {
				searchQuery.createdAt.$lte = parseLocalDateToUTC(endDate, timezoneOffset, true);
			}
		}

		// Amount range filter
		if (minAmount || maxAmount) {
			searchQuery.amountNTE = {};
			if (minAmount) {
				searchQuery.amountNTE.$gte = parseFloat(minAmount);
			}
			if (maxAmount) {
				searchQuery.amountNTE.$lte = parseFloat(maxAmount);
			}
		}

		// Text search across multiple fields
		if (search && search.trim()) {
			searchQuery.$or = [
				{ username: { $regex: search.trim(), $options: "i" } },
				{ toAddress: { $regex: search.trim(), $options: "i" } },
				{ selectedWithdrawalType: { $regex: search.trim(), $options: "i" } },
			];
		}

		// Build sort object
		const sortObj = {};
		sortObj[sortBy] = sortOrder === "asc" ? 1 : -1;

		// Calculate pagination
		const skip = (pageNum - 1) * limitNum;

		// Execute queries in parallel
		const [requests, totalCount, summaryStats] = await Promise.all([
			WithdrawalRequest.find(searchQuery)
				.sort(sortObj)
				.skip(skip)
				.limit(limitNum)
				.lean(),
			WithdrawalRequest.countDocuments(searchQuery),
			this._getSummaryStats(searchQuery),
		]);

		// Calculate pagination info
		const totalPages = Math.ceil(totalCount / limitNum);
		const hasNext = pageNum < totalPages;
		const hasPrev = pageNum > 1;

		return {
			requests,
			pagination: {
				total: totalCount,
				page: pageNum,
				limit: limitNum,
				totalPages,
				hasNext,
				hasPrev,
			},
			summary: summaryStats,
		};
	},

	/**
	 * Get summary statistics for withdrawal requests matching query
	 * @param {Object} matchQuery - MongoDB match query
	 * @returns {Promise<Object>} Summary statistics
	 */
	async _getSummaryStats(matchQuery) {
		const result = await WithdrawalRequest.aggregate([
			{ $match: matchQuery },
			{
				$group: {
					_id: null,
					totalRequests: { $sum: 1 },
					totalUsers: { $addToSet: "$username" },
					totalAmountNTE: { $sum: { $ifNull: ["$amountNTE", 0] } },
					totalAmountUSD: { $sum: { $ifNull: ["$amountUSD", 0] } },
					averageAmountNTE: { $avg: { $ifNull: ["$amountNTE", 0] } },
					averageAmountUSD: { $avg: { $ifNull: ["$amountUSD", 0] } },
					statusCounts: { $push: "$status" },
					typeCounts: { $push: "$selectedWithdrawalType" },
				},
			},
			{
				$project: {
					totalRequests: 1,
					totalUsers: { $size: "$totalUsers" },
					totalAmountNTE: { $ifNull: ["$totalAmountNTE", 0] },
					totalAmountUSD: { $ifNull: ["$totalAmountUSD", 0] },
					averageAmountNTE: { $ifNull: ["$averageAmountNTE", 0] },
					averageAmountUSD: { $ifNull: ["$averageAmountUSD", 0] },
					statusBreakdown: {
						$arrayToObject: {
							$map: {
								input: { $setUnion: ["$statusCounts"] },
								as: "status",
								in: {
									k: "$$status",
									v: {
										$size: {
											$filter: {
												input: "$statusCounts",
												cond: { $eq: ["$$this", "$$status"] },
											},
										},
									},
								},
							},
						},
					},
					typeBreakdown: {
						$arrayToObject: {
							$map: {
								input: { $setUnion: ["$typeCounts"] },
								as: "type",
								in: {
									k: "$$type",
									v: {
										$size: {
											$filter: {
												input: "$typeCounts",
												cond: { $eq: ["$$this", "$$type"] },
											},
										},
									},
								},
							},
						},
					},
				},
			},
		]);

		return (
			result[0] || {
				totalRequests: 0,
				totalUsers: 0,
				totalAmountNTE: 0,
				totalAmountUSD: 0,
				averageAmountNTE: 0,
				averageAmountUSD: 0,
				statusBreakdown: {},
				typeBreakdown: {},
			}
		);
	},
};

module.exports = { withdrawalRequestHistoryService };
