const EventEmitter = require("events");
const { v4: uuidv4 } = require("uuid");
const { queueConfig } = require("../config/queue.config");
const { logger } = require("../core/logger/logger");
const { Job } = require("../core/models/job.model");

class InMemoryLifoQueue extends EventEmitter {
	constructor(name, defaultJobOptions = {}) {
		super();
		this.name = name;
		this.defaultJobOptions = defaultJobOptions;
		this.processing = false;
		this.handlers = new Map();
		this.stats = { processed: 0, failed: 0, retried: 0 };

		this.on("failed", (job, err) => {
			this.stats.failed++;
			logger.error("Job failed permanently", {
				jobId: job.id,
				jobName: job.name,
				attempts: job.attemptsMade,
				error: err.message,
			});
		});

		this.on("completed", (job) => {
			this.stats.processed++;
		});

		// Load pending jobs on startup
		this.#loadPendingJobs();
	}

	async add(name, data, opts = {}) {
		const job = {
			id: uuidv4(),
			name,
			data,
			opts: { ...this.defaultJobOptions, ...opts },
			attemptsMade: 0,
			createdAt: Date.now(),
			priority: opts.priority || 0,
		};

		try {
			// Save to database
			await Job.create({
				id: job.id,
				name: job.name,
				data: job.data,
				opts: job.opts,
				attemptsMade: job.attemptsMade,
				createdAt: new Date(job.createdAt),
				priority: job.priority,
				status: "pending",
			});

			logger.debug("Job added to queue", { jobId: job.id, jobName: name });
			setImmediate(() => this.#drain());
			return job;
		} catch (err) {
			logger.error("Failed to add job to queue", {
				error: err.message,
				jobName: name,
			});
			throw err;
		}
	}

	registerProcessor(jobName, handler) {
		this.handlers.set(jobName, handler);
		logger.info("Processor registered", { jobName });
		setImmediate(() => this.#drain());
	}

	async #runJob(job) {
		const handler = this.handlers.get(job.name);
		if (!handler) {
			await Job.findOneAndUpdate(
				{ id: job.id },
				{
					$set: {
						status: "failed",
						failedAt: new Date(),
						error: `No handler for job ${job.name}`,
					},
				},
			);
			this.emit("failed", job, new Error(`No handler for job ${job.name}`));
			return;
		}

		const startTime = Date.now();
		try {
			await handler(job);

			// Update job as completed
			await Job.findOneAndUpdate(
				{ id: job.id },
				{
					$set: {
						status: "completed",
						completedAt: new Date(),
					},
				},
			);

			this.emit("completed", job);
			logger.debug("Job completed", {
				jobId: job.id,
				durationMs: Date.now() - startTime,
			});
		} catch (err) {
			job.attemptsMade += 1;
			const maxAttempts = job.opts.attempts || 1;

			if (job.attemptsMade < maxAttempts) {
				const delay = this.#calculateBackoff(job);
				this.stats.retried++;
				logger.warn("Job failed, will retry", {
					jobId: job.id,
					attempt: job.attemptsMade,
					maxAttempts,
					retryInMs: delay,
					error: err.message,
				});

				// Update job for retry
				await Job.findOneAndUpdate(
					{ id: job.id },
					{
						$set: {
							status: "pending",
							attemptsMade: job.attemptsMade,
						},
						$unset: {
							startedAt: 1,
						},
					},
				);

				setTimeout(() => this.#drain(), delay);
			} else {
				// Mark as permanently failed
				await Job.findOneAndUpdate(
					{ id: job.id },
					{
						$set: {
							status: "failed",
							failedAt: new Date(),
							error: err.message,
							attemptsMade: job.attemptsMade,
						},
					},
				);

				this.emit("failed", job, err);
			}
		}
	}

	#calculateBackoff(job) {
		const { backoff } = job.opts;
		if (!backoff) return 1000;

		if (backoff.type === "exponential") {
			return Math.min(backoff.delay * Math.pow(2, job.attemptsMade - 1), 60000);
		}
		return backoff.delay || 1000;
	}

	async #drain() {
		if (this.processing) return;

		try {
			// Find the next pending job
			const nextJobDoc = await Job.findOneAndUpdate(
				{ status: "pending" },
				{
					$set: {
						status: "processing",
						startedAt: new Date(),
					},
				},
				{
					sort: { priority: -1, createdAt: 1 },
					new: true,
				},
			);

			if (!nextJobDoc) return;

			this.processing = true;

			const job = {
				id: nextJobDoc.id,
				name: nextJobDoc.name,
				data: nextJobDoc.data,
				opts: nextJobDoc.opts,
				attemptsMade: nextJobDoc.attemptsMade,
				createdAt: nextJobDoc.createdAt.getTime(),
			};

			await this.#runJob(job);
			this.processing = false;

			// Continue processing if there are more jobs
			setImmediate(() => this.#drain());
		} catch (err) {
			logger.error("Error in queue drain", { error: err.message });
			this.processing = false;
			// Retry drain after a delay
			setTimeout(() => this.#drain(), 5000);
		}
	}

	async #loadPendingJobs() {
		try {
			// Reset any jobs that were in processing state
			const resetResult = await Job.updateMany(
				{ status: "processing" },
				{
					$set: { status: "pending" },
					$unset: { startedAt: 1 },
				},
			);

			if (resetResult.modifiedCount > 0) {
				logger.info("Reset processing jobs to pending", {
					count: resetResult.modifiedCount,
				});
			}

			// Count pending jobs
			const pendingCount = await Job.countDocuments({ status: "pending" });
			logger.info("Queue initialized", { pendingJobs: pendingCount });

			// Start processing if there are pending jobs
			if (pendingCount > 0) {
				setImmediate(() => this.#drain());
			}
		} catch (err) {
			logger.error("Failed to load pending jobs", { error: err.message });
		}
	}
}

let queueInstance;

const getWorkQueue = () => {
	if (queueInstance) return queueInstance;
	queueInstance = new InMemoryLifoQueue(
		queueConfig.jobQueueName,
		queueConfig.defaultJobOptions,
	);
	return queueInstance;
};

const enqueueJob = async (name, payload, opts) => {
	const queue = getWorkQueue();
	return queue.add(name, payload, opts);
};

const registerProcessor = (name, processor) => {
	const queue = getWorkQueue();
	queue.registerProcessor(name, processor);
	return queue;
};

module.exports = { getWorkQueue, enqueueJob, registerProcessor };
